// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Math/Vector/hkVector4Util.h>

/// Utility functions for hkQuaternion
template <typename FT>
struct hkQuaternionUtilImpl
{
    /// Sets qOut to be the shortest rotation which brings 'from' to 'to'.
    /// NOTE: The vectors 'from ' and 'to' must be normalized.
    static HK_INLINE void HK_CALL _computeShortestRotation(typename hkFloatTypes<FT>::Vec4_ from, typename hkFloatTypes<FT>::Vec4_ to, typename hkFloatTypes<FT>::Quaternion& qOut);

    /// Sets qOut to be the shortest rotation which brings 'from' to 'to'.
    /// NOTE: The vectors 'from ' and 'to' must be normalized.
    /// This version is damped and the result interpolates from 'from' to to' as gain goes from 0 to 1.
    /// This is similar to scaling the angle of rotation according to the gain.
    
    static HK_INLINE void HK_CALL _computeShortestRotationDamped(typename hkFloatTypes<FT>::Vec4_ from, typename hkFloatTypes<FT>::Vec4_ to, typename hkFloatTypes<FT>::Scalar_ gain, typename hkFloatTypes<FT>::Quaternion& qOut);

    /// Computes a 4x4 matrix that allows you to compute any product of the form (this * q) by multiplying (opOut * q).
    /// This is also equal to Transpose(getInversePostMultiplyOperator(this)).
    /// Also, the first 3x3 block is equal to the first 3x3 block of Transpose(getPreMultiplyOperator(this))
    static HK_INLINE void HK_CALL _computePostMultiplyOperator(typename hkFloatTypes<FT>::Quaternion_ q, typename hkFloatTypes<FT>::Matrix4& opOut);

    /// Computes a 4x4 matrix that allows you to compute any product of the form (q * this) by multiplying (opOut * q).
    /// This is also equal to Transpose(getInversePreMultiplyOperator(this)).
    /// Also, the first 3x3 block is equal to the first 3x3 block of Transpose(getPostMultiplyOperator(this))
    static HK_INLINE void HK_CALL _computePreMultiplyOperator(typename hkFloatTypes<FT>::Quaternion_ q, typename hkFloatTypes<FT>::Matrix4& opOut);

    /// Computes a 4x4 matrix that allows you to compute any product of the form (Inverse(this) * q) by multiplying (opOut * q).
    /// This is also equal to Transpose(getPostMultiplyOperator(this)).
    static HK_INLINE void HK_CALL _computeInversePostMultiplyOperator(typename hkFloatTypes<FT>::Quaternion_ q, typename hkFloatTypes<FT>::Matrix4& opOut);

    /// Computes a 4x4 matrix that allows you to compute any product of the form (q * Inverse(this)) by multiplying (opOut * q).
    /// This is also equal to Transpose(getPreMultiplyOperator(this)).
    static HK_INLINE void HK_CALL _computeInversePreMultiplyOperator(typename hkFloatTypes<FT>::Quaternion_ q, typename hkFloatTypes<FT>::Matrix4& opOut);

    /// Computes qOut = Transpose(op) * qIn, where op is a 4x4 matrix operator and qIn is a quaternion interpreted as a 4-element vector.
    static HK_INLINE void HK_CALL _computeTransposeMul(const typename hkFloatTypes<FT>::Matrix4& op, typename hkFloatTypes<FT>::Quaternion_ qIn, typename hkFloatTypes<FT>::Quaternion& qOut);

    /// Computes the logarithm of a quaternion
    static HK_INLINE void HK_CALL _computeLog(typename hkFloatTypes<FT>::Quaternion_ q, typename hkFloatTypes<FT>::Vector& vOut);

    /// Computes the the exponential (quaternion) of the given (axis-angle) vector
    static HK_INLINE void HK_CALL _computeExp(typename hkFloatTypes<FT>::Vec4_ vIn, typename hkFloatTypes<FT>::Quaternion& qOut);
    static HK_INLINE void HK_CALL _computeExp_Approximate(typename hkFloatTypes<FT>::Vec4_ vIn, typename hkFloatTypes<FT>::Quaternion& qOut);

    /// N-lerp
    static HK_INLINE void HK_CALL _computeNlerp(typename hkFloatTypes<FT>::Quaternion_ q0, typename hkFloatTypes<FT>::Quaternion_ q1, typename hkFloatTypes<FT>::Scalar_ t, typename hkFloatTypes<FT>::Quaternion& qOut);

    /// Computes a clamped quaternion such that the rotation angle is less than the given value
    static HK_INLINE hkBool32 HK_CALL _computeClampedRotation(typename hkFloatTypes<FT>::Quaternion_ q0, typename hkFloatTypes<FT>::Scalar_ cosMaxAngle, typename hkFloatTypes<FT>::Quaternion& qClamped);

    /// Computes the maximum angular velocity that can be applied to quaternion qA along rotation axis dw (ASSUMED normalized) that will keep
    /// the relative angle between qA and qB under the given threshold.
    static HK_INLINE const typename hkFloatTypes<FT>::Scalar HK_CALL _computeMaximumAllowedVelocity(typename hkFloatTypes<FT>::Quaternion_ qA, typename hkFloatTypes<FT>::Quaternion_ qB, typename hkFloatTypes<FT>::Vec4_ dw, typename hkFloatTypes<FT>::Scalar_ cosMaxAngle);

    /// Increments (integrates) the given quaternion with the given axis-angle delta
    static HK_INLINE void HK_CALL _increment(typename hkFloatTypes<FT>::Quaternion_ qSrc, typename hkFloatTypes<FT>::Vec4_ axisAngleDelta, typename hkFloatTypes<FT>::Quaternion& qDst);

    /// Computes the angular velocity that rotates qSrc to qDst
    static HK_INLINE void HK_CALL _computeAngularVelocity(typename hkFloatTypes<FT>::Quaternion_ qSrc, typename hkFloatTypes<FT>::Quaternion_ qDst, typename hkFloatTypes<FT>::Scalar_ invDeltaTime, typename hkFloatTypes<FT>::Vector& wOut);

    /// Computes the cosine of the rotation angle
    static HK_INLINE const typename hkFloatTypes<FT>::Scalar HK_CALL _computeCosRotationAngle(typename hkFloatTypes<FT>::Quaternion_ q);

    /// Returns true if the given quaternions are almost equal
    static HK_INLINE hkBool32 _areApproximatelyEqual(typename hkFloatTypes<FT>::Quaternion_ qA, typename hkFloatTypes<FT>::Quaternion_ qB, typename hkFloatTypes<FT>::Scalar_ epsilon);
};

#include <Common/Base/Math/Quaternion/hkQuaternionUtil.inl>

typedef hkQuaternionUtilImpl<hkFloat32>  hkQuaternionfUtil;
typedef hkQuaternionUtilImpl<hkDouble64> hkQuaterniondUtil;

#if defined(HK_REAL_IS_DOUBLE)
typedef hkQuaterniondUtil hkQuaternionUtil;
#else
typedef hkQuaternionfUtil hkQuaternionUtil;
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
