// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#ifndef HK_MATH_MATH_H
#   error Please include Common/Base/hkBase.h instead of this file.
#endif

/// A 4x4 matrix of hkReals, representing a transformation.
/// This transform representation is used by tools, cloth and display libraries, as
/// it can represent scale and skew.
/// Elements are stored in column major format,
/// i.e., contiguous memory locations are (x00, x10, x20, x30), (x01, x11,...)
/// where x10 means row 1, column 0 for example.
/// For orthonormal transforms, use Transform instead (storage is equivalent).
/// Some operations in hkMatrix4 are optimized for affine matrices (those where
/// the last row is 0,0,0,1).
template<typename FT>
class hkMatrix4Impl
{
    public:
        HK_DECLARE_CLASS(hkMatrix4Impl, New, Pod, Reflect);
        HK_REFLECT_AS_ARRAY_FIXED(16, FT);
        HK_RECORD_ATTR( hk::Default({1,0,0,0}, {0,1,0,0}, {0,0,1,0}, {0,0,0,1}) );
        HK_RECORD_ATTR(hk::IncludeInMgd(false));

        typedef hkAlignedQuad<FT> ReflectDefaultType[4];

        typedef typename hkFloatTypes<FT>::Vec4  Vec4;
        typedef typename hkFloatTypes<FT>::Vec4_ Vec4_;

        typedef typename hkFloatTypes<FT>::Scalar  Scalar;
        typedef typename hkFloatTypes<FT>::Scalar_ Scalar_;

        typedef hkMatrix4Impl<FT>  Matrix4;
        typedef hkRotationImpl<FT> Rotation;

        typedef typename hkFloatTypes<FT>::Transform    Transform;
        typedef typename hkFloatTypes<FT>::QTransform   QTransform;
        typedef typename hkFloatTypes<FT>::QsTransform  QsTransform;
        typedef typename hkFloatTypes<FT>::Quaternion Quaternion;
        typedef typename hkFloatTypes<FT>::Comparison Comparison;

#ifndef HK_DISABLE_MATH_CONSTRUCTORS
            /// Empty constructor. The elements of the matrix are not initialized.
        HK_ALWAYS_INLINE hkMatrix4Impl() { } 
#endif
        /// Gets a read-write reference to the i'th column.
        HK_ALWAYS_INLINE Vec4& getColumn(int i);

        /// Gets a read reference to the i'th column.
        HK_ALWAYS_INLINE const Vec4& getColumn(int i) const;

        /// Gets a read-write reference to the i'th column.
        template <int I> HK_ALWAYS_INLINE Vec4& getColumn();

        /// Gets a read reference to the i'th column.
        template <int I> HK_ALWAYS_INLINE const Vec4& getColumn() const;

        /// set column
        template <int I> HK_ALWAYS_INLINE void setColumn(Vec4_ c);

        HK_ALWAYS_INLINE void setColumn(int i, Vec4_ c);

        /// Returns a copy of the i'th row.
        HK_ALWAYS_INLINE void getRow( int row, Vec4& r) const;
        template <int I> HK_ALWAYS_INLINE void getRow(Vec4& r) const;

        template <int I> HK_INLINE void setRow( Vec4_ r );
        HK_INLINE void setRow( int row, Vec4_ r );


        /// Gets read-write access to the specified element.
        HK_ALWAYS_INLINE FT& operator() (int row, int col);

        /// Gets read-only access to the specified elements.
        HK_ALWAYS_INLINE const FT& operator() (int row, int col) const;

        template <int ROW, int COL> HK_ALWAYS_INLINE Scalar getElement() const;
        template <int ROW, int COL> HK_ALWAYS_INLINE void setElement(Scalar_ s);

        /// Sets all rows at once.
        HK_INLINE void setRows( Vec4_ r0, Vec4_ r1, Vec4_ r2, Vec4_ r3);

        /// Writes the rows 0 to 3 in to the parameters r0, r1, r2, r3 respectively.
        HK_INLINE void getRows( Vec4& r0, Vec4& r1, Vec4& r2, Vec4& r3) const;

        /// Sets all columns of the current matrix. Where column is set to r0 and so on.
        HK_INLINE void setCols( Vec4_ c0, Vec4_ c1, Vec4_ c2, Vec4_ c3);

        /// Writes the columns 0 to 3 into the parameters c0, c1, c2 and c3 respectively.
        HK_INLINE void getCols( Vec4& c0, Vec4& c1, Vec4& c2, Vec4& c3) const;

        /// Zeroes all values in this matrix.
        HK_INLINE void setZero();

        /// Sets the specified diagonal values, zeroes the non-diagonal values.
        HK_INLINE void setDiagonal( FT m00, FT m11, FT m22, FT m33 );

        HK_INLINE void setDiagonal( Scalar_ m00, Scalar_ m11, Scalar_ m22, Scalar_ m33 );

        HK_INLINE void setDiagonal( Vec4_ v );

        HK_INLINE void getDiagonal( Vec4& v ) const;

        /// Sets the diagonal values to 1, zeroes the non-diagonal values.
        HK_INLINE void setIdentity();

        /// Returns a global identity transform.
        HK_INLINE static const Matrix4& HK_CALL getIdentity();

        /// Set the contents based on the given Transform. Will set the bottom row to (0,0,0,1) in this hkMatrix4 as
        /// it is undefined in a Transform (not used)
        HK_INLINE void set (const Transform& t);

        /// Set the contents based on the given QTransform. Will set the bottom row to (0,0,0,1).
        HK_EXPORT_COMMON void set (const QTransform& qt);

        /// Set the contents based on the given QsTransform. Will set the bottom row to (0,0,0,1).
        HK_EXPORT_COMMON void set (const QsTransform& qt);


        /// Configure the given transform with the content of self. As the transform is missing the bottom matrix row,
        /// it is simply discarded. Converting back and forth will thus not give the same matrix.
        HK_INLINE void get (Transform& t) const;

            /// Writes a 4x4 matrix to \a p in column-major order.
        /// The pointer \a p must be suitably aligned, depending on the math lib build.
        /// See the notes for aligned store in Vec4.
        HK_EXPORT_COMMON void get4x4ColumnMajor(_Out_writes_all_(16) float* HK_RESTRICT p) const;
        HK_EXPORT_COMMON void get4x4ColumnMajor(_Out_writes_all_(16) double* HK_RESTRICT p) const;

        /// Reads a 4x4 matrix from \a p in column-major order.
        /// The pointer \a p must be suitably aligned, depending on the math lib build.
        /// See the notes for aligned load in Vec4.
        HK_EXPORT_COMMON void set4x4ColumnMajor(_In_reads_(16) const float* p);
        HK_EXPORT_COMMON void set4x4ColumnMajor(_In_reads_(16) const double* p);

            /// Writes a 4x4 matrix to \a p in row-major order.
        /// The pointer \a p must be suitably aligned, depending on the math lib build.
        /// See the notes for aligned store in Vec4.
        HK_EXPORT_COMMON void get4x4RowMajor(_Out_writes_all_(16) float* HK_RESTRICT p) const;
        HK_EXPORT_COMMON void get4x4RowMajor(_Out_writes_all_(16) double* HK_RESTRICT p) const;

        /// Reads a 4x4 matrix from \a p in row-major order.
        /// The pointer \a p must be suitably aligned, depending on the math lib build.
        /// See the notes for aligned load in Vec4.
        HK_EXPORT_COMMON void set4x4RowMajor(_In_reads_(16) const float* p);
        HK_EXPORT_COMMON void set4x4RowMajor(_In_reads_(16) const double* p);

        /// Checks if this matrix is equal to m within an optional epsilon.
        HK_INLINE bool isApproximatelyEqual ( const Matrix4& m, Scalar_ epsilon ) const;

        /// Checks if this matrix is identity within \a epsilon.
        HK_INLINE hkBool32 isApproximatelyIdentity ( Scalar_ epsilon ) const;

        /// Set this as the inverse of "m".
        HK_EXPORT_COMMON hkResult setInverse(const Matrix4& m, Scalar_ epsilon = Scalar::getConstant(HK_QUADREAL_0) );

        /// Invert in place
        HK_EXPORT_COMMON hkResult invert(Scalar_ epsilon = Scalar::getConstant(HK_QUADREAL_0) );

        /// Transposes this matrix in place.
        HK_EXPORT_COMMON void transpose();

        /// set to the transpose of another matrix
        HK_INLINE void setTranspose( const Matrix4& s );

        /// (Assumes transforms are affine) Set this matrix to be the product of a and b. (this = a * b)
        HK_INLINE void setMulAffine ( const Matrix4& a, const Matrix4& b );

        /// (Non-affine version - slower) Set this matrix to be the product of a and b. (this = a * b)
        HK_INLINE void setMul ( const Matrix4& a, const Matrix4& b );

        /// Sets this matrix to be the product of a and scale (this = a * scale)
        HK_INLINE void setMul( Scalar_ scale, const Matrix4& a );

        /// Modifies this matrix by adding the matrix a to it. (this += a)
        HK_EXPORT_COMMON void add ( const Matrix4& a );

        /// Modifies this matrix by adding the matrix a to it. (this += a)
        /// This method is enforced inline.
        HK_INLINE void _add ( const Matrix4& a );

        /// Modifies this matrix by subtracting the matrix a from it. (this += a)
        HK_EXPORT_COMMON void sub ( const Matrix4& a );

        /// (Assumes transforms are affine) Modifies this matrix by post multiplying it by the matrix a. (this = this*a)
        HK_INLINE void mulAffine ( const Matrix4& a);

        /// (Non-affine version - slower) modifies this matrix by post multiplying it by the matrix a. (this = this*a)
        HK_INLINE void mul ( const Matrix4& a);

        /// Modifies this matrix by multiplying by scale (this *= scale)
        HK_INLINE void mul ( Scalar_ scale );

        /// Copies all elements from a into this matrix.
        HK_INLINE void operator= ( const Matrix4& a );

        /// Checks for bad values (denormals or infinities)
        HK_EXPORT_COMMON bool isOk() const;

        /// Checks whether the matrix represents an affine transformation (the 4th row is 0,0,0,1)
        HK_EXPORT_COMMON hkBool32 isAffineTransformation() const;

        /// Forces the matrix to represent an affine transformation (resets the 4th row to 0,0,0,1)
        HK_INLINE void resetFourthRow ();

        /// Transforms a position - implicitly sets positionIn.w to 1.0f. positionOut.w is undefined
        HK_INLINE void transformPosition (Vec4_ positionIn, Vec4& positionOut) const;

        /// Transforms a direction - implicitly sets directionIn.w to 0.0f, the directionOut.w is undefined.
        HK_INLINE void transformDirection (Vec4_ directionIn, Vec4& directionOut) const;

        /// Transforms a direction with the transposed of this - implicitly sets directionIn.w to 0.0f.
        HK_INLINE void transformInverseDirection (Vec4_ directionIn, Vec4& directionOut) const;

        /// Multiplies a 4-element vector by this matrix4. Notice that the 4th component of the vector (w) is very relevant here.
        /// Use "transformPosition" or "transformDirection" to transform vectors representing positions or directions by matrices
        /// representing transformations.
        HK_INLINE void multiplyVector (Vec4_ vectorIn, Vec4& resultOut) const;

        /// Computes a multiply of this*vectorIn and then divides the result by its last element.
        /// Note that vectorIn(3) must be set.
        HK_INLINE void transformHomogeneousAndNormalize(Vec4_ vectorIn, Vec4& resultOut) const;

    public:

        Vec4 m_col0;
        Vec4 m_col1;
        Vec4 m_col2;
        Vec4 m_col3;
};


typedef hkMatrix4Impl<float> hkMatrix4f;
typedef hkMatrix4Impl<double> hkMatrix4d;

HK_ON_REAL_IS_FLOAT( typedef hkMatrix4f hkMatrix4; )
HK_ON_REAL_IS_DOUBLE( typedef hkMatrix4d hkMatrix4; )

#if defined(HK_COMPILER_ARMCC) || defined(HK_COMPILER_GHS)
template<> hkReflect::Detail::TypeData hkMatrix4Impl<float>::typeData;
template<> hkReflect::Detail::TypeData hkMatrix4Impl<double>::typeData;
#endif

#include <Common/Base/_Auto/TemplateTypes/hkMatrix4f_Types.inl>

HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkMatrix4f, hkMatrix4f_Tag);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkMatrix4d, hkMatrix4d_Tag);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
