// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>

#if defined(HK_COMPILER_MSVC)
#   pragma warning(disable:4661) //  no suitable definition provided for explicit template instantiation request
#endif

template<typename FT>
bool hkMatrix4Impl<FT>::isOk() const
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    bool col0Ok = m_col0.template isOk<4>();
    bool col1Ok = m_col1.template isOk<4>();
    bool col2Ok = m_col2.template isOk<4>();
    bool col3Ok = m_col3.template isOk<4>();
    return col0Ok && col1Ok && col2Ok && col3Ok;
#else
    const float* f = reinterpret_cast<const float*>(this);
    for(int i=0; i<16; ++i)
    {
        if( hkMath::isFinite(f[i]) == false )
        {
            return false;
        }
    }
    return true;
#endif
}

template<typename FT>
hkResult hkMatrix4Impl<FT>::setInverse(const Matrix4& m, Scalar_ eps)
{
#if HK_CONFIG_SIMD != HK_CONFIG_SIMD_ENABLED
    FT fA0 = m(0,0)*m(1,1) - m(0,1)*m(1,0);
    FT fA1 = m(0,0)*m(1,2) - m(0,2)*m(1,0);
    FT fA2 = m(0,0)*m(1,3) - m(0,3)*m(1,0);
    FT fA3 = m(0,1)*m(1,2) - m(0,2)*m(1,1);
    FT fA4 = m(0,1)*m(1,3) - m(0,3)*m(1,1);
    FT fA5 = m(0,2)*m(1,3) - m(0,3)*m(1,2);
    FT fB0 = m(2,0)*m(3,1) - m(2,1)*m(3,0);
    FT fB1 = m(2,0)*m(3,2) - m(2,2)*m(3,0);
    FT fB2 = m(2,0)*m(3,3) - m(2,3)*m(3,0);
    FT fB3 = m(2,1)*m(3,2) - m(2,2)*m(3,1);
    FT fB4 = m(2,1)*m(3,3) - m(2,3)*m(3,1);
    FT fB5 = m(2,2)*m(3,3) - m(2,3)*m(3,2);

    FT det = fA0*fB5 - fA1*fB4 + fA2*fB3 + fA3*fB2 - fA4*fB1 + fA5*fB0;

    if ( hkMath::fabs(det) <= eps.getReal() )
    {
        return HK_FAILURE;
    }

    Scalar invDet; invDet.template setReciprocal<HK_ACC_FULL,HK_DIV_IGNORE>( Scalar::fromFloat(det) );

    Matrix4 temp;
    temp(0,0) = + m(1,1)*fB5 - m(1,2)*fB4 + m(1,3)*fB3;
    temp(1,0) = - m(1,0)*fB5 + m(1,2)*fB2 - m(1,3)*fB1;
    temp(2,0) = + m(1,0)*fB4 - m(1,1)*fB2 + m(1,3)*fB0;
    temp(3,0) = - m(1,0)*fB3 + m(1,1)*fB1 - m(1,2)*fB0;
    temp(0,1) = - m(0,1)*fB5 + m(0,2)*fB4 - m(0,3)*fB3;
    temp(1,1) = + m(0,0)*fB5 - m(0,2)*fB2 + m(0,3)*fB1;
    temp(2,1) = - m(0,0)*fB4 + m(0,1)*fB2 - m(0,3)*fB0;
    temp(3,1) = + m(0,0)*fB3 - m(0,1)*fB1 + m(0,2)*fB0;
    temp(0,2) = + m(3,1)*fA5 - m(3,2)*fA4 + m(3,3)*fA3;
    temp(1,2) = - m(3,0)*fA5 + m(3,2)*fA2 - m(3,3)*fA1;
    temp(2,2) = + m(3,0)*fA4 - m(3,1)*fA2 + m(3,3)*fA0;
    temp(3,2) = - m(3,0)*fA3 + m(3,1)*fA1 - m(3,2)*fA0;
    temp(0,3) = - m(2,1)*fA5 + m(2,2)*fA4 - m(2,3)*fA3;
    temp(1,3) = + m(2,0)*fA5 - m(2,2)*fA2 + m(2,3)*fA1;
    temp(2,3) = - m(2,0)*fA4 + m(2,1)*fA2 - m(2,3)*fA0;
    temp(3,3) = + m(2,0)*fA3 - m(2,1)*fA1 + m(2,2)*fA0;

    setMul(invDet, temp);

    return HK_SUCCESS;
#else
    HK_ASSERT_NO_MSG(0x2454a322, this != &m);

    Vec4 a, b, c;
    Vec4 r1, r2, r3, tt, tt2;
    Vec4 sum;
    Vec4 m0, m1, m2, m3;
    Vec4 t0, t1;

    // Calculating the minterms for the first line.

    Comparison pnpn; pnpn.template set<hkVector4ComparisonMask::MASK_XZ>();
    Comparison npnp; npnp.template set<hkVector4ComparisonMask::MASK_YW>();

    const Vec4* cols = &(m.getColumn<0>());

    //
    tt = cols[3];
    tt2.template setPermutation<hkVectorPermutation::YZWX>(cols[2]);
    c.setMul(tt2, tt);                 // V3' V4
    t0.template setPermutation<hkVectorPermutation::ZWXY>(tt);
    a.setMul(tt2, t0);                 // V3' V4"
    t0.template setPermutation<hkVectorPermutation::WXYZ>(tt);
    b.setMul(tt2, t0);                 // V3' V4^

    t0.template setPermutation<hkVectorPermutation::YZWX>(a);
    t1.template setPermutation<hkVectorPermutation::ZWXY>(c);
    r1.setSub(t0, t1);                 // V3" V4^ - V3^ V4"
    t0.template setPermutation<hkVectorPermutation::ZWXY>(b);
    r2.setSub(t0, b);                  // V3^ V4' - V3' V4^
    t0.template setPermutation<hkVectorPermutation::YZWX>(c);
    r3.setSub(a, t0);                  // V3' V4" - V3" V4'

    tt = cols[1];
    a.template setPermutation<hkVectorPermutation::YZWX>(tt);
    sum.setMul(a, r1);
    b.template setPermutation<hkVectorPermutation::ZWXY>(tt);
    sum.addMul(b, r2);
    c.template setPermutation<hkVectorPermutation::WXYZ>(tt);
    sum.addMul(c, r3);

    Vec4 detTmp; detTmp.setMul(sum, cols[0]);
    // Calculating the determinant.
    //Scalar det = detTmp.getSimdAt(0) - detTmp.getSimdAt(1) + detTmp.getSimdAt(2) - detTmp.getSimdAt(3);

    Vec4 detTmp2; detTmp2.setFlipSign(detTmp, pnpn);
    Scalar det = detTmp2.template horizontalAdd<4>();
    Scalar absDet; absDet.setAbs(det);

    // Check for determinant
    if (absDet.isLessEqual(eps))
    {
        return HK_FAILURE;
    }

    m0.setFlipSign(sum, pnpn);

    // Calculating the minterms of the second line (using previous results).
    tt.template setPermutation<hkVectorPermutation::YZWX>(cols[0]);
    sum.setMul(tt, r1);
    tt.template setPermutation<hkVectorPermutation::YZWX>(tt);
    sum.addMul(tt, r2);
    tt.template setPermutation<hkVectorPermutation::YZWX>(tt);
    sum.addMul(tt, r3);
    m1.setFlipSign(sum, npnp);

    // Calculating the minterms of the third line.
    tt.template setPermutation<hkVectorPermutation::YZWX>(cols[0]);
    a.setMul(tt, b);                               // V1' V2"
    b.setMul(tt, c);                               // V1' V2^
    c.setMul(tt, cols[1]);                         // V1' V2

    t0.template setPermutation<hkVectorPermutation::YZWX>(a);
    t1.template setPermutation<hkVectorPermutation::ZWXY>(c);
    r1.setSub(t0, t1);                             // V1" V2^ - V1^ V2"
    t0.template setPermutation<hkVectorPermutation::ZWXY>(b);
    r2.setSub(t0, b);                              // V1^ V2' - V1' V2^
    t0.template setPermutation<hkVectorPermutation::YZWX>(c);
    r3.setSub(a, t0);                              // V1' V2" - V1" V2'

    tt.template setPermutation<hkVectorPermutation::YZWX>(cols[3]);
    sum.setMul(tt, r1);
    tt.template setPermutation<hkVectorPermutation::YZWX>(tt);
    sum.addMul(tt, r2);
    tt.template setPermutation<hkVectorPermutation::YZWX>(tt);
    sum.addMul(tt, r3);
    m2.setFlipSign(sum, pnpn);

    // Recip det
    Scalar recipDet;
    recipDet.template setReciprocal<HK_ACC_FULL,HK_DIV_IGNORE>(det);

    // Divide the first 12 minterms with the determinant.
    m0.mul(recipDet);
    m1.mul(recipDet);
    m2.mul(recipDet);

    // Calculate the minterms of the forth line and divide by the determinant.
    tt.template setPermutation<hkVectorPermutation::YZWX>(cols[2]);
    sum.setMul(tt, r1);
    tt.template setPermutation<hkVectorPermutation::YZWX>(tt);
    sum.addMul(tt, r2);
    tt.template setPermutation<hkVectorPermutation::YZWX>(tt);
    sum.addMul(tt, r3);

    m3.setFlipSign(sum, npnp);
    m3.mul(recipDet);

    setRows(m0,m1,m2,m3);

    return HK_SUCCESS;

#endif
}

template<typename FT>
hkResult hkMatrix4Impl<FT>::invert(Scalar_ epsilon)
{
    Matrix4 m = *this;
    return setInverse(m);
}

template<typename FT>
void hkMatrix4Impl<FT>::transpose()
{
    hkMath::transpose(m_col0,m_col1,m_col2,m_col3);
}

template<typename FT>
void hkMatrix4Impl<FT>::add( const Matrix4& a )
{
    m_col0.add( a.getColumn<0>() );
    m_col1.add( a.getColumn<1>() );
    m_col2.add( a.getColumn<2>() );
    m_col3.add( a.getColumn<3>() );
}

template<typename FT>
void hkMatrix4Impl<FT>::sub( const Matrix4& a )
{
    m_col0.sub( a.getColumn<0>() );
    m_col1.sub( a.getColumn<1>() );
    m_col2.sub( a.getColumn<2>() );
    m_col3.sub( a.getColumn<3>() );
}

// aTc = aTb * bTc

template<typename FT>
hkBool32 hkMatrix4Impl<FT>::isAffineTransformation() const
{
    Vec4 row3; getRow<3>(row3);
    
#if !defined(HK_DEBUG_SLOW)
    return row3.template allExactlyEqual<3>(Vec4::template getConstant<HK_QUADREAL_0001>());
#else
    // same as above, but you can set breakpoints
    if (row3.template allExactlyEqual<3>(Vec4::template getConstant<HK_QUADREAL_0001>()))
    {
        return true;
    }
    return false;
#endif
}

template<typename FT>
void hkMatrix4Impl<FT>::get4x4RowMajor(_Out_writes_all_(16) float* HK_RESTRICT d) const
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    Vec4 c0 = m_col0;
    Vec4 c1 = m_col1;
    Vec4 c2 = m_col2;
    Vec4 c3 = m_col3;

    hkMath::transpose(c0,c1,c2,c3);

    c0.template store<4>(d);
    c1.template store<4>(d+4);
    c2.template store<4>(d+8);
    c3.template store<4>(d+12);
#else
    const float* p = (const float*)this;
    for (int i = 0; i<4; i++)
    {
        float d0 = float(p[0]);
        float d4 = float(p[1]);
        float d8 = float(p[2]);
        float d12 = float(p[3]);
        d[0] = d0;
        d[4] = d4;
        d[8] = d8;
        d[12] = d12;
        d+= 1;
        p+= 4;
    }
#endif
}

template<typename FT>
void hkMatrix4Impl<FT>::get4x4RowMajor(_Out_writes_all_(16) double* HK_RESTRICT d) const
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    Vec4 c0 = m_col0;
    Vec4 c1 = m_col1;
    Vec4 c2 = m_col2;
    Vec4 c3 = m_col3;

    hkMath::transpose(c0,c1,c2,c3);

    c0.template store<4>(d);
    c1.template store<4>(d+4);
    c2.template store<4>(d+8);
    c3.template store<4>(d+12);
#else
    const float* p = (const float*)this;
    for (int i = 0; i<4; i++)
    {
        double d0 = double(p[0]);
        double d4 = double(p[1]);
        double d8 = double(p[2]);
        double d12 = double(p[3]);
        d[0] = d0;
        d[4] = d4;
        d[8] = d8;
        d[12] = d12;
        d+= 1;
        p+= 4;
    }
#endif
}

template<typename FT>
void hkMatrix4Impl<FT>::set4x4RowMajor(_In_reads_(16) const float* p)
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    Vec4 c0; c0.template load<4>(p);
    Vec4 c1; c1.template load<4>(p+4);
    Vec4 c2; c2.template load<4>(p+8);
    Vec4 c3; c3.template load<4>(p+12);

    hkMath::transpose(c0,c1,c2,c3);

    m_col0 = c0;
    m_col1 = c1;
    m_col2 = c2;
    m_col3 = c3;
#else
    float* HK_RESTRICT d = (float*)this;
    for (int i = 0; i<4; i++)
    {
        float d0 = float(p[0]);
        float d1 = float(p[4]);
        float d2 = float(p[8]);
        float d3 = float(p[12]);
        d[0] = d0;
        d[1] = d1;
        d[2] = d2;
        d[3] = d3;
        p+= 1;
        d+= 4;
    }
#endif
}

template<typename FT>
void hkMatrix4Impl<FT>::set4x4RowMajor(_In_reads_(16) const double* p)
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    Vec4 c0; c0.template load<4>(p);
    Vec4 c1; c1.template load<4>(p+4);
    Vec4 c2; c2.template load<4>(p+8);
    Vec4 c3; c3.template load<4>(p+12);

    hkMath::transpose(c0,c1,c2,c3);

    m_col0 = c0;
    m_col1 = c1;
    m_col2 = c2;
    m_col3 = c3;
#else
    float* HK_RESTRICT d = (float*)this;
    for (int i = 0; i<4; i++)
    {
        float d0 = float(p[0]);
        float d1 = float(p[4]);
        float d2 = float(p[8]);
        float d3 = float(p[12]);
        d[0] = d0;
        d[1] = d1;
        d[2] = d2;
        d[3] = d3;
        p+= 1;
        d+= 4;
    }
#endif
}

template<typename FT>
void hkMatrix4Impl<FT>::get4x4ColumnMajor(_Out_writes_all_(16) float* HK_RESTRICT d) const
{
    m_col0.template store<4>(d);
    m_col1.template store<4>(d+4);
    m_col2.template store<4>(d+8);
    m_col3.template store<4>(d+12);
}

template<typename FT>
void hkMatrix4Impl<FT>::set4x4ColumnMajor(_In_reads_(16) const float* p)
{
    m_col0.template load<4>(p);
    m_col1.template load<4>(p+4);
    m_col2.template load<4>(p+8);
    m_col3.template load<4>(p+12);
}

template<typename FT>
void hkMatrix4Impl<FT>::get4x4ColumnMajor(_Out_writes_all_(16) double* HK_RESTRICT d) const
{
    m_col0.template store<4>(d);
    m_col1.template store<4>(d+4);
    m_col2.template store<4>(d+8);
    m_col3.template store<4>(d+12);
}

template<typename FT>
void hkMatrix4Impl<FT>::set4x4ColumnMajor(_In_reads_(16) const double* p)
{
    m_col0.template load<4>(p);
    m_col1.template load<4>(p+4);
    m_col2.template load<4>(p+8);
    m_col3.template load<4>(p+12);
}

//
//  Set the contents based on the given hkTransformf. Will set the bottom row to (0,0,0,1) in this hkMatrix4f as
//  it is undefined in a Transform (not used)

template<typename FT>
void hkMatrix4Impl<FT>::set(const QTransform& qt)
{
    Transform t;
    t.set(qt.getRotation(), qt.getTranslation());
    set(t);
}

template<typename FT>
void hkMatrix4Impl<FT>::set(const QsTransform& qst)
{
    Rotation rotMatrix; rotMatrix.set (qst.m_rotation);

    // This is equivalent to rotMatrix*diag(scale), but with fewer intermediate steps.
    Vec4 c0; c0.setMul( rotMatrix.template getColumn<0>(), qst.m_scale.template getComponent<0>() );
    Vec4 c1; c1.setMul( rotMatrix.template getColumn<1>(), qst.m_scale.template getComponent<1>() );
    Vec4 c2; c2.setMul( rotMatrix.template getColumn<2>(), qst.m_scale.template getComponent<2>() );

    // this sets the w components equivalent to how resetFourthRow() would.
    m_col0.setXYZ_0( c0 );
    m_col1.setXYZ_0( c1 );
    m_col2.setXYZ_0( c2 );
    m_col3.setXYZ_W( qst.m_translation, Scalar::template getConstant<HK_QUADREAL_1>() );
}


template class hkMatrix4Impl<float>;
template class hkMatrix4Impl<double>;

#define DEFINE_IMPL_TYPE(TYPE) \
    template<> \
    hkReflect::Detail::TypeData TYPE::typeData = \
{ \
    hkReflect::Opt::NAME, \
    (hkUlong)&hkReflect::ReflectionOf< TYPE >::typeData, \
    HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::NAME, # TYPE) \
}; \
    static hkReflect::Detail::TypeRegNode TYPE ## _regNode(reinterpret_cast<hkReflect::Type*>(&TYPE::typeData));

//DEFINE_IMPL_TYPE(hkMatrix4f);
//DEFINE_IMPL_TYPE(hkMatrix4d);

#undef DEFINE_IMPL_TYPE

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
