// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Math/Matrix/hkMatrix4CameraUtil.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/Base/Math/Matrix/hkMatrix4Util.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Types/Geometry/Plane/hkPlane.h>

void hkMatrix4CameraUtil::setPerspectiveProjection( hkFloat32 viewWidth, hkFloat32 viewHeight, hkFloat32 nearZ, hkFloat32 farZ, ClipSpaceDepthRange::Enum depthRange, ClipSpaceYRange::Enum yRange, hkMatrix4& matrixOut )
{
    float vw = viewWidth * 0.5f;
    float vh = viewHeight * 0.5f;
    setPerspectiveProjection( -vw, vw, -vh, vh, nearZ, farZ, depthRange, yRange, matrixOut );
}

void hkMatrix4CameraUtil::setPerspectiveProjection( hkFloat32 left, hkFloat32 right, hkFloat32 bottom, hkFloat32 top, hkFloat32 nearZ, hkFloat32 farZ, ClipSpaceDepthRange::Enum depthRange, ClipSpaceYRange::Enum yRange, hkMatrix4& matrixOut )
{
    HK_ASSERT(0x7d156036, nearZ < farZ, "The near clipping plane has to be before the far clipping plane. Near: {}, Far: {} ", nearZ, farZ);

    const hkFloat32 twoNearZ = nearZ + nearZ;
    const hkFloat32 oneDivNearMinusFar = 1.0f / (nearZ - farZ);

    matrixOut.setZero();

    if (depthRange == ClipSpaceDepthRange::MinusOneToOne)
    {
        matrixOut( 2, 2 ) = (farZ + nearZ) * oneDivNearMinusFar;
        matrixOut( 2, 3 ) = 2.f * farZ * nearZ * oneDivNearMinusFar;
    }
    else
    {
        matrixOut( 2, 2 ) = farZ * oneDivNearMinusFar;
        matrixOut( 2, 3 ) = farZ * nearZ * oneDivNearMinusFar;
    }

    if (yRange == ClipSpaceYRange::OneToMinusOne)
    {
        hkAlgorithm::swap<hkFloat32>( bottom, top );
    }

    const hkFloat32 oneDivRightMinusLeft = 1.0f / (right - left);
    const hkFloat32 oneDivTopMinusBottom = 1.0f / (top - bottom);
    matrixOut( 0, 0 ) = twoNearZ * oneDivRightMinusLeft;
    matrixOut( 0, 2 ) = (left + right) * oneDivRightMinusLeft;
    matrixOut( 1, 1 ) = twoNearZ * oneDivTopMinusBottom;
    matrixOut( 1, 2 ) = (top + bottom) * oneDivTopMinusBottom;
    matrixOut( 3, 2 ) = -1.f;
}

void hkMatrix4CameraUtil::setPerspectiveProjectionFromFovX( hkFloat32 fieldOfViewXDegrees, hkFloat32 aspectRatioWidthDivHeight, hkFloat32 nearZ, hkFloat32 farZ, ClipSpaceDepthRange::Enum depthRange, ClipSpaceYRange::Enum yRange, hkMatrix4& matrixOut )
{
    hkFloat32 fvx = fieldOfViewXDegrees * 0.5f * HK_FLOAT_DEG_TO_RAD;
    const hkFloat32 xm = nearZ * hkMath::tan( fvx );
    const hkFloat32 ym = xm / aspectRatioWidthDivHeight;
    setPerspectiveProjection( -xm, xm, -ym, ym, nearZ, farZ, depthRange, yRange, matrixOut );
}

void hkMatrix4CameraUtil::setPerspectiveProjectionFromFovY( hkFloat32 fieldOfViewYDegrees, hkFloat32 aspectRatioWidthDivHeight, hkFloat32 nearZ, hkFloat32 farZ, ClipSpaceDepthRange::Enum depthRange, ClipSpaceYRange::Enum yRange, hkMatrix4& matrixOut )
{
    hkFloat32 fvy = fieldOfViewYDegrees * 0.5f * HK_FLOAT_DEG_TO_RAD;
    const hkFloat32 ym = nearZ * hkMath::tan( fvy );
    const hkFloat32 xm = ym * aspectRatioWidthDivHeight;
    setPerspectiveProjection( -xm, xm, -ym, ym, nearZ, farZ, depthRange, yRange, matrixOut );
}

void hkMatrix4CameraUtil::setOrthographicProjection( hkFloat32 viewWidth, hkFloat32 viewHeight, hkFloat32 nearZ, hkFloat32 farZ, ClipSpaceDepthRange::Enum depthRange, ClipSpaceYRange::Enum yRange, hkMatrix4& matrixOut )
{
    float vw = viewWidth * 0.5f;
    float vh = viewHeight * 0.5f;
    setOrthographicProjection( -vw, vw, -vh, vh, nearZ, farZ, depthRange, yRange, matrixOut );
}

void hkMatrix4CameraUtil::setOrthographicProjection( hkFloat32 left, hkFloat32 right, hkFloat32 bottom, hkFloat32 top, hkFloat32 nearZ, hkFloat32 farZ, ClipSpaceDepthRange::Enum depthRange, ClipSpaceYRange::Enum yRange, hkMatrix4& matrixOut )
{
    const hkFloat32 oneDivFarMinusNear = 1.0f / (farZ - nearZ);
    const hkFloat32 oneDivRightMinusLeft = 1.0f / (right - left);
    const hkFloat32 oneDivTopMinusBottom = 1.0f / (top - bottom);

    matrixOut.setZero();
    if (depthRange == ClipSpaceDepthRange::MinusOneToOne)
    {
        matrixOut( 2, 2 ) = -2.0f * oneDivFarMinusNear;
        matrixOut( 2, 3 ) = -(farZ + nearZ) * oneDivFarMinusNear;
    }
    else // ZeroToOne. DX style, assume we want horizontal flip too to go from our X to the left if y up, z depth [Right handed], to X to the right [Left handed, NDC space].
    {
        matrixOut( 2, 2 ) = -oneDivFarMinusNear;
        matrixOut( 2, 3 ) = -(nearZ)* oneDivFarMinusNear;
    }

    if (yRange == ClipSpaceYRange::OneToMinusOne)
    {
        hkAlgorithm::swap<hkFloat32>( top, bottom );
    }

    matrixOut( 0, 0 ) = 2.0f * oneDivRightMinusLeft;
    matrixOut( 0, 3 ) = -(right + left) * oneDivRightMinusLeft;

    matrixOut( 1, 1 ) = 2.0f * oneDivTopMinusBottom;
    matrixOut( 1, 3 ) = -(top + bottom) * oneDivTopMinusBottom;

    matrixOut( 3, 3 ) = 1.0f;
}

void hkMatrix4CameraUtil::setLookAtView( hkVector4Parameter startPos, hkVector4Parameter targetPos, hkVector4Parameter upDir, hkMatrix4& matrixOut, ViewHandedness::Enum handedness )
{
    hkVector4 negViewDir;
    negViewDir.setSub( startPos, targetPos );
    negViewDir.normalize<3>();

    hkVector4 up = upDir;
    if (hkMath::fabs( negViewDir.dot<3>( upDir ).getReal() ) > 0.999f)
    {
        // We have to pick a random up vector as the view direction is pretty much (anti-)parallel to the given up vector.
        hkVector4Util::calculatePerpendicularVector( negViewDir, up );
    }
    up.normalize<3>();

    hkVector4 right;
    right.setCross( up, negViewDir );
    right.normalize<3>();
    up.setCross( negViewDir, right );

    // In the left-handed case, we map +z to the view direction (and not to the negative view direction).
    if (handedness == ViewHandedness::LeftHanded)
    {
        negViewDir.setNeg<3>( negViewDir );
    }

    hkVector4 eye;
    eye.setMul( startPos, hkSimdReal::getConstant<HK_QUADREAL_MINUS1>() );

    // Patch the translation into the w-components of our directions, which will soon be the bottom row of the
    // transposed result matrix, to make it end up in the right column after transposition.
    right.setW( right.dot<3>( eye ) );
    up.setW( up.dot<3>( eye ) );
    negViewDir.setW( negViewDir.dot<3>( eye ) );

    hkMatrix4 transposedResult;
    transposedResult.setCols( right, up, negViewDir, hkVector4::getConstant<HK_QUADREAL_0001>() );

    matrixOut.setTranspose( transposedResult );
}

void hkMatrix4CameraUtil::setLookAtViewInv( hkVector4Parameter startPos, hkVector4Parameter targetPos, hkVector4Parameter upDir, hkMatrix4& matrixOut, ViewHandedness::Enum handedness )
{
    hkVector4 negViewDir;
    negViewDir.setSub( startPos, targetPos );
    negViewDir.normalize<3>();
    negViewDir.setW( hkSimdReal_0 );

    hkVector4 up = upDir;
    if (hkMath::abs( negViewDir.dot<3>( upDir ).getReal() ) > 0.999f)
    {
        // We have to pick a random up vector as the view direction is pretty much (anti-)parallel to the given up vector.
        hkVector4Util::calculatePerpendicularVector( negViewDir, up );
    }
    up.normalize<3>();

    // Calculate / orthogonalize our direction vectors.
    // Note that we don't have to explicitly set the w-components to 0, as this is done implicitly by hkVector4::setCross().
    hkVector4 right;
    right.setCross( up, negViewDir );
    right.normalize<3>();
    up.setCross( negViewDir, right );

    // In the left-handed case, we map +z to the view direction (as opposed to the negative view direction).
    if (handedness == ViewHandedness::LeftHanded)
    {
        negViewDir.setNeg<3>( negViewDir );
    }

    hkVector4 trans = startPos;
    trans.setW( hkSimdReal::getConstant<HK_QUADREAL_1>() );

    matrixOut.setCols( right, up, negViewDir, trans );
}

void hkMatrix4CameraUtil::getFrustumAabb(const hkMatrix4& viewProjection, hkAabb& aabb)
{
    aabb.setEmpty();

    hkMatrix4 invMatrix;
    if( invMatrix.setInverse(viewProjection, hkSimdReal_0).isSuccess() )
    {
        for ( int cornerIdx = 0; cornerIdx < 8; cornerIdx++ )
        {
            hkVector4 normalizedCorner;
            normalizedCorner.set( ( cornerIdx & 1 ) ? 1.0f : -1.0f, ( cornerIdx & 2 ) ? 1.0f : -1.0f, ( cornerIdx & 4 ) ? 1.0f : -1.0f, 1.0f );

            hkVector4 worldCornerF;
            invMatrix.transformPosition( normalizedCorner, worldCornerF );
            worldCornerF.mul( worldCornerF.getComponent<3>().reciprocal() );

            hkVector4 worldCorner;
            worldCorner.set( worldCornerF );
            aabb.includePoint( worldCorner );
        }
    }
    else
    {
        HK_WARN_ONCE(0x285b0e3, "Cannot compute frustum AABB - unable to invert matrix!" );
    }
}

void hkMatrix4CameraUtil::getFieldOfView(const hkMatrix4& proj, hkReal& fovX, hkReal& fovY)
{
    hkVector4 row0, row1, row2, row3;
    proj.getRows(row0, row1, row2, row3);

    hkVector4 leftPlane, rightPlane, bottomPlane, topPlane;

    leftPlane.setAdd(row3, row0);
    leftPlane.normalize<3>();

    rightPlane.setSub(row3, row0);
    rightPlane.normalize<3>();

    bottomPlane.setAdd(row3, row1);
    bottomPlane.normalize<3>();

    topPlane.setSub(row3, row1);
    topPlane.normalize<3>();

    fovX = 180.0f - hkMath::acos(leftPlane.dot<3>(rightPlane).getReal()) * HK_FLOAT_RAD_TO_DEG;
    fovY = 180.0f - hkMath::acos(topPlane.dot<3>(bottomPlane).getReal()) * HK_FLOAT_RAD_TO_DEG;
}

HK_INLINE static hkSimdReal lerp(hkSimdRealParameter x0, hkSimdRealParameter x1, hkReal t)
{
    hkSimdReal x;
    x.setInterpolate(x0, x1, hkSimdReal::fromFloat(t));
    return x;
}

void hkMatrix4CameraUtil::getInterpolatedFrustumPlane(FrustumPlaneDirection::Enum dir, hkReal t, const hkMatrix4& proj, ClipSpaceDepthRange::Enum depthRange, hkPlane& planeOut)
{
    hkVector4 rowA, rowB;

    switch (dir)
    {
    case FrustumPlaneDirection::LeftToRight:
        proj.getRow<0>(rowA);
        proj.getRow<3>(rowB);
        rowB.mul(lerp(hkSimdReal_Minus1, hkSimdReal_1, t));
        break;
    case FrustumPlaneDirection::BottomToTop:
        proj.getRow<1>(rowA);
        proj.getRow<3>(rowB);
        rowB.mul(lerp(hkSimdReal_Minus1, hkSimdReal_1, t));
        break;
    case FrustumPlaneDirection::NearToFar:
        proj.getRow<2>(rowA);
        proj.getRow<3>(rowB);
        rowB.mul(lerp(depthRange == ClipSpaceDepthRange::ZeroToOne ? hkSimdReal_0 : hkSimdReal_Minus1, hkSimdReal_1, t));
        break;
    default:
        HK_ASSERT_NOT_IMPLEMENTED(0x41113882);
        break;
    }

    planeOut.m_equation.setSub(rowA, rowB);
    planeOut.m_equation.normalize<3>();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
