// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#ifndef HK_MATH_MATRIX3_UTIL_H
#define HK_MATH_MATRIX3_UTIL_H

#include <Common/Base/Math/Vector/hkVector4Util.h>

/// Utility functions for hkMatrix3
template <typename FT>
class hkMatrix3UtilImpl
{
public:

    typedef hkMatrix3UtilImpl<FT> ThisType;
    typedef typename hkFloatTypes<FT>::Vec4 Vec4;
    typedef typename hkFloatTypes<FT>::Vec4_ Vec4_;

    typedef typename hkFloatTypes<FT>::Scalar Scalar;
    typedef typename hkFloatTypes<FT>::Scalar_ Scalar_;

    typedef typename hkFloatTypes<FT>::Transform Transform;
    typedef typename hkFloatTypes<FT>::Matrix3 Matrix3;
    typedef typename hkFloatTypes<FT>::QTransform QTransform;
    typedef typename hkFloatTypes<FT>::QsTransform QsTransform;
    typedef typename hkFloatTypes<FT>::Quaternion Quaternion;
    typedef typename hkFloatTypes<FT>::Rotation Rotation;
    typedef typename hkFloatTypes<FT>::Comparison Comparison;

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_MATH, ThisType);

    /// Sets reflectionOut to be the reflection of matrix D through matrix P, i.e., reflectionOut = Transpose(P) * A * P.
    /// Matrix D is assumed diagonal.
    static HK_INLINE void HK_CALL _computeReflectedDiagonal(const Matrix3& P, Vec4_ D, Matrix3& reflectionOut);
    static void HK_CALL computeReflectedDiagonal(const Matrix3& P, Vec4_ D, Matrix3& reflectionOut);

    /// Sets matrixOut = Q * D * Inverse(Q), where D is interpreted as a diagonal matrix.
    static HK_INLINE void HK_CALL _changeBasis(const Quaternion& Q, Vec4_ D, Matrix3& matrixOut);
    static void HK_CALL changeBasis(const Quaternion& Q, Vec4_ D, Matrix3& matrixOut);

    /// Sets matrixOut = R * D * Transpose(R), where D is interpreted as a diagonal matrix.
    static HK_INLINE void HK_CALL _changeBasis(const typename hkFloatTypes<FT>::Rotation& R, Vec4_ D, Matrix3& matrixOut);
    static void HK_CALL changeBasis(const typename hkFloatTypes<FT>::Rotation& R, Vec4_ D, Matrix3& matrixOut);

    /// Sets inertiaOut = Identity * dot(v, v) - OuterProduct(v, v). The matrix is used to displace inertia tensors
    /// by v from the center of mass.
    static HK_INLINE void HK_CALL _computeInertiaDisplacement(Vec4_ v, Matrix3& inertiaOut);
    static void HK_CALL computeInertiaDisplacement(Vec4_ v, Matrix3& inertiaOut);

    /// Returns the diagonal of the given matrix.
    static HK_INLINE    void HK_CALL _getDiagonal(const Matrix3& matrixIn, Vec4& diagonalOut);
    static void HK_CALL getDiagonal(const Matrix3& matrixIn, Vec4& diagonalOut);

    /// Returns the off-diagonal elements of the given matrix. The matrix is assumed to be symmetric. The returned vector is
    /// [m01, m02, m12, 0]
    static HK_INLINE void HK_CALL _getSymmetricOffDiagonal(const Matrix3& matrixIn, Vec4& offDiagonalOut);
    static void HK_CALL getSymmetricOffDiagonal(const Matrix3& matrixIn, Vec4& offDiagonalOut);

    /// Returns the absolute maximum element
    static HK_INLINE const typename hkFloatTypes<FT>::Scalar HK_CALL _computeMaxAbsElement(const Matrix3& matrixIn);
    static const typename hkFloatTypes<FT>::Scalar HK_CALL computeMaxAbsElement(const Matrix3& matrixIn);

    /// Computes matrixOut = matrixIn * hat(v), whereh hat(v) is the skew-symmetric matrix of v
    /// Specifically, it sets mtxOut.column(i) = cross( matrixIn.row(x), v )
    static HK_INLINE void HK_CALL _computeCrossProduct(const Matrix3& matrixIn, Vec4_ v, Matrix3& matrixOut);

    /// Sets aTb = Transpose(a) * b.
    static HK_INLINE void HK_CALL _computeTransposeMul(const Matrix3& a, const Matrix3& b, Matrix3& aTb);

    /// Sets this matrix as the cross product between the given vector and each of the given matrix's columns.
    /// Specifically, it sets mtxOut.column(i) = cross(v, src.column(i))
    static HK_INLINE void HK_CALL _computeCrossMul(Vec4_ v, const Matrix3& src, Matrix3& mtxOut);

    /// Checks if matrixIn is equal to the identity matrix within an epsilon.
    static HK_INLINE hkBool32 HK_CALL _isApproximatelyIdentity(const Matrix3& matrixIn, typename hkFloatTypes<FT>::Scalar_ eps);

    /// Sets the specified diagonal values, zeroes the non-diagonal values.
    static HK_INLINE void HK_CALL _setDiagonal(typename hkFloatTypes<FT>::Float m00, typename hkFloatTypes<FT>::Float m11, typename hkFloatTypes<FT>::Float m22, Matrix3& matrixOut);
    static HK_INLINE void HK_CALL _setDiagonal(typename hkFloatTypes<FT>::Scalar_ m00, typename hkFloatTypes<FT>::Scalar_ m11, typename hkFloatTypes<FT>::Scalar_ m22, Matrix3& matrixOut);
    static HK_INLINE void HK_CALL _setDiagonal(Vec4_ vDiagonal, Matrix3& matrixOut);
    static HK_INLINE void HK_CALL _setDiagonal(typename hkFloatTypes<FT>::Scalar_ diag, Matrix3& matrixOut);

    /// Sets the specified diagonal values, leaves the non-diagonal values alone.
    static HK_INLINE void HK_CALL _setDiagonalOnly(Vec4_ vDiagonal, Matrix3& matrixOut);
    static HK_INLINE void HK_CALL _setDiagonalOnly(typename hkFloatTypes<FT>::Scalar_ diag, Matrix3& matrixOut);


    /// Computes the outer product of the given vectors, i.e. mtxOut = u * Transpose[v]
    static HK_INLINE void HK_CALL _computeOuterProduct(Vec4_ u, Vec4_ v, Matrix3& mtxOut);

    /// Computes the displacement given by the two vectors rA and rB, i.e. mtxOut = Hat[rA] * Hat[rB]
    static HK_INLINE void HK_CALL _computeDisplacement(Vec4_ rA, Vec4_ rB, Matrix3& mtxOut);

    /// Converts a hkMatrix3 to a hkMatrix4
    static HK_INLINE void HK_CALL _convert(const Matrix3& fromMtx, typename hkFloatTypes<FT>::Matrix4& toMtx);
};

#include <Common/Base/Math/Matrix/hkMatrix3Util.inl>

typedef hkMatrix3UtilImpl<hkFloat32>  hkMatrix3fUtil;
typedef hkMatrix3UtilImpl<hkDouble64> hkMatrix3dUtil;

#if defined(HK_REAL_IS_DOUBLE)
typedef hkMatrix3dUtil hkMatrix3Util;
#else
typedef hkMatrix3fUtil hkMatrix3Util;
#endif

#endif  //  HK_MATH_MATRIX3_UTIL_H

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
