// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

template<typename FT>
HK_ALWAYS_INLINE typename hkMatrix3Impl<FT>::Vec4& hkMatrix3Impl<FT>::getColumn(int x)
{
    return (&m_col0)[x];
}


template<typename FT>
HK_ALWAYS_INLINE const typename hkMatrix3Impl<FT>::Vec4& hkMatrix3Impl<FT>::getColumn(int x) const
{
    return (&m_col0)[x];
}

template<typename FT>
template <int N>
HK_ALWAYS_INLINE const typename hkMatrix3Impl<FT>::Vec4& hkMatrix3Impl<FT>::getColumn() const
{
    return (&m_col0)[N];
}

template<typename FT>
template <int N>
HK_ALWAYS_INLINE typename hkMatrix3Impl<FT>::Vec4& hkMatrix3Impl<FT>::getColumn()
{
    return (&m_col0)[N];
}

template<typename FT>
template <int N>
HK_ALWAYS_INLINE void hkMatrix3Impl<FT>::setColumn(Vec4_ c)
{
    (&m_col0)[N] = c;
}


template<typename FT>
HK_ALWAYS_INLINE void hkMatrix3Impl<FT>::setDiagonal( Vec4_ f)
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    Vec4 c0;    c0.setMul(hkFloatTypes<FT>::Vector::template getConstant<HK_QUADREAL_1000>(), f);
    Vec4 c1;    c1.setMul(hkFloatTypes<FT>::Vector::template getConstant<HK_QUADREAL_0100>(), f);
    Vec4 c2;    c2.setMul(hkFloatTypes<FT>::Vector::template getConstant<HK_QUADREAL_0010>(), f);
    setCols(c0, c1, c2);
#else
    setZero();
    this->template setElement<0, 0>(f.template getComponent<0>());
    this->template setElement<1, 1>(f.template getComponent<1>());
    this->template setElement<2, 2>(f.template getComponent<2>());
#endif
}



#ifndef HK_DISABLE_MATH_CONSTRUCTORS
template<typename FT>
HK_INLINE hkMatrix3Impl<FT>::hkMatrix3Impl( const hkMatrix3Impl& t )
{
    m_col0 = t.m_col0;
    m_col1 = t.m_col1;
    m_col2 = t.m_col2;
}
#endif

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::set( const Rotation& r )
{
    m_col0 = r.m_col0;
    m_col1 = r.m_col1;
    m_col2 = r.m_col2;
}

template<typename FT>
HK_ALWAYS_INLINE FT& hkMatrix3Impl<FT>::operator() (int r, int c)
{
    return getColumn(c)(r);
}

template<typename FT>
HK_ALWAYS_INLINE const FT& hkMatrix3Impl<FT>::operator() (int r, int c) const
{
    return getColumn(c)(r);
}


template<typename FT>
template <int ROW, int COL>
HK_ALWAYS_INLINE typename hkMatrix3Impl<FT>::Scalar hkMatrix3Impl<FT>::getElement() const
{
// our reflection parser cannot compile this:   return getColumn<COL>().getComponent<ROW>();
    Vec4 col = (&m_col0)[COL];
    return col.template getComponent<ROW>();
}

template<typename FT>
template <int ROW, int COL>
HK_ALWAYS_INLINE void hkMatrix3Impl<FT>::setElement(Scalar_ element)
{
    (&m_col0)[COL].template setComponent<ROW>(element);
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::setRows( Vec4_ r0, Vec4_ r1, Vec4_ r2)
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    m_col0 = r0;
    m_col1 = r1;
    m_col2 = r2;
    hkMath::transpose(m_col0,m_col1,m_col2);
#else
    Vec4 c0; c0.set( r0(0), r1(0), r2(0) );
    Vec4 c1; c1.set( r0(1), r1(1), r2(1) );
    Vec4 c2; c2.set( r0(2), r1(2), r2(2) );
    Vec4 c3; c3.set( r0(3), r1(3), r2(3) );

    setColumn<0>(c0);
    setColumn<1>(c1);
    setColumn<2>(c2);
#endif
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::getRows( Vec4& r0, Vec4& r1, Vec4& r2) const
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    Vec4 c0 = m_col0;
    Vec4 c1 = m_col1;
    Vec4 c2 = m_col2;

    hkMath::transpose(c0,c1,c2);
#else
    Vec4 c0; c0.set( m_col0(0), m_col1(0), m_col2(0) );
    Vec4 c1; c1.set( m_col0(1), m_col1(1), m_col2(1) );
    Vec4 c2; c2.set( m_col0(2), m_col1(2), m_col2(2) );
#endif

    r0 = c0;
    r1 = c1;
    r2 = c2;
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::setCols( Vec4_ c0, Vec4_ c1, Vec4_ c2)
{
    m_col0 = c0;
    m_col1 = c1;
    m_col2 = c2;
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::operator= ( const hkMatrix3Impl& a )
{
    m_col0 = a.m_col0;
    m_col1 = a.m_col1;
    m_col2 = a.m_col2;
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::getCols( Vec4& c0, Vec4& c1, Vec4& c2) const
{
    c0 = m_col0;
    c1 = m_col1;
    c2 = m_col2;
}

template<typename FT>
HK_ALWAYS_INLINE void hkMatrix3Impl<FT>::getRow( int row, Vec4& r) const
{
    r.set( m_col0(row), m_col1(row), m_col2(row) );
}

template<typename FT>
template <int I>
HK_ALWAYS_INLINE void hkMatrix3Impl<FT>::getRow( Vec4& r ) const
{
    r.set( getElement<I,0>(), getElement<I,1>(), getElement<I,2>(), getElement<I,2>() );
}

    /// copy and transpose
template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::_setTranspose( const hkMatrix3Impl& s )
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    Vec4 c0 = s.m_col0;
    Vec4 c1 = s.m_col1;
    Vec4 c2 = s.m_col2;
    hkMath::transpose(c0,c1,c2);
    m_col0 = c0;
    m_col1 = c1;
    m_col2 = c2;
#else
    hkMatrix3Impl   cs = s;
    m_col0.set( cs.getElement<0,0>(), cs.getElement<0,1>(), cs.getElement<0,2>(), cs.getElement<0,2>() );
    m_col1.set( cs.getElement<1,0>(), cs.getElement<1,1>(), cs.getElement<1,2>(), cs.getElement<1,2>() );
    m_col2.set( cs.getElement<2,0>(), cs.getElement<2,1>(), cs.getElement<2,2>(), cs.getElement<2,2>() );
#endif
}

//
//  Sets this matrix to be the product of a diagonal matrix (given as a vector) and matrix a (this = diag * a)

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::setDiagonalMul(Vec4_ vDiag, const hkMatrix3Impl& a)
{
    m_col0.setMul(vDiag, a.m_col0);
    m_col1.setMul(vDiag, a.m_col1);
    m_col2.setMul(vDiag, a.m_col2);
}


template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::setMul( Scalar_ scale, const hkMatrix3Impl& a)
{
    m_col0.setMul(scale, a.m_col0);
    m_col1.setMul(scale, a.m_col1);
    m_col2.setMul(scale, a.m_col2);
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::add( const hkMatrix3Impl& a )
{
    m_col0.add( a.m_col0 );
    m_col1.add( a.m_col1 );
    m_col2.add( a.m_col2 );
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::sub( const hkMatrix3Impl& a )
{
    m_col0.sub( a.m_col0 );
    m_col1.sub( a.m_col1 );
    m_col2.sub( a.m_col2 );
}

template<typename FT>
template<hkMatrixSpecialization TYPE, hkMathAccuracyMode ACCURACY, hkDeterminantCheck CHECK, hkInversionFailure FAILURE >
HK_INLINE typename hkMatrix3Impl<FT>::Scalar hkMatrix3Impl<FT>::_setInverse( const hkMatrix3Impl& src )
{
    // Get source columns
    const Vec4 c0 = src.m_col0;
    const Vec4 c1 = src.m_col1;
    const Vec4 c2 = src.m_col2;

    Vec4 r0; r0.setCross(c1, c2);
    Vec4 r1; r1.setCross(c2, c0);
    Vec4 r2; r2.setCross(c0, c1);

    const Scalar determinant = c0.template dot<3>(r0);

    Comparison isNotSingular;
    switch (CHECK)
    {
    case HK_DETERMINANT_CHECK_EPS:
        {
            // Compute 1 / determinant. Set it to zero in case of singular matrices!
            // We detect singularity by scaling the determinant by the inverse
            // L1 norms of the columns of the matrix and comparing to epsilon.

#if (defined(HK_PLATFORM_LINUX) && HK_CONFIG_SIMD == HK_CONFIG_SIMD_DISABLED) // see COM-2686
            Scalar absDet = (determinant.isGreaterEqualZero()) ? determinant : -determinant;    //avoid libc abs
#else
            Scalar absDet; absDet.setAbs(determinant);
#endif

            Vec4 c0_abs; c0_abs.setAbs(c0);
            Vec4 c1_abs; c1_abs.setAbs(c1);
            Vec4 c2_abs; c2_abs.setAbs(c2);
            Scalar feps = Scalar::template getConstant<HK_QUADREAL_EPS>();
            Scalar eps =   (feps * c0_abs.template horizontalAdd<3>())
                                * (c1_abs.template horizontalAdd<3>() * c2_abs.template horizontalAdd<3>());
            isNotSingular = absDet.greater( eps );
            break;
        }
    case HK_DETERMINANT_CHECK_ZERO:
        {
            // we are doing this strange determinant * hkSimdReal_Inv2 to get correct nan and inf behavior
#if (defined(HK_PLATFORM_LINUX) && HK_CONFIG_SIMD == HK_CONFIG_SIMD_DISABLED) // see COM-2686
            Scalar absDet = (determinant.isGreaterEqualZero()) ? determinant : -determinant;    //avoid libc abs
#else
            Scalar absDet; absDet.setAbs(determinant);
#endif

            isNotSingular = absDet.greater( absDet  * Scalar::template getConstant<HK_QUADREAL_INV_2>() );
            break;
        }
    default:
        {
            break;
        }
    }

    Scalar dInv;    dInv.template setReciprocal<ACCURACY, HK_DIV_IGNORE>(determinant);

    if (FAILURE == HK_INVERSION_FAILURE_SET_ZERO_KEEP_NANS)
    {
        dInv.zeroIfFalse(isNotSingular);
    }
    if ( TYPE == HK_MATRIX_ANY )
    {
        hkMath::transpose( r0,r1,r2 );
    }
    else
    {
        HK_ON_DEBUG( FT norm = (src.frobeniusNormSqr().template sqrt<HK_ACC_12_BIT, HK_SQRT_SET_ZERO>().getReal()) );
        HK_ON_DEBUG( bool isSym = src.isSymmetric( HK_FLOAT_EPSILON*4.0f * norm ) );
        HK_WARN_ONCE_ON_DEBUG_IF( !isSym, 0xf0546798, "Trying to invert a non symmetric matrix" );
    }

    m_col0.setMul(r0, dInv);
    m_col1.setMul(r1, dInv);
    m_col2.setMul(r2, dInv);
    if ( FAILURE == HK_INVERSION_FAILURE_SET_IDENTITY )
    {
        m_col0.setSelect( isNotSingular, m_col0, Vec4::template getConstant<HK_QUADREAL_1000>() );
        m_col1.setSelect( isNotSingular, m_col1, Vec4::template getConstant<HK_QUADREAL_0100>() );
        m_col2.setSelect( isNotSingular, m_col2, Vec4::template getConstant<HK_QUADREAL_0010>() );
    }
    if ( FAILURE == HK_INVERSION_FAILURE_SET_ZERO )
    {
        m_col0.zeroIfFalse(isNotSingular);
        m_col1.zeroIfFalse(isNotSingular);
        m_col2.zeroIfFalse(isNotSingular);
    }

    return determinant;
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::mul( Scalar_ scale)
{
    m_col0.mul(scale);
    m_col1.mul(scale);
    m_col2.mul(scale);
}


//
//  Adds the same value to all elements of the diagonal
template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::addDiagonal(Scalar_ diag)
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    m_col0.addMul(diag, Vec4::template getConstant<HK_QUADREAL_1000>());
    m_col1.addMul(diag, Vec4::template getConstant<HK_QUADREAL_0100>());
    m_col2.addMul(diag, Vec4::template getConstant<HK_QUADREAL_0010>());
#else
    const FT d = diag.getReal();
    m_col0(0) += d;
    m_col1(1) += d;
    m_col2(2) += d;
#endif
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::multiplyVector (Vec4_ v, Vec4& resultOut) const
{
    const Scalar xb = v.template getComponent<0>();
    const Scalar yb = v.template getComponent<1>();
    const Scalar zb = v.template getComponent<2>();

    Vec4 t0;
    t0.setMul( xb, m_col0 );
    t0.addMul( yb, m_col1 );
    t0.addMul( zb, m_col2 );

    resultOut = t0;
}

//
//  Add the product of a and scale (this += a * scale)

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::addMul( Scalar_ scale, const hkMatrix3Impl& a)
{
    m_col0.addMul(scale, a.m_col0);
    m_col1.addMul(scale, a.m_col1);
    m_col2.addMul(scale, a.m_col2);
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::setZero()
{
    m_col0.setZero();
    m_col1.setZero();
    m_col2.setZero();
}

template<typename FT>
HK_INLINE void hkMatrix3Impl<FT>::setIdentity()
{
    hkMatrix3Impl* HK_RESTRICT d = this;
    d->m_col0 = Vec4::template getConstant<HK_QUADREAL_1000>();
    d->m_col1 = Vec4::template getConstant<HK_QUADREAL_0100>();
    d->m_col2 = Vec4::template getConstant<HK_QUADREAL_0010>();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
