// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#ifndef HK_MATH_MATH_H
#   error Please include Common/Base/hkBase.h instead of this file.
#endif

/// A 3x3 matrix of hkReals.
/// Due to alignment issues internal storage is 12 hkReals in a 4x3 matrix.
/// Elements are stored in column major format.
///
/// i.e., contiguous memory locations are (x00, x10, x20, pad), (x01, x11,...)
/// where x10 means row 1, column 0 for example.
template<typename FT>
class hkMatrix3Impl
{
public:
    HK_DECLARE_CLASS(hkMatrix3Impl, New, Pod, Reflect);
    HK_REFLECT_AS_ARRAY_FIXED(12, FT);
    HK_RECORD_ATTR(hk::TypeLayout(hkMath::computeTypeSizeAlign));
    HK_RECORD_ATTR(hk::ReflectDetails(fields=false));
    HK_RECORD_ATTR(hk::Default({1, 0, 0, 0}, {0, 1, 0, 0}, {0, 0, 1, 0}));
    HK_RECORD_ATTR(hk::IncludeInMgd(false));

    typedef hkAlignedQuad<FT> ReflectDefaultType[3];

    typedef typename hkFloatTypes<FT>::Vec4  Vec4;
    typedef typename hkFloatTypes<FT>::Vec4_ Vec4_;

    typedef typename hkFloatTypes<FT>::Scalar  Scalar;
    typedef typename hkFloatTypes<FT>::Scalar_ Scalar_;

    typedef hkMatrix3Impl<FT>  Matrix3;
    typedef hkRotationImpl<FT> Rotation;

    typedef typename hkFloatTypes<FT>::Transform  Transform;
    typedef typename hkFloatTypes<FT>::Quaternion Quaternion;
    typedef typename hkFloatTypes<FT>::Comparison Comparison;


#ifndef HK_DISABLE_MATH_CONSTRUCTORS
    /// Empty constructor. The elements of the matrix are not initialized.
    HK_ALWAYS_INLINE hkMatrix3Impl() {}

    /// Copy matrix from 'other'.
    HK_INLINE hkMatrix3Impl( const hkMatrix3Impl& other );
#endif

    /// convert from a rotation
    HK_INLINE void set( const Rotation& r );

    /// Copies all elements from a into this matrix.
    HK_INLINE void operator= ( const hkMatrix3Impl& a );

    /// Gets a read-write reference to the i'th column.
    HK_ALWAYS_INLINE Vec4& getColumn(int i);

    /// Gets a read reference to the i'th column.
    HK_ALWAYS_INLINE const Vec4& getColumn(int i) const;

    /// Gets a read-write reference to the i'th column.
    template <int I> HK_ALWAYS_INLINE Vec4& getColumn();

    /// Gets a read reference to the i'th column.
    template <int I> HK_ALWAYS_INLINE const Vec4& getColumn() const;

    /// set column
    template <int I> HK_ALWAYS_INLINE void setColumn(Vec4_ c);


    /// Returns a copy of the i'th row.
    HK_ALWAYS_INLINE void getRow( int row, Vec4& r) const;
    template <int I> HK_ALWAYS_INLINE void getRow(Vec4& r) const;

    /// Gets read-write access to the specified element.
    HK_ALWAYS_INLINE FT& operator() (int row, int col);

    /// Gets read-only access to the specified elements.
    HK_ALWAYS_INLINE const FT& operator() (int row, int col) const;

    template <int ROW, int COL> HK_ALWAYS_INLINE Scalar getElement() const;
    template <int ROW, int COL> HK_ALWAYS_INLINE void setElement(Scalar_ s);

    /// Sets all rows at once.
    HK_INLINE void setRows( Vec4_ r0, Vec4_ r1, Vec4_ r2);

    /// Writes the rows 0 to 2 in to the parameters r0, r1 and r2 respectively.
    HK_INLINE void getRows( Vec4& r0, Vec4& r1, Vec4& r2) const;

    /// Sets all columns of the current matrix. Where column is set to c0 and so on.
    HK_INLINE void setCols( Vec4_ c0, Vec4_ c1, Vec4_ c2);

    /// Writes the columns 0 to 2 into the parameters c0, c1 and c2 respectively.
    HK_INLINE void getCols( Vec4& c0, Vec4& c1, Vec4& c2) const;

    /// Zeroes all values in this matrix.
    HK_INLINE void setZero();

    /// Sets the diagonal values to 1, zeroes the non-diagonal values.
    HK_INLINE void setIdentity();

    /// Returns a global identity matrix3.
    
    HK_ALWAYS_INLINE static const hkMatrix3Impl& HK_CALL getIdentity()
    {
        union { const Vec4* r; const hkMatrix3Impl<FT>* m; } r2m;
        r2m.r = &Vec4::getConstant(HK_QUADREAL_1000);
        return *r2m.m;
    }

    /// Checks if this matrix is equal to m within an optional epsilon.
    HK_EXPORT_COMMON bool isApproximatelyEqual(const hkMatrix3Impl& m, const FT& epsilon = FT(1e-3f)) const;

    /// Checks if this matrix is symmetric to within an optional epsilon.
    HK_EXPORT_COMMON bool isSymmetric(FT epsilon = FT(1e-3f)) const;

    /// Sets this to be r star (cross skew symmetric of vector r).
    HK_EXPORT_COMMON void setCrossSkewSymmetric(Vec4_ r);

    /// Set the inverse if possible, else produces a zero matrix
    HK_EXPORT_COMMON hkResult setInverse(const hkMatrix3Impl& src);

    /// Sets this to the inverse of m and return the determinant of m, see enum
    template<hkMatrixSpecialization TYPE, hkMathAccuracyMode ACCURACY, hkDeterminantCheck CHECK, hkInversionFailure FAILURE >
    HK_INLINE Scalar _setInverse( const hkMatrix3Impl& m );

    /// Frobenius norm squared = sum of all elements squared
    HK_EXPORT_COMMON Scalar frobeniusNormSqr() const;

    /// Transposes this matrix in place.
    HK_EXPORT_COMMON void transpose();

    /// set to the transpose of another matrix
    HK_EXPORT_COMMON void setTranspose(const hkMatrix3Impl& s);

    /// set to the transpose of another matrix, inline version
    HK_INLINE void _setTranspose( const hkMatrix3Impl& s );

    HK_INLINE void setDiagonal( Vec4_ f);

    /// Set this matrix to be the product of a and b. (this = a * b)
    HK_EXPORT_COMMON void setMul(const hkMatrix3Impl& a, const hkMatrix3Impl& b);

    /// Set this matrix to be the product of qa and b. (this = qa * b)
    HK_EXPORT_COMMON void setMul(const Quaternion& qa, const hkMatrix3Impl& b);

    /// Sets this matrix to be the product of a and the inverse of b. (this = a * b^-1)
    HK_EXPORT_COMMON void setMulInverse(const hkMatrix3Impl& a, const Rotation& b);

    /// Sets this matrix to be the product of the inverse of a and b. (this = a^-1 * b)
    HK_EXPORT_COMMON void setMulInverseMul(const Rotation& a, const hkMatrix3Impl& b);

    /// Sets this matrix to be the product of a and scale. (this = a * scale)
    HK_INLINE void setMul( Scalar_ scale, const hkMatrix3Impl& a );

    /// Sets this matrix to be the product of a diagonal matrix (given as a vector) and matrix a. (this = diag * a)
    HK_INLINE void setDiagonalMul(Vec4_ vDiag, const hkMatrix3Impl& a);

    /// Rotates this matrix by the rotation matrix t. (this = t * this * t^-1)
    HK_EXPORT_COMMON void changeBasis(const Rotation& t);

    /// Adds the same value to all elements of the diagonal
    HK_INLINE void addDiagonal(Scalar_ diag);

    /// Sets this = Transpose(a) * b.
    HK_EXPORT_COMMON void setTransposeMul(const hkMatrix3Impl& a, const hkMatrix3Impl& b);

    /// Add the product of a and scale. (this += a * scale)
    HK_INLINE void addMul( Scalar_ scale, const hkMatrix3Impl& a );

    /// Modifies this matrix by adding the matrix a to it. (this += a)
    HK_INLINE void add( const hkMatrix3Impl& a );

    /// Modifies this matrix by subtracting the matrix a from it. (this += a)
    HK_INLINE void sub( const hkMatrix3Impl& a );

    /// Modifies this matrix by post multiplying it by the matrix a. (this = this*a)
    HK_EXPORT_COMMON void mul(const hkMatrix3Impl& a);

    /// Modifies this matrix by multiplying by scale. (this *= scale)
    HK_INLINE void mul( Scalar_ scale );


    /// Checks for bad values (denormals or infinities)
    HK_EXPORT_COMMON bool isOk() const;

    /// Multiplies a 3-element vector by this 3x3 matrix.
    HK_INLINE void multiplyVector (Vec4_ vectorIn, Vec4& resultOut) const;

    /// operator *
    HK_ALWAYS_INLINE Vec4 operator*( Vec4_ a ) const { Vec4 r; multiplyVector(a,r); return r; }

    /// Returns the determinant of this 3x3 matrix.
    HK_EXPORT_COMMON const Scalar getDeterminant() const;

    /// Diagonalizes a symmetric matrix, returning the eigenvectors in the columns of eigenVec (which is orthogonal) and the
    /// corresponding eigenvalues in eigenVal, with a specified tolerance epsilon and maximum number of Jacobi iterations maxIter.
    /// Returns HK_SUCCESS if the fractional off-diagonal norm was reduced below the specified epsilon, otherwise HK_FAILURE.
    /// You can interpret the eigenVec as a matrix which rotates points from eigenVal space to this space.
    HK_EXPORT_COMMON hkResult diagonalizeSymmetric(Rotation& eigenVec, Vec4& eigenVal, int maxIter = 20, FT epsilon = hkFloatTypes<FT>::epsilon()) const;

    /// Diagonalizes a symmetric matrix using cached eigenvectors, returning the eigenvectors in the columns of eigenVec (which is orthogonal) and the
    /// corresponding eigenvalues in eigenVal, with a specified tolerance epsilon and maximum number of Jacobi iterations maxIter.
    /// Returns HK_SUCCESS if the fractional off-diagonal norm was reduced below the specified epsilon, otherwise HK_FAILURE.
    HK_EXPORT_COMMON hkResult diagonalizeSymmetricWarmStart(hkMatrix3Impl& eigenVec, Vec4& eigenVal, int maxIter = 20, FT epsilon = hkFloatTypes<FT>::epsilon()) const;

    /// Like diagonalizeSymmetric() but without epsilon checks. Diagonalize by just iterating the max iter of times, no error checking.
    HK_EXPORT_COMMON void diagonalizeSymmetricApproximation(Rotation& eigenVec, Vec4& eigenVal, int maxIter = 10) const;

public:

    Vec4 m_col0;
    Vec4 m_col1;
    Vec4 m_col2;
};

#if 0 && defined( HK_DYNAMIC_DLL )
/*HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION*/ template class /*HK_EXPORT_COMMON*/ hkMatrix3Impl<float>;
/*HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION*/ template class /*HK_EXPORT_COMMON*/ hkMatrix3Impl<double>;
#endif

extern "C"
{
    // implementation of _setMul as out of line function with C-binding
    void HK_EXPORT_COMMON hkMatrix3f_setMulMat3Mat3(_Inout_ hkMatrix3f* thisMatrix, const hkMatrix3f& aTb, const hkMatrix3f& bTc );
    void HK_EXPORT_COMMON hkMatrix3f_invertSymmetric( hkMatrix3f& thisMatrix );
    void HK_EXPORT_COMMON hkMatrix3d_setMulMat3Mat3(_Inout_ hkMatrix3d* thisMatrix, const hkMatrix3d& aTb, const hkMatrix3d& bTc );
    void HK_EXPORT_COMMON hkMatrix3d_invertSymmetric( hkMatrix3d& thisMatrix );

# if defined(HK_REAL_IS_DOUBLE)
    // implementation of _setMul as out of line function with C-binding
    HK_INLINE void hkMatrix3_setMulMat3Mat3(_Inout_ hkMatrix3d* thisMatrix, const hkMatrix3d& aTb, const hkMatrix3d& bTc) { hkMatrix3d_setMulMat3Mat3(thisMatrix, aTb, bTc); }
    HK_INLINE void hkMatrix3_invertSymmetric( hkMatrix3d& thisMatrix )  {   hkMatrix3d_invertSymmetric( thisMatrix );   }
# else
    // implementation of _setMul as out of line function with C-binding
    HK_INLINE void hkMatrix3_setMulMat3Mat3(_Inout_ hkMatrix3f* thisMatrix, const hkMatrix3f& aTb, const hkMatrix3f& bTc )  {   hkMatrix3f_setMulMat3Mat3( thisMatrix, aTb, bTc );  }
    HK_INLINE void hkMatrix3_invertSymmetric( hkMatrix3f& thisMatrix )  {   hkMatrix3f_invertSymmetric( thisMatrix );   }
# endif
}

HK_ON_REAL_IS_FLOAT ( typedef hkMatrix3f hkMatrix3; )
HK_ON_REAL_IS_DOUBLE( typedef hkMatrix3d hkMatrix3; )

#if defined(HK_COMPILER_ARMCC) || defined(HK_COMPILER_GHS)
template<> hkReflect::Detail::TypeData hkMatrix3Impl<float>::typeData;
template<> hkReflect::Detail::TypeData hkMatrix3Impl<double>::typeData;
#endif

#include <Common/Base/_Auto/TemplateTypes/hkMatrix3_Types.inl>

HK_REFLECT_TYPEDEF( , hkMatrix3f, hkMatrix3f_Tag);
HK_REFLECT_TYPEDEF( , hkMatrix3d, hkMatrix3d_Tag);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
