// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>

HK_DETAIL_DIAG_MSVC_OFF(4661) //  no suitable definition provided for explicit template instantiation request

#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/Base/Math/Matrix/hkMatrix3Util.h>
#include <cmath>

template<typename FT>
bool hkMatrix3Impl<FT>::isOk() const
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    bool col0Ok = m_col0.template isOk<3>();
    bool col1Ok = m_col1.template isOk<3>();
    bool col2Ok = m_col2.template isOk<3>();
    return col0Ok && col1Ok && col2Ok;
#else
    for(int c=0; c<3; ++c)
    {
        for (int r=0; r<3; ++r)
        {
            const FT& ff = ((&m_col0)[c])(r);
            if( hkMath::isFinite(ff) == false )
            {
                return false;
            }
        }
    }
    return true;
#endif
}

template<typename FT>
void hkMatrix3Impl<FT>::transpose()
{
    hkMath::transpose(m_col0,m_col1,m_col2);
}

template<typename FT>
void hkMatrix3Impl_setMulMat3Mat3( hkMatrix3Impl<FT>* thisMtx, const hkMatrix3Impl<FT>& aTb, const hkMatrix3Impl<FT>& bTc )
{
    HK_ASSERT_NO_MSG(0x6d9d1d43,  thisMtx != &aTb );
    hkVector4fUtil::rotatePoints( aTb, &bTc.getColumn(0), 3, &thisMtx->getColumn(0) );
}

template<typename FT>
void hkMatrix3Impl_invertSymmetric( hkMatrix3Impl<FT>& thisMatrix )
{
    thisMatrix._setInverse<HK_MATRIX_SYMMETRIC, HK_ACC_MID, HK_DETERMINANT_CHECK_EPS, HK_INVERSION_FAILURE_SET_ZERO>(thisMatrix);
}


template<typename FT>
void hkMatrix3Impl<FT>::setTranspose( const hkMatrix3Impl<FT>& s )
{
    _setTranspose( s );
}


//
//  Set this matrix to be the product of qa and b.  (this = qa * b)
template<typename FT>
void hkMatrix3Impl<FT>::setMul(const Quaternion& qa, const hkMatrix3Impl& b)
{
    Vec4 c0,c1,c2;  // temp variables to allow the compiler to interleave operations
    c0._setRotatedDir(qa, b.m_col0);
    c1._setRotatedDir(qa, b.m_col1);
    c2._setRotatedDir(qa, b.m_col2);

    m_col0 = c0;
    m_col1 = c1;
    m_col2 = c2;
}

/// aTc = aTb * bTc
template<typename FT>
void hkMatrix3Impl<FT>::setMul( const hkMatrix3Impl& aTb, const hkMatrix3Impl& bTc )
{
    HK_ASSERT_NO_MSG(0x6d9d1d43,  this != &aTb );

    Vec4 a,b,c; // temp variables to allow the compiler to interleave operations
    a._setRotatedDir(aTb, bTc.m_col0);
    b._setRotatedDir(aTb, bTc.m_col1);
    c._setRotatedDir(aTb, bTc.m_col2);
    m_col0 = a;
    m_col1 = b;
    m_col2 = c;
}


template<typename FT>
void hkMatrix3Impl<FT>::changeBasis(const Rotation& r)
{
    Rotation temp;
    temp.setMulInverse(*this, r);
    this->setMul(r, temp);
}

template<typename FT>
void hkMatrix3Impl<FT>::setMulInverse( const hkMatrix3Impl<FT>& aTb, const Rotation& cTb )
{
    HK_ASSERT_NO_MSG(0xf032e412, this != (hkMatrix3Impl<FT>*)&aTb );
    typename hkFloatTypes<FT>::Rotation invB;   invB._setTranspose(cTb);
    hkVector4UtilImpl<FT>::rotatePoints(aTb, &invB.template getColumn<0>(), 3, &this->template getColumn<0>());
}

template<typename FT>
void hkMatrix3Impl<FT>::setMulInverseMul( const Rotation& bTa, const hkMatrix3Impl<FT>& bTc )
{
    HK_ASSERT_NO_MSG(0xf032e412, this != (hkMatrix3Impl<FT>*)&bTa );
    hkVector4UtilImpl<FT>::rotateInversePoints( bTa, &bTc.getColumn(0), 3, &m_col0 );
}

//
//  Sets this = Transpose(a) * b.

template<typename FT>
void hkMatrix3Impl<FT>::setTransposeMul(const hkMatrix3Impl<FT>& a, const hkMatrix3Impl<FT>& b)
{
    hkMatrix3UtilImpl<FT>::_computeTransposeMul(a, b, *this);
}


template<typename FT>
void hkMatrix3Impl<FT>::setCrossSkewSymmetric( Vec4_ r )
{
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    Scalar zero; zero.setZero();
    const Scalar r0 = r.template getComponent<0>();
    const Scalar r1 = r.template getComponent<1>();
    const Scalar r2 = r.template getComponent<2>();

    m_col0.set( zero,   r2,  -r1, zero );
    m_col1.set(  -r2, zero,   r0, zero );
    m_col2.set(   r1,  -r0, zero, zero );
#else
    m_col0.set(  FT(0)   ,  r(2), -r(1) );
    m_col1.set( -r(2),     FT(0), +r(0) );
    m_col2.set(  r(1), -r(0),     FT(0) );
#endif
}

//
//  Add the product of a and scale (this += a * scale)


template<typename FT>
bool hkMatrix3Impl<FT>::isApproximatelyEqual( const hkMatrix3Impl<FT>& m, const FT& zero) const
{
    Scalar eps; eps.setFromFloat(zero);

#if HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED
    Vec4 d0; d0.setSub( m.getColumn(0), getColumn(0) );
    Vec4 d1; d1.setSub( m.getColumn(1), getColumn(1) );
    Vec4 d2; d2.setSub( m.getColumn(2), getColumn(2) );
    d0.setAbs(d0);
    d1.setAbs(d1);
    d2.setAbs(d2);
    d0.setMax(d0,d1);
    d0.setMax(d0,d2);
    Vec4 epsV; epsV.setAll( eps );
    return d0.lessEqual( epsV ).template allAreSet<hkVector4ComparisonMask::MASK_XYZ>();
#else
    return     m_col0.template allEqual<3>( m.getColumn<0>(), eps )
            && m_col1.template allEqual<3>( m.getColumn<1>(), eps )
            && m_col2.template allEqual<3>( m.getColumn<2>(), eps );
#endif
}

template<typename FT>
bool hkMatrix3Impl<FT>::isSymmetric(FT epsilon) const
{
    hkMatrix3Impl<FT> T;
    T._setTranspose(*this);
    bool result = this->isApproximatelyEqual(T, epsilon);
    return result;
}


template<typename FT>
hkResult hkMatrix3Impl<FT>::setInverse( const hkMatrix3Impl<FT>& src )
{
    // Get source columns
    const Vec4 c0 = src.template getColumn<0>();
    const Vec4 c1 = src.template getColumn<1>();
    const Vec4 c2 = src.template getColumn<2>();

    Vec4 r0; r0.setCross(c1, c2);
    Vec4 r1; r1.setCross(c2, c0);
    Vec4 r2; r2.setCross(c0, c1);

    const Scalar determinant = c0.template dot<3>(r0);

    {
        // Compute 1 / determinant. Set it to zero in case of singular matrices!
        // We detect singularity by scaling the determinant by the inverse
        // L1 norms of the columns of the matrix and comparing to epsilon.
        Scalar absDet; absDet.setAbs(determinant);
        Vec4 c0_abs; c0_abs.setAbs(c0);
        Vec4 c1_abs; c1_abs.setAbs(c1);
        Vec4 c2_abs; c2_abs.setAbs(c2);

        Scalar eps = (Scalar::template getConstant<HK_QUADREAL_EPS>() * c0_abs.template horizontalAdd<3>())
            * (c1_abs.template horizontalAdd<3>() * c2_abs.template horizontalAdd<3>());
#if (defined(HK_ARCH_X64) || defined(HK_ARCH_IA32)) && (HK_CONFIG_SIMD == HK_CONFIG_SIMD_DISABLED)
#   if defined(_MSC_VER) && (_MSC_VER == 1700)
        // VS2012 does not correctly declare std::isnan
        if ( (absDet <= eps) || _isnan(eps.getReal()) )
#   else
        if ( (absDet <= eps) || std::isnan(eps.getReal()) )
#   endif
#elif defined(_MSC_VER) && (_MSC_VER == 1900) && defined(_MSC_FULL_VER) && (_MSC_FULL_VER < 190024210) && (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
        // NaN behavior of _mm_ucomigt_ss is strange in VS2015 RTM, fixed in Update 3
        if (absDet <= eps)
#else
        if ( !(absDet > eps) )
#endif
        {
            m_col0.setZero();
            m_col1.setZero();
            m_col2.setZero();
            return HK_FAILURE;
        }
    }

    Scalar dInv;    dInv.template setReciprocal<HK_ACC_MID, HK_DIV_IGNORE>(determinant);

    Matrix3 h; h.setRows( r0,r1,r2 );

    m_col0.setMul( h.m_col0, dInv);
    m_col1.setMul( h.m_col1, dInv);
    m_col2.setMul( h.m_col2, dInv);
    return HK_SUCCESS;
}


template<typename FT>
void hkMatrix3Impl<FT>::mul( const hkMatrix3Impl<FT>& a )
{
    
    hkMatrix3Impl<FT> temp;
    temp.setMul( *this, a );
    *this = temp;
}

template<typename FT>
const typename hkMatrix3Impl<FT>::Scalar hkMatrix3Impl<FT>::getDeterminant() const
{
    Vec4 r0, r1, r2;
    getRows(r0, r1, r2);

    Vec4 r1r2;
    r1r2.setCross(r1,r2);
    return r0.template dot<3>(r1r2);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// Diagonalization of symmetric matrix - based on sec 8.4, Jacobi Methods, of Golub & Van Loan
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////

template<typename FT>
struct HK_EXPORT_COMMON SchurMatrix
{
    FT c, s;
    int p, q;
};

template<typename FT>
static HK_INLINE void _constructSchurMatrix(const hkMatrix3Impl<FT>& M, int p, int q, SchurMatrix<FT>& S)
{
    S.p = p;
    S.q = q;
    FT Mpq = M(p,q);
    if ( Mpq != FT(0) )
    {
        FT tau = ( M(q,q) - M(p,p) ) / (FT(2) * Mpq);
        FT t = hkMath::sqrt(FT(1) + tau*tau);
        if (tau>=FT(0))  t = FT(1) / (tau + t);
        else                 t = FT(1) / (tau - t);
        S.c = hkMath::sqrtInverse(FT(1) + t*t);
        S.s = t * S.c;
    }
    else
    {
        S.c = FT(1);
        S.s = FT(0);
    }
}

template<typename FT>
static HK_INLINE void _constructSchurMatrixUnchecked(const hkMatrix3Impl<FT>& M, int p, int q, SchurMatrix<FT>& S)
{
    S.p = p;
    S.q = q;
    FT Mpq = M(p,q);
    FT tau = ( M(q,q) - M(p,p) ) / (FT(2) * Mpq);
    FT t = hkMath::sqrt(FT(1) + tau*tau);
    if (tau>=FT(0))  t = FT(1) / (tau + t);
    else                 t = FT(1) / (tau - t);
    S.c = hkMath::sqrtInverse(FT(1) + t*t);
    S.s = t * S.c;
}

template<typename FT>
static HK_INLINE const typename hkMatrix3Impl<FT>::Scalar _frobeniusNormSqr(const hkMatrix3Impl<FT>& M)
{
    return M.template getColumn<0>().template lengthSquared<3>() + M.template getColumn<1>().template lengthSquared<3>() + M.template getColumn<2>().template lengthSquared<3>();
}

template<typename FT>
typename hkMatrix3Impl<FT>::Scalar hkMatrix3Impl<FT>::frobeniusNormSqr() const
{
    return _frobeniusNormSqr(*this);
}


template<typename FT>
static HK_INLINE const typename hkMatrix3Impl<FT>::Scalar _offDiagNormSqr(const hkMatrix3Impl<FT>& M)
{
    const typename hkMatrix3Impl<FT>::Scalar m01 = M.template getElement<0,1>();
    const typename hkMatrix3Impl<FT>::Scalar m02 = M.template getElement<0,2>();
    const typename hkMatrix3Impl<FT>::Scalar m12 = M.template getElement<1,2>();
    return hkMatrix3Impl<FT>::Scalar::getConstant(HK_QUADREAL_2)*(m01*m01 + m02*m02 + m12*m12);
}

// returns the fabs of the largest off diagonal
template<typename FT>
static HK_INLINE FT _findLargestOffDiagEntry(const hkMatrix3Impl<FT>& M, int& p, int& q)
{
    p = 0; q = 1;
    FT maxent = hkMath::fabs(M(0,1));
    FT mag02  = hkMath::fabs(M(0,2));
    FT mag12  = hkMath::fabs(M(1,2));
    if ( mag02 > maxent )
    {
        p = 0; q = 2; maxent = mag02;
    }
    if ( mag12 > maxent )
    {
        p = 1; q = 2;
        return mag12;
    }
    return maxent;
}

template<typename FT,bool transpose>
static HK_INLINE void _constructJacobiRotation(hkMatrix3Impl<FT>& J, const SchurMatrix<FT>& S)
{
    J.setIdentity();
    J(S.p, S.p) = S.c;
    if ( !transpose)
    {
        J(S.p, S.q) = S.s;
        J(S.q, S.p) = -S.s;
    }
    else
    {
        J(S.p, S.q) = -S.s;
        J(S.q, S.p) = S.s;
    }

    J(S.q, S.q) = S.c;
}


template<typename FT>
hkResult hkMatrix3Impl<FT>::diagonalizeSymmetric(
    typename hkMatrix3Impl<FT>::Rotation& eigenVec,
    typename hkMatrix3Impl<FT>::Vec4& eigenVal, int maxIter, FT epsilon) const
{
#ifdef HK_DEBUG_SLOW
    if (!isSymmetric(epsilon))
    {
        HK_WARN(0x15e84635, "Attempted to diagonalize an unsymmetric matrix in hkMatrix3Impl<FT>::diagonalizeSymmetric");
        return HK_FAILURE;
    }
#endif

    hkMatrix3Impl<FT> M(*this);
    eigenVec.setIdentity();

    Scalar eps; eps.setFromFloat(epsilon);
    const Scalar epsSqr = eps*eps * _frobeniusNormSqr(*this);
    int nIter = 0;
    Scalar normSqr; normSqr.setZero();

    for (;;)
    {
        normSqr = _offDiagNormSqr(M);
        if ( normSqr<=epsSqr || nIter>=maxIter ) break;

        int p = 0, q = 0;
        _findLargestOffDiagEntry(M, p, q);

        SchurMatrix<FT> S;
        _constructSchurMatrix(M, p, q, S);

        hkMatrix3Impl<FT> J;  _constructJacobiRotation<FT,false>(J, S);
        hkMatrix3Impl<FT> Jt; _constructJacobiRotation<FT,true>(Jt, S);

        // M <- J^T M J
        M.mul(J);
        M.setMul(Jt,M);

        // V <- V J
        eigenVec.mul(J);

        nIter++;
    }

    eigenVal.set(M.getElement<0,0>(), M.getElement<1,1>(), M.getElement<2,2>(), M.getElement<2,2>());

    if (normSqr>epsSqr)
    {
        return HK_FAILURE;
    }
    return HK_SUCCESS;
}


template<typename FT>
hkResult hkMatrix3Impl<FT>::diagonalizeSymmetricWarmStart(
    hkMatrix3Impl<FT>& eigenVec, typename hkMatrix3Impl<FT>::Vec4& eigenVal, int maxIter, FT epsilon) const
{
#ifdef HK_DEBUG_SLOW
    if (!isSymmetric(epsilon))
    {
        HK_WARN(0x15e84635, "Attempted to diagonalize an unsymmetric matrix in hkMatrix3Impl<FT>::diagonalizeSymmetricWarmStart");
        return HK_FAILURE;
    }
#endif

    //Warm start, m = eigenVecTrans*m*eigenVec
    hkMatrix3Impl<FT> tempM(*this);
    hkMatrix3Impl<FT> eigenVecTrans(eigenVec);
    eigenVecTrans.transpose();

    hkMatrix3Impl<FT> m;
    m.setMul(eigenVecTrans, tempM);
    m.mul(eigenVec);

    Scalar eps; eps.setFromFloat(epsilon);
    const Scalar epsSqr = eps*eps * _frobeniusNormSqr(*this);
    int nIter = 0;
    Scalar normSqr; normSqr.setZero();

    for (;;)
    {
        normSqr = _offDiagNormSqr(m);
        if ( normSqr<epsSqr || nIter>=maxIter ) break;

        int p = 0, q = 0;
        _findLargestOffDiagEntry(m, p, q);

        SchurMatrix<FT> schur;
        _constructSchurMatrix(m, p, q, schur);

        hkMatrix3Impl<FT> jacobi;
        _constructJacobiRotation<FT,false>(jacobi, schur);

        // m <- jacobiTrans*m*jacobi
        hkMatrix3Impl<FT> jacobiTrans(jacobi); jacobiTrans.transpose();
        m.mul(jacobi);
        m.setMul(jacobiTrans,m);

        // eigenVec <- eigenVec*jacobi
        eigenVec.mul(jacobi);

        nIter++;
    }

    Vec4& col0 = eigenVec.getColumn(0);
    Vec4& col1 = eigenVec.getColumn(1);
    Vec4& col2 = eigenVec.getColumn(2);

    col0.template normalize<3>();
    col1.template normalize<3>();
    col2.template normalize<3>();

    eigenVal.set(m.getElement<0,0>(), m.getElement<1,1>(), m.getElement<2,2>(), m.getElement<2,2>());

    if (normSqr>epsSqr)
    {
        return HK_FAILURE;
    }
    return HK_SUCCESS;
}


template<typename FT>
void hkMatrix3Impl<FT>::diagonalizeSymmetricApproximation(Rotation& eigenVec, Vec4& eigenVal, int maxIter) const
{
    hkMatrix3Impl<FT> M(*this);
    eigenVec.setIdentity();

    const FT eps = hkFloatTypes<FT>::epsilon();
    const FT epsSqr = eps*eps * _frobeniusNormSqr(*this).getReal();

    for (int nIter = 0; nIter<maxIter; nIter++)
    {
        int p = 0, q = 0;
        FT fabsOffDiag = _findLargestOffDiagEntry(M, p, q);

        if ( fabsOffDiag*fabsOffDiag < epsSqr )
        {
            break;
        }

        SchurMatrix<FT> S; _constructSchurMatrixUnchecked(M, p, q, S);

        hkMatrix3Impl<FT> J;  _constructJacobiRotation<FT,false>(J, S);
        hkMatrix3Impl<FT> Jt; _constructJacobiRotation<FT,true>(Jt, S);

        // M <- J^T M J
        M.mul(J);
        M.setMul(Jt,M);

        // V <- V J
        eigenVec.mul(J);

    }
    eigenVal.set(M.getElement<0,0>(), M.getElement<1,1>(), M.getElement<2,2>(), M.getElement<2,2>());
}

void HK_EXPORT_COMMON hkMatrix3f_setMulMat3Mat3(_Inout_ hkMatrix3f* thisMatrix, const hkMatrix3f& aTb, const hkMatrix3f& bTc )
{
    thisMatrix->setMul( aTb, bTc );
}

void HK_EXPORT_COMMON hkMatrix3f_invertSymmetric( hkMatrix3f& thisMatrix )
{
    thisMatrix.setInverse( thisMatrix );
}

void HK_EXPORT_COMMON hkMatrix3d_setMulMat3Mat3(_Inout_ hkMatrix3d* thisMatrix, const hkMatrix3d& aTb, const hkMatrix3d& bTc )
{
    thisMatrix->setMul( aTb, bTc );
}

void HK_EXPORT_COMMON hkMatrix3d_invertSymmetric( hkMatrix3d& thisMatrix )
{
    thisMatrix.setInverse( thisMatrix );
}

template class /*HK_EXPORT_COMMON*/ hkMatrix3Impl<float>;
template class /*HK_EXPORT_COMMON*/ hkMatrix3Impl<double>;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
