// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL !REFLECT
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#if defined(HK_ARCH_X64) && defined(HK_COMPILER_MSVC) && HK_COMPILER_MSVC_VERSION >= 1800
#define HK_LARGE_INT_SUPPORTS_X64_INTRINSICS
#endif

namespace hkLargeIntImpl
{
    typedef char Carry;
    typedef char Borrow;

    HK_INLINE Carry add(hkUint64 a, hkUint64 b, hkUint64& sum)
    {
#if defined(HK_LARGE_INT_SUPPORTS_X64_INTRINSICS)
        // Compiles to a single ADD instruction
        return _addcarry_u64(0, a, b, &sum);
#else
        hkUint64 ab = a + b;
        sum = ab;
        return ab < a;
#endif
    }

    HK_INLINE Carry addWithCarry(Carry carry, hkUint64 a, hkUint64 b, hkUint64& sum)
    {
#if defined(HK_LARGE_INT_SUPPORTS_X64_INTRINSICS)
        // Compiles to a single ADC instruction
        return _addcarry_u64(carry, a, b, &sum);
#else
        hkUint64 ab = a + b;
        Carry carry1 = ab < a;
        hkUint64 abc = ab + carry;
        Carry carry2 = abc < ab;
        sum = abc;
        return carry1 | carry2;
#endif
    }

    HK_INLINE Borrow sub(hkUint64 a, hkUint64 b, hkUint64& diff)
    {
#if defined(HK_LARGE_INT_SUPPORTS_X64_INTRINSICS)
        // Compiles to a single SUB instruction
        return _subborrow_u64(0, a, b, &diff);
#else
        hkUint64 ab = a - b;
        diff = ab;
        return a < b;
#endif
    }

    HK_INLINE Borrow subWithBorrow(Borrow borrow, hkUint64 a, hkUint64 b, hkUint64& diff)
    {
#if defined(HK_LARGE_INT_SUPPORTS_X64_INTRINSICS)
        // Compiles to a single SBB instruction
        return _subborrow_u64(borrow, a, b, &diff);
#else
        hkUint64 ab = a - b;
        Borrow borrow1 = a < b;
        hkUint64 abc = ab - borrow;
        Borrow borrow2 = ab < borrow;
        diff = abc;
        return borrow1 | borrow2;
#endif
    }

    HK_INLINE hkUint64 multiplyUnsigned64(hkUint32 a, hkUint32 b)
    {
#if defined(HK_ARCH_X64)
        return hkUint64(a) * hkUint64(b);
#else
        // This causes MSVC (in some situations) to emit a fast 32x32=>64 mul instead of a full 64bit multiplication
        return hkUint64(a) * hkUint32(b);
#endif
    }

    HK_INLINE hkInt64 multiplySigned64(hkInt32 a, hkInt32 b)
    {
#if defined(HK_ARCH_X64)
        return hkInt64(a) * hkInt64(b);
#else
        // This causes MSVC (in some situations) to emit a fast 32x32=>64 mul instead of a full 64bit multiplication
        return hkInt64(a) * hkInt32(b);
#endif
    }

    HK_INLINE void multiplyUnsigned128(hkUint64 a, hkUint64 b, hkUint64& lowOut, hkUint64& highOut)
    {
#if defined(HK_LARGE_INT_SUPPORTS_X64_INTRINSICS)
        // Compiles to a single MUL instruction
        lowOut = _umul128(a, b, &highOut);
#else
        hkUint32 aHigh = a >> 32;
        hkUint32 aLow = a & 0xFFFFFFFF;

        hkUint32 bHigh = b >> 32;
        hkUint32 bLow = b & 0xFFFFFFFF;

        hkUint64 t = multiplyUnsigned64(aLow, bLow);
        hkUint64 w3 = t & 0xFFFFFFFF;
        hkUint64 k = t >> 32;

        t = multiplyUnsigned64(aHigh, bLow) + k;
        hkUint64 w2 = t & 0xFFFFFFFF;
        hkUint64 w1 = t >> 32;

        t = multiplyUnsigned64(aLow, bHigh) + w2;
        k = t >> 32;

        highOut = multiplyUnsigned64(aHigh, bHigh) + w1 + k;
        lowOut = (t << 32) + w3;
#endif
    }

    HK_INLINE void multiplySigned128(hkInt64 a, hkInt64 b, hkUint64& lowOut, hkUint64& highOut)
    {
#if  defined(HK_LARGE_INT_SUPPORTS_X64_INTRINSICS)
        // Compiles to a single IMUL instruction
        lowOut = _mul128(a, b, reinterpret_cast<hkInt64*>(&highOut));
#else
        hkInt32 aHigh = a >> 32;
        hkUint32 aLow = a & 0xFFFFFFFF;
        hkInt32 bHigh = b >> 32;
        hkUint32 bLow = b & 0xFFFFFFFF;

        hkUint64 t = multiplyUnsigned64(aLow, bLow);
        hkUint64 w3 = t & 0xFFFFFFFF;
        hkUint64 k = t >> 32;

        t = hkInt64(aHigh) * hkInt64(bLow) + k;
        hkUint64 w2 = t & 0xFFFFFFFF;
        hkUint64 w1 = hkInt64(t) >> 32;

        t = hkInt64(aLow) * hkInt64(bHigh) + w2;
        k = hkInt64(t) >> 32;

        highOut = multiplySigned64(aHigh, bHigh) + w1 + k;
        lowOut = (t << 32) + w3;
#endif
    }

    HK_INLINE hkUint64 shiftLeftUnsigned128(hkUint64 low, hkUint64 high, unsigned int shift)
    {
#if defined(HK_LARGE_INT_SUPPORTS_X64_INTRINSICS)
        // Compiles to a single SHLD instruction
        return __shiftleft128(low, high, static_cast<unsigned char>(shift));
#else
        if (shift >= 128)
            return 0;
        if (shift == 0)
            return high;
        if (shift >= 64)
        {
            high = low;
            low = 0;
            shift -= 64;
        }
        high <<= shift;
        low >>= (64 - shift);
        return high | low;
#endif
        }

    HK_INLINE hkUint64 shiftRightUnsigned128(hkUint64 low, hkUint64 high, unsigned int shift)
    {
#if defined(HK_LARGE_INT_SUPPORTS_X64_INTRINSICS)
        // Compiles to a single SHRD instruction
        return __shiftright128(low, high, static_cast<unsigned char>(shift));
#else
        if (shift >= 128)
            return 0;
        if (shift == 0)
            return low;
        if (shift >= 64)
        {
            low = high;
            high = 0;
            shift -= 64;
        }
        low >>= shift;
        high <<= (64 - shift);
        return high | low;
#endif
    }

    HK_EXPORT_COMMON void shiftLeft(_In_reads_(N) const hkUint64* a, _Out_writes_all_(N) hkUint64* out, unsigned int N, unsigned int shift);

    HK_EXPORT_COMMON void shiftRight(_In_reads_(N) const hkUint64* a, _Out_writes_all_(N) hkUint64* out, unsigned int N, unsigned int shift);

    HK_EXPORT_COMMON void multiplyUnsigned(_In_reads_(Na) const hkUint64* a, _In_reads_(Nb) const hkUint64* b, _Out_writes_all_(Nprod) hkUint64* prod, unsigned int Na, unsigned int Nb, unsigned int Nprod);

    HK_EXPORT_COMMON int countLeadingZeros(_In_reads_(N) const hkUint64* a, unsigned int N);
    HK_EXPORT_COMMON int countTrailingZeros(_In_reads_(N) const hkUint64* a, unsigned int N);

    HK_EXPORT_COMMON bool equalZero(_In_reads_(N) const hkUint64* a, unsigned int N);

    HK_EXPORT_COMMON bool equal(_In_reads_(N) const hkUint64* a, _In_reads_(N) const hkUint64* b, unsigned int N);
    }

HK_INLINE void hkInt128::setZero()
{
    m_limbs[0] = 0;
    m_limbs[1] = 0;
}

HK_INLINE void hkInt128::setFromUint32(hkUint32 i)
{
    setFromUint64(hkUint64(i));
}

HK_INLINE void hkInt128::setFromUint64(hkUint64 i)
{
    m_limbs[0] = i;

    m_limbs[1] = 0;
}

HK_INLINE void hkInt128::setFromInt32(hkInt32 i)
{
    setFromInt64(hkInt64(i));
}

HK_INLINE void hkInt128::setFromInt64(hkInt64 i)
{
    m_limbs[0] = hkUint64(i);

    hkUint64 signExt = hkUint64(i >> 63);
    m_limbs[1] = signExt;
}

HK_INLINE void hkInt128::setAbs(const hkInt128& a)
{
    hkUint64 signExt = hkInt64(a.m_limbs[1]) >> 63;

    hkLargeIntImpl::Borrow borrow;
    borrow = hkLargeIntImpl::sub(a.m_limbs[0] ^ signExt, signExt, m_limbs[0]);
    borrow = hkLargeIntImpl::subWithBorrow(borrow, a.m_limbs[1] ^ signExt, signExt, m_limbs[1]);
}

HK_INLINE void hkInt128::setFlipSign(const hkInt128& a, bool cmp)
{
    if (cmp)
    {
        setNeg(a);
    }
    else
    {
        *this = a;
    }
}

HK_INLINE void hkInt128::setSelect(bool cmp, const hkInt128& a, const hkInt128& b)
{
    if (cmp)
    {
        *this = a;
    }
    else
    {
        *this = b;
    }
}

HK_INLINE void hkInt128::setAnd(const hkInt128& a, const hkInt128& b)
{
    m_limbs[0] = a.m_limbs[0] & b.m_limbs[0];
    m_limbs[1] = a.m_limbs[1] & b.m_limbs[1];
}

HK_INLINE void hkInt128::setOr(const hkInt128& a, const hkInt128& b)
{
    m_limbs[0] = a.m_limbs[0] | b.m_limbs[0];
    m_limbs[1] = a.m_limbs[1] | b.m_limbs[1];
}

HK_INLINE void hkInt128::setXor(const hkInt128& a, const hkInt128& b)
{
    m_limbs[0] = a.m_limbs[0] ^ b.m_limbs[0];
    m_limbs[1] = a.m_limbs[1] ^ b.m_limbs[1];
}

HK_INLINE void hkInt128::setNot(const hkInt128& a)
{
    m_limbs[0] = ~a.m_limbs[0];
    m_limbs[1] = ~a.m_limbs[1];
}

HK_INLINE void hkInt128::setShiftLeft(const hkInt128& a, unsigned int shift)
{
    if (shift < 64)
    {
        m_limbs[1] = hkLargeIntImpl::shiftLeftUnsigned128(a.m_limbs[0], a.m_limbs[1], shift);
        m_limbs[0] = a.m_limbs[0] << shift;
    }
    else
    {
        m_limbs[1] = a.m_limbs[0] << (shift - 64);
        m_limbs[0] = 0;
    }
}

HK_INLINE void hkInt128::setShiftRight(const hkInt128& a, unsigned int shift)
{
    m_limbs[0] = hkLargeIntImpl::shiftRightUnsigned128(a.m_limbs[0], a.m_limbs[1], shift);
    m_limbs[1] = a.m_limbs[1] >> shift;
}

HK_INLINE void hkInt128::setShiftRightSigned(const hkInt128& a, unsigned int shift)
{
    if (shift < 64)
    {
        m_limbs[0] = hkLargeIntImpl::shiftRightUnsigned128(a.m_limbs[0], a.m_limbs[1], shift);
        m_limbs[1] = hkInt64(a.m_limbs[1]) >> shift;
    }
    else
    {
        HK_DETAIL_DIAG_MSVC_PUSH()
        HK_DETAIL_DIAG_MSVC_OFF(4293) // shift count negative or too big, undefined behavior
        m_limbs[0] = hkInt64(a.m_limbs[1]) >> (shift - 64);
        m_limbs[1] = hkInt64(a.m_limbs[1]) >> 63;
        HK_DETAIL_DIAG_MSVC_POP()
    }
}


HK_INLINE void hkInt128::increment(hkUint64 a)
{
    hkLargeIntImpl::Carry carry;
    carry = hkLargeIntImpl::add(m_limbs[0], a, m_limbs[0]);
    carry = hkLargeIntImpl::addWithCarry(carry, m_limbs[1], 0, m_limbs[1]);
}

HK_INLINE void hkInt128::decrement(hkUint64 a)
{
    hkLargeIntImpl::Borrow borrow;
    borrow = hkLargeIntImpl::sub(m_limbs[0], a, m_limbs[0]);
    borrow = hkLargeIntImpl::subWithBorrow(borrow, m_limbs[1], 0, m_limbs[1]);
}

HK_INLINE void hkInt128::setNeg(const hkInt128& a)
{
    hkLargeIntImpl::Carry carry;
    carry = hkLargeIntImpl::add(~a.m_limbs[0], 1, m_limbs[0]);
    carry = hkLargeIntImpl::addWithCarry(carry, ~a.m_limbs[1], 0, m_limbs[1]);
}

HK_INLINE void hkInt128::setAdd(const hkInt128& a, const hkInt128& b)
{
    hkLargeIntImpl::Carry carry;
    carry = hkLargeIntImpl::add(a.m_limbs[0], b.m_limbs[0], m_limbs[0]);
    carry = hkLargeIntImpl::addWithCarry(carry, a.m_limbs[1], b.m_limbs[1], m_limbs[1]);
}

HK_INLINE void hkInt128::setSub(const hkInt128& a, const hkInt128& b)
{
    hkLargeIntImpl::Borrow borrow;
    borrow = hkLargeIntImpl::sub(a.m_limbs[0], b.m_limbs[0], m_limbs[0]);
    borrow = hkLargeIntImpl::subWithBorrow(borrow, a.m_limbs[1], b.m_limbs[1], m_limbs[1]);
}

HK_INLINE void hkInt128::setMul(hkUint64 a, hkUint64 b)
{
    hkLargeIntImpl::multiplyUnsigned128(a, b, m_limbs[0], m_limbs[1]);
}

HK_INLINE void hkInt128::setMul(hkInt64 a, hkInt64 b)
{
    hkLargeIntImpl::multiplySigned128(a, b, m_limbs[0], m_limbs[1]);
}

HK_INLINE void hkInt128::setMul(const hkInt128& a, hkInt64 b)
{
    hkUint64 low, high;
    hkLargeIntImpl::multiplyUnsigned128(a.m_limbs[0], b, low, high);

    m_limbs[0] = low;
    m_limbs[1] = high + a.m_limbs[1] * hkUint64(b) - (a.m_limbs[0] & hkUint64(b >> 63));
}

HK_INLINE void hkInt128::setMul(const hkInt128& a, hkUint64 b)
{
    hkUint64 low, high;
    hkLargeIntImpl::multiplyUnsigned128(a.m_limbs[0], b, low, high);

    m_limbs[0] = low;
    m_limbs[1] = high + a.m_limbs[1] * hkUint64(b);
}

HK_INLINE void hkInt128::setMul(const hkInt128& a, const hkInt128& b)
{
    hkUint64 low, high;
    hkLargeIntImpl::multiplyUnsigned128(a.m_limbs[0], b.m_limbs[0], low, high);

    m_limbs[0] = low;
    m_limbs[1] = high + a.m_limbs[0] * b.m_limbs[1] + a.m_limbs[1] * b.m_limbs[0];
}

HK_INLINE int hkInt128::countLeadingZeros() const
{
    return hkLargeIntImpl::countLeadingZeros(m_limbs, HK_COUNT_OF(m_limbs));
}

HK_INLINE int hkInt128::countTrailingZeros() const
{
    return hkLargeIntImpl::countTrailingZeros(m_limbs, HK_COUNT_OF(m_limbs));
}

HK_INLINE bool hkInt128::equalZero() const
{
    return (m_limbs[0] | m_limbs[1]) == 0;
}

HK_INLINE bool hkInt128::lessZero() const
{
    return hkInt64(m_limbs[1]) < 0;
}

HK_INLINE bool hkInt128::greaterZero() const
{
    return !lessZero() && !equalZero();
}


HK_INLINE bool hkInt128::equal(const hkInt128& a) const
{
    return m_limbs[0] == a.m_limbs[0] && m_limbs[1] == a.m_limbs[1];
}

HK_INLINE int hkInt128::getSign() const
{
    return lessZero() ? -1 : equalZero() ? 0 : 1;
}

HK_INLINE hkUint64 hkInt128::getDoubleWord(int N) const
{
    return m_limbs[N];
}

HK_INLINE void hkInt128::setDoubleWord(int N, hkUint64 a)
{
    m_limbs[N] = a;
}

HK_INLINE int hkInt256::getSign() const
{
    return lessZero() ? -1 : equalZero() ? 0 : 1;
}

HK_INLINE hkUint64 hkInt256::getDoubleWord(int N) const
{
    return m_limbs[N];
}

HK_INLINE void hkInt256::setDoubleWord(int N, hkUint64 a)
{
    m_limbs[N] = a;
}

HK_INLINE int hkInt512::getSign() const
{
    return lessZero() ? -1 : equalZero() ? 0 : 1;
}

HK_INLINE hkUint64 hkInt512::getDoubleWord(int N) const
{
    return m_limbs[N];
}

HK_INLINE void hkInt512::setDoubleWord(int N, hkUint64 a)
{
    m_limbs[N] = a;
}

HK_INLINE void hkInt256::setMul(const hkInt256& origA, hkUint64 b)
{
    hkInt256 a = origA;
    hkLargeIntImpl::multiplyUnsigned(a.m_limbs, &b, m_limbs, HK_COUNT_OF(a.m_limbs), 1, HK_COUNT_OF(m_limbs));
}

HK_INLINE void hkInt256::setMul(const hkInt256& origA, const hkInt256& origB)
{
    hkInt256 a = origA;
    hkInt256 b = origB;
    hkLargeIntImpl::multiplyUnsigned(a.m_limbs, b.m_limbs, m_limbs, HK_COUNT_OF(a.m_limbs), HK_COUNT_OF(b.m_limbs), HK_COUNT_OF(m_limbs));
}

HK_INLINE void hkInt256::setAbs(const hkInt256& a)
{
    if (a.lessZero())
    {
        setNeg(a);
    }
    else
    {
        *this = a;
    }

}

HK_INLINE void hkInt256::setFlipSign(const hkInt256& a, bool cmp)
{
    if (cmp)
    {
        setNeg(a);
    }
    else
    {
        *this = a;
    }
}

HK_INLINE void hkInt256::setSelect(bool cmp, const hkInt256& a, const hkInt256& b)
{
    if (cmp)
    {
        *this = a;
    }
    else
    {
        *this = b;
    }
}

HK_INLINE void hkInt256::setShiftLeft(const hkInt256& a, unsigned int shift)
{
    hkLargeIntImpl::shiftLeft(a.m_limbs, m_limbs, HK_COUNT_OF(m_limbs), shift);
}

HK_INLINE void hkInt256::setShiftRight(const hkInt256& a, unsigned int shift)
{
    hkLargeIntImpl::shiftRight(a.m_limbs, m_limbs, HK_COUNT_OF(m_limbs), shift);
}

HK_INLINE int hkInt256::countLeadingZeros() const
{
    return hkLargeIntImpl::countLeadingZeros(m_limbs, HK_COUNT_OF(m_limbs));
}

HK_INLINE bool hkInt256::equalZero() const
{
    return hkLargeIntImpl::equalZero(m_limbs, HK_COUNT_OF(m_limbs));
}

HK_INLINE bool hkInt256::lessZero() const
{
    return hkInt64(m_limbs[3]) < 0;
}

HK_INLINE bool hkInt256::greaterZero() const
{
    return !lessZero() && !equalZero();
}

HK_INLINE bool hkInt256::equal(const hkInt256& a) const
{
    return hkLargeIntImpl::equal(m_limbs, a.m_limbs, HK_COUNT_OF(m_limbs));
}

HK_INLINE void hkInt512::setAbs(const hkInt512& a)
{
    if (a.lessZero())
    {
        setNeg(a);
    }
    else
    {
        *this = a;
    }
}

HK_INLINE void hkInt512::setFlipSign(const hkInt512& a, bool cmp)
{
    if (cmp)
    {
        setNeg(a);
    }
    else
    {
        *this = a;
    }
}

HK_INLINE void hkInt512::setSelect(bool cmp, const hkInt512& a, const hkInt512& b)
{
    if (cmp)
    {
        *this = a;
    }
    else
    {
        *this = b;
    }
}

HK_INLINE void hkInt512::setShiftLeft(const hkInt512& a, unsigned int shift)
{
    hkLargeIntImpl::shiftLeft(a.m_limbs, m_limbs, HK_COUNT_OF(m_limbs), shift);
}

HK_INLINE void hkInt512::setShiftRight(const hkInt512& a, unsigned int shift)
{
    hkLargeIntImpl::shiftRight(a.m_limbs, m_limbs, HK_COUNT_OF(m_limbs), shift);
}

HK_INLINE int hkInt512::countLeadingZeros() const
{
    return hkLargeIntImpl::countLeadingZeros(m_limbs, HK_COUNT_OF(m_limbs));
}

HK_INLINE bool hkInt512::equalZero() const
{
    return hkLargeIntImpl::equalZero(m_limbs, HK_COUNT_OF(m_limbs));
}

HK_INLINE bool hkInt512::lessZero() const
{
    return hkInt64(m_limbs[7]) < 0;
}

HK_INLINE bool hkInt512::greaterZero() const
{
    return !lessZero() && !equalZero();
}

HK_INLINE bool hkInt512::equal(const hkInt512& a) const
{
    return hkLargeIntImpl::equal(m_limbs, a.m_limbs, HK_COUNT_OF(m_limbs));
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
