// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL !REFLECT
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#ifndef HK_MATH_LARGE_INT_TYPES_H
#   error Include Common/Base/Math/LargeInt/hkLargeIntTypes.h instead of this file!
#endif

class hkInt128;
class hkInt256;
class hkInt512;

/// 128 bit integer class in two's complement representation.
class HK_EXPORT_COMMON hkInt128
{
    friend class hkInt256;
    friend class hkInt512;

public:
    /// Sets this = 0
    HK_INLINE void setZero();

    /// Sets this = i
    HK_INLINE void setFromUint32(hkUint32 i);
    HK_INLINE void setFromUint64(hkUint64 i);
    HK_INLINE void setFromInt32(hkInt32 i);
    HK_INLINE void setFromInt64(hkInt64 i);

    /// Sets this = a & b
    HK_INLINE void setAnd(const hkInt128& a, const hkInt128& b);

    /// Sets this = a | b
    HK_INLINE void setOr(const hkInt128& a, const hkInt128& b);

    /// Sets this = a ^ b
    HK_INLINE void setXor(const hkInt128& a, const hkInt128& b);

    /// Sets this = ~a
    HK_INLINE void setNot(const hkInt128& a);

    /// Sets this = a << shift
    HK_INLINE void setShiftLeft(const hkInt128& a, unsigned int shift);

    /// Sets this = a >> shift
    HK_INLINE void setShiftRightSigned(const hkInt128& a, unsigned int shift);
    HK_INLINE void setShiftRight(const hkInt128& a, unsigned int shift);

    /// Sets this = this + a
    HK_INLINE void increment(hkUint64 a = 1);

    /// Sets this = this - a
    HK_INLINE void decrement(hkUint64 a = 1);

    /// Sets this = -a
    HK_INLINE void setNeg(const hkInt128& a);

    /// Set this = abs(a)
    HK_INLINE void setAbs(const hkInt128& a);

    /// Sets this = sign * v
    HK_INLINE void setFlipSign(const hkInt128& a, bool cmp);

    /// Sets this = cmp ? trueValue : falseValue
    HK_INLINE void setSelect(bool cmp, const hkInt128& trueValue, const hkInt128& falseValue);

    /// Sets this = a + b
    HK_INLINE void setAdd(const hkInt128& a, const hkInt128& b);

    /// Sets this = a - b
    HK_INLINE void setSub(const hkInt128& a, const hkInt128& b);

    /// Sets this = a * b. The operands are assumed to be small enough so they are not causing overflow
    HK_INLINE void setMul(hkInt64 a, hkInt64 b);
    HK_INLINE void setMul(const hkInt128& a, hkInt64 b);
    HK_INLINE void setMul(const hkInt128& a, hkUint64 b);
    HK_INLINE void setMul(const hkInt128& a, const hkInt128& b);
    HK_INLINE void setMul(hkUint64 a, hkUint64 b);

    /// Sets this = a / b, assuming a and b are positive
    void setUnsignedDiv(const hkInt128& a, const hkInt128& b);

    /// Returns a / b, assuming it fits in an int
    static int computeDiv32(const hkInt128& a, const hkInt128& b);

    /// Returns a / b and a % b
    static void computeUnsignedDivMod(const hkInt128& a, const hkInt128& b, hkInt128& divOut, hkInt128& modOut);

    /// Sets this = gcd(a, b)
    void setGreatestCommonDivisor(const hkInt128& a, const hkInt128& b);

    /// Compares the fractions (a / b) and (c / d), returning 1 if (a/b) > (c/d), 0 if they are equal, or -1 otherwise
    static int compareFractions(const hkInt128& a, const hkInt128& b, const hkInt128& c, const hkInt128& d);

    // Returns the number of leading zeros
    HK_INLINE int countLeadingZeros() const;

    // Returns the number of trailing zeros
    HK_INLINE int countTrailingZeros() const;

    /// Returns this == 0
    HK_INLINE bool equalZero() const;

    /// Returns this < 0
    HK_INLINE bool lessZero() const;

    /// Returns this > 0
    HK_INLINE bool greaterZero() const;

    /// Returns +1, 0 or -1 depending on the sign
    HK_INLINE int getSign() const;

    /// Returns this == a
    HK_INLINE bool equal(const hkInt128& a) const;

    /// Returns the N-th 64-bit word, where LSB = 0, MSB = 1
    HK_INLINE hkUint64 getDoubleWord(int N) const;

    /// Sets the N-th 64-bit word, where LSB = 0, MSB = 1
    HK_INLINE void setDoubleWord(int N, hkUint64 a);

private:
    hkUint64 m_limbs[2];
};

/// 256 bit integer class in two's complement representation.
class HK_EXPORT_COMMON hkInt256
{
    friend class hkInt512;

public:
    /// Sets this = 0
    void setZero();

    /// Sets this = i
    void setFromUint32(hkUint32 i);
    void setFromUint64(hkUint64 i);
    void setFromInt32(hkInt32 i);
    void setFromInt64(hkInt64 i);
    void setFromInt128(const hkInt128& i);

    /// Sets this = a & b
    void setAnd(const hkInt256& a, const hkInt256& b);

    /// Sets this = a | b
    void setOr(const hkInt256& a, const hkInt256& b);

    /// Sets this = a ^ b
    void setXor(const hkInt256& a, const hkInt256& b);

    /// Sets this = ~a
    void setNot(const hkInt256& a);

    /// Sets this = a << shift
    HK_INLINE void setShiftLeft(const hkInt256& a, unsigned int shift);

    /// Sets this = a >> shift
    HK_INLINE void setShiftRight(const hkInt256& a, unsigned int shift);

    /// Sets this = this + a
    void increment(hkUint64 a = 1);

    /// Sets this = this - a
    void decrement(hkUint64 a = 1);

    /// Sets this = -a
    void setNeg(const hkInt256& a);

    /// Set this = abs(a)
    HK_INLINE void setAbs(const hkInt256& a);

    /// Sets this = sign * v
    HK_INLINE void setFlipSign(const hkInt256& a, bool cmp);

    /// Sets this = cmp ? trueValue : falseValue
    HK_INLINE void setSelect(bool cmp, const hkInt256& trueValue, const hkInt256& falseValue);

    /// Sets this = a + b
    void setAdd(const hkInt256& a, const hkInt256& b);

    /// Sets this = a - b
    void setSub(const hkInt256& a, const hkInt256& b);

    /// Sets this = a * b. The operands are assumed to be small enough so they are not causing overflow
    void setMul(const hkInt128& a, const hkInt128& b);
    void setMul(const hkInt128& a, hkInt64 b);
    HK_INLINE void setMul(const hkInt256& a, const hkInt256& b);
    HK_INLINE void setMul(const hkInt256& a, hkUint64 b);

    /// Sets this += ab
    void addMul(const hkInt128& a, const hkInt128& b);

    /// Sets this -= ab
    void subMul(const hkInt128& a, const hkInt128& b);

    /// Returns a / b, assuming it fits in an int
    static int computeDiv32(const hkInt256& a, const hkInt256& b);

    /// Returns a / b and a % b
    static void computeUnsignedDivMod(const hkInt256& a, const hkInt256& b, hkInt256& divOut, hkInt256& modOut);

    /// Compares the fractions (a / b) and (c / d), returning 1 if (a/b) > (c/d), 0 if they are equal, or -1 otherwise
    static int compareFractions(const hkInt256& a, const hkInt256& b, const hkInt256& c, const hkInt256& d);

    // Returns the number of leading zeros
    HK_INLINE int countLeadingZeros() const;

    /// Returns this == 0
    HK_INLINE bool equalZero() const;

    /// Returns this < 0
    HK_INLINE bool lessZero() const;

    /// Returns this > 0
    HK_INLINE bool greaterZero() const;

    /// Returns +1, 0 or -1 depending on the sign
    HK_INLINE int getSign() const;

    /// Returns this == a
    HK_INLINE bool equal(const hkInt256& a) const;

    /// Returns the N-th 64-bit word, where LSB = 0, MSB = 3
    HK_INLINE hkUint64 getDoubleWord(int N) const;

    /// Sets the N-th 64-bit word, where LSB = 0, MSB = 3
    HK_INLINE void setDoubleWord(int N, hkUint64 a);

private:
    hkUint64 m_limbs[4];
};

/// 512 bit integer class in two's complement representation.
class HK_EXPORT_COMMON hkInt512
{
public:
    /// Sets this = 0
    void setZero();

    /// Sets this = i
    void setFromUint32(hkUint32 i);
    void setFromUint64(hkUint64 i);
    void setFromInt32(hkInt32 i);
    void setFromInt64(hkInt64 i);
    void setFromInt128(const hkInt128& i);
    void setFromInt256(const hkInt256& i);

    /// Sets this = a & b
    void setAnd(const hkInt512& a, const hkInt512& b);

    /// Sets this = a | b
    void setOr(const hkInt512& a, const hkInt512& b);

    /// Sets this = a ^ b
    void setXor(const hkInt512& a, const hkInt512& b);

    /// Sets this = ~a
    void setNot(const hkInt512& a);

    /// Sets this = a << shift
    HK_INLINE void setShiftLeft(const hkInt512& a, unsigned int shift);

    /// Sets this = a >> shift
    HK_INLINE void setShiftRight(const hkInt512& a, unsigned int shift);

    /// Sets this = this + a
    void increment(hkUint64 a = 1);

    /// Sets this = this - a
    void decrement(hkUint64 a = 1);

    /// Sets this = -a
    void setNeg(const hkInt512& a);

    /// Set this = abs(a)
    void setAbs(const hkInt512& a);

    /// Sets this = sign * v
    HK_INLINE void setFlipSign(const hkInt512& a, bool cmp);

    /// Sets this = cmp ? trueValue : falseValue
    HK_INLINE void setSelect(bool cmp, const hkInt512& trueValue, const hkInt512& falseValue);

    /// Sets this = a + b
    void setAdd(const hkInt512& a, const hkInt512& b);

    /// Sets this = a - b
    void setSub(const hkInt512& a, const hkInt512& b);

    /// Sets this = a * b. The operands are assumed to be small enough so they are not causing overflow
    void setMul(const hkInt256& a, const hkInt256& b);
    void setMul(const hkInt512& a, const hkInt512& b);
    void setMul(const hkInt512& a, hkUint64 b);

    /// Sets this += ab
    void addMul(const hkInt256& a, const hkInt256& b);

    /// Sets this -= ab
    void subMul(const hkInt256& a, const hkInt256& b);

    /// Returns a / b, assuming it fits in an int
    static int computeDiv32(const hkInt512& a, const hkInt512& b);

    /// Returns a / b and a % b
    static void computeUnsignedDivMod(const hkInt512& a, const hkInt512& b, hkInt512& divOut, hkInt512& modOut);

    /// Compares the fractions (a / b) and (c / d), returning 1 if (a/b) > (c/d), 0 if they are equal, or -1 otherwise
    static int compareFractions(const hkInt512& a, const hkInt512& b, const hkInt512& c, const hkInt512& d);

    // Returns the number of leading zeros
    HK_INLINE int countLeadingZeros() const;

    /// Returns this == 0
    HK_INLINE bool equalZero() const;

    /// Returns this < 0
    HK_INLINE bool lessZero() const;

    /// Returns this > 0
    HK_INLINE bool greaterZero() const;

    /// Returns +1, 0 or -1 depending on the sign
    HK_INLINE int getSign() const;

    /// Returns this == a
    HK_INLINE bool equal(const hkInt512& a) const;

    /// Returns the N-th 64-bit word, where LSB = 0, MSB = 7
    HK_INLINE hkUint64 getDoubleWord(int N) const;

    /// Sets the N-th 64-bit word, where LSB = 0, MSB = 7
    HK_INLINE void setDoubleWord(int N, hkUint64 a);

private:
    hkUint64 m_limbs[8];
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
