// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
namespace hkMath
{
#if !defined(HK_MATH_sqrt) && !defined(HK_MATH_sqrt_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL sqrt(const hkFloat32 r)
    {
        return ::sqrtf(r);
    }
#endif

#if !defined(HK_MATH_sqrt) && !defined(HK_MATH_sqrt_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL sqrt(const hkDouble64 r)
    {
        return ::sqrt(r);
    }
#endif
#if !defined(HK_MATH_fabs) && !defined(HK_MATH_fabs_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL fabs(const hkFloat32 r)
    {
        return ::fabsf(r);
    }
#endif
#if !defined(HK_MATH_fabs) && !defined(HK_MATH_fabs_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL fabs(const hkDouble64 r)
    {
        return ::fabs(r);
    }
#endif

#if !defined(HK_MATH_pow) && !defined(HK_MATH_pow_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL pow(const hkFloat32 r, const hkFloat32 s)
    {
        return ::powf(r, s);
    }
#endif
#if !defined(HK_MATH_pow) && !defined(HK_MATH_pow_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL pow(const hkDouble64 r, const hkDouble64 s)
    {
        return ::pow(r, s);
    }
#endif

#if !defined(HK_MATH_ceil) && !defined(HK_MATH_ceil_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL ceil(const hkFloat32 r)
    {
        return ::ceilf(r);
    }
#endif
#if !defined(HK_MATH_ceil) && !defined(HK_MATH_ceil_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL ceil(const hkDouble64 r)
    {
        return ::ceil(r);
    }
#endif

#if !defined(HK_MATH_sin) && !defined(HK_MATH_sin_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL sin(const hkFloat32 r)
    {
        return ::sinf(r);
    }
#endif
#if !defined(HK_MATH_sin) && !defined(HK_MATH_sin_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL sin(const hkDouble64 r)
    {
        return ::sin(r);
    }
#endif

#if !defined(HK_MATH_cos) && !defined(HK_MATH_cos_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL cos(const hkFloat32 r)
    {
        return ::cosf(r);
    }
#endif
#if !defined(HK_MATH_cos) && !defined(HK_MATH_cos_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL cos(const hkDouble64 r)
    {
        return ::cos(r);
    }
#endif

#if !defined(HK_MATH_acos) && !defined(HK_MATH_acos_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL acos(const hkFloat32 r)
    {
        // be generous about numbers slightly outside range
        HK_ASSERT_NO_MSG(0x41278654, hkMath::fabs(r) < 1.001f);
        if (hkMath::fabs(r) >= 1.0f)
        {
            hkFloat32 ret = (r > 0.0f) ? 0.0f : float(HK_FLOAT_PI);
            return ret;
        }
        return ::acosf(r);
    }
#endif
#if !defined(HK_MATH_acos) && !defined(HK_MATH_acos_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL acos(const hkDouble64 r)
    {
        // be generous about numbers slightly outside range
        HK_ASSERT_NO_MSG(0x41278654, hkMath::fabs(r) < 1.001);
        if (hkMath::fabs(r) >= 1.0)
        {
            hkDouble64 ret = (r > 0.0) ? 0.0 : hkDouble64(HK_DOUBLE_PI);
            return ret;
        }
        return ::acos(r);
    }
#endif

#if !defined(HK_MATH_asin) && !defined(HK_MATH_asin_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL asin(const hkFloat32 r)
    {
        // be generous about numbers outside range
        HK_ASSERT_NO_MSG(0x286a6f5f, hkMath::fabs(r) < 1.001f);
        if (hkMath::fabs(r) >= 1.0f)
        {
            hkFloat32 ret = (r > 0.0f) ? 0.5f * float(HK_FLOAT_PI) : -0.5f * float(HK_FLOAT_PI);
            return ret;
        }
        return ::asinf(r);
    }
#endif
#if !defined(HK_MATH_asin) && !defined(HK_MATH_asin_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL asin(const hkDouble64 r)
    {
        // be generous about numbers outside range
        HK_ASSERT_NO_MSG(0x286a6f5f, hkMath::fabs(r) < 1.001);
        if (hkMath::fabs(r) >= 1.0)
        {
            hkDouble64 ret = (r > 0.0) ? 0.5 * hkDouble64(HK_DOUBLE_PI) : -0.5 * hkDouble64(HK_DOUBLE_PI);
            return ret;
        }
        return ::asin(r);
    }
#endif

#if !defined(HK_MATH_log) && !defined(HK_MATH_log_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL log(const hkFloat32 r)
    {
        return ::logf(r);
    }
#endif
#if !defined(HK_MATH_log) && !defined(HK_MATH_log_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL log(const hkDouble64 r)
    {
        return ::log(r);
    }
#endif

#if !defined(HK_MATH_atan2) && !defined(HK_MATH_atan2_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL atan2(const hkFloat32 y, const hkFloat32 x)
    {
        return ::atan2f(y, x);
    }
#endif
#if !defined(HK_MATH_atan2) && !defined(HK_MATH_atan2_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL atan2(const hkDouble64 y, const hkDouble64 x)
    {
        return ::atan2(y, x);
    }
#endif
#if !defined(HK_MATH_tan) && !defined(HK_MATH_tan_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL tan(const hkFloat32 r)
    {
        return ::tan(r);
    }
#endif
#if !defined(HK_MATH_tan) && !defined(HK_MATH_tan_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL tan(const hkDouble64 r)
    {
        return ::tan(r);
    }
#endif
#if !defined(HK_MATH_floor) && !defined(HK_MATH_floor_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL floor(const hkFloat32 r)
    {
        return ::floorf(r);
    }
#endif
#if !defined(HK_MATH_floor) && !defined(HK_MATH_floor_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL floor(const hkDouble64 r)
    {
        return ::floor(r);
    }
#endif
#if !defined(HK_MATH_round) && !defined(HK_MATH_round_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL round(const hkFloat32 r)
    {
        return (r > 0.0f) ? floorf(r + 0.5f) : ::ceilf(r - 0.5f);
    }
#endif
#if !defined(HK_MATH_round) && !defined(HK_MATH_round_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL round(const hkDouble64 r)
    {
        return (r > 0.0f) ? floor(r + 0.5f) : ::ceil(r - 0.5f);
    }
#endif

#if !defined(HK_MATH_fmod) && !defined(HK_MATH_fmod_f)
    HK_EXPORT_COMMON hkFloat32 HK_CALL fmod(const hkFloat32 n, const hkFloat32 d)
    {
        return ::fmodf(n, d);
    }
#endif
#if !defined(HK_MATH_fmod) && !defined(HK_MATH_fmod_d)
    HK_EXPORT_COMMON hkDouble64 HK_CALL fmod(const hkDouble64 n, const hkDouble64 d)
    {
        return ::fmod(n, d);
    }
#endif

    HK_EXPORT_COMMON hkInt32 HK_CALL doubleToFloat( hkInt64 dHex )
    {
        hkInt64 dSign = (dHex >> 63) & 1LL; // 1-bit
        hkInt64 dExponent = ((dHex >> 52) & ((1LL << 11) - 1LL)) - 1023LL; // 11-bits with 1023 offset
        hkInt64 dMantissa = dHex & ((1LL << 52) - 1LL); // 52-bits
        HK_ASSERT( 0x387AF33A, (dSign & ~0x1) == 0, "Sign Overflow!" );
        HK_ASSERT( 0x387AF33B, (dExponent > -1024LL) && (dExponent <= 1024LL), "Exponent Overflow!" );
        HK_ASSERT( 0x387AF33C, (dMantissa & ~0xFFFFFFFFFFFFF) == 0, "Mantissa Overflow!" );

        hkInt64 fSign = dSign; // 1-bit
        hkInt64 fExponent; // 8-bits with 127 offset
        hkInt64 fMantissa; // 23-bits
        if( dExponent == 1024 )
        {
            // Inf double
            fExponent = 0xFF;
            fMantissa = dMantissa >> 29;
            // Tailing one
            fMantissa |= ((dMantissa > 0) << 22);
        }
        else if( dExponent == -1023 )
        {
            // Denormal double
            fExponent = 0;
            fMantissa = 0;
        }
        else
        {
            if( dExponent >= 128 )
            {
                // Inf float
                fExponent = 255LL;
                fMantissa = 0LL;
            }
            else if( dExponent <= -127 )
            {
                // Denormal float
                fExponent = 0LL;
                if( dExponent >= -149 ) // 1 bit to 23 bits of mantissa
                {
                    // Rounding
                    const hkInt64 carryOver = dMantissa >> (-98 - dExponent); // N + 1 bits
                    fMantissa = ((carryOver + 1) >> 1); // round to N bits
                    // Leading 1
                    const hkInt64 leadingOne = 1LL << (149 + dExponent);
                    fMantissa += leadingOne;
                }
                else if( dExponent == -150 ) // 0 bit, set to 1 if anything
                {
                    fMantissa = (dMantissa > 0);
                }
                else // Too far, always 0
                {
                    fMantissa = 0LL;
                }
            }
            else
            {
                // Normal conversion
                fMantissa = dMantissa >> 29;
                fExponent = dExponent + 127LL;
                // Rounding
                if( fMantissa == 0x7FFFFFLL )
                {
                    if( (dMantissa >> 28) & 0x1LL )
                    {
                        fExponent++;
                        fMantissa = 0;
                    }
                }
            }
        }

        HK_ASSERT( 0x387AF34A, (fSign & ~0x1) == 0, "Sign Overflow!" );
        HK_ASSERT( 0x387AF34B, (fExponent & ~0xFF) == 0, "Exponent Overflow!" );
        HK_ASSERT( 0x387AF34C, (fMantissa & ~0x7FFFFF) == 0, "Mantissa Overflow!" );
        hkInt64 fi = fMantissa ^ (fExponent << 23) ^ (fSign << 31);
        HK_ASSERT( 0x387AF34D, (fi & ~0xFFFFFFFF) == 0, "Overflow!" );
        return *reinterpret_cast<hkInt32*>(&fi);
    }
} // namespace hkMath

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
