// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/System/Io/OStream/hkOStream.h>
#include <Common/Base/Container/String/hkStringBuf.h>

#define HK_MP_PRINT_RATIONAL_AS_DOUBLE

/// Multi-precision unsigned integer.
/// Adapted from Matt McCutchen's 'C++ Big Integer Library' @ https://mattmccutchen.net/bigint/
struct HK_EXPORT_COMMON hkMpUint
{
    HK_DECLARE_CLASS( hkMpUint, New, Reflect );

    //
    typedef hkUint32    Atom;
    typedef hkUint16    HalfAtom;

    //
    enum
    {
        BITS_PER_ATOM       =   sizeof(Atom) * 8,
        BITS_PER_HALF_ATOM  =   BITS_PER_ATOM >> 1,
    };

    //
    typedef hkArray<Atom> AtomArray;

    //
    HK_INLINE           hkMpUint() {}

    //
    HK_INLINE           hkMpUint(const hkMpUint& other) { operator=(other); }

    //
    HK_INLINE int       getSize() const { return m_atoms.getSize(); }

    //
    HK_INLINE int       getSizeInBits() const { return getSize() * BITS_PER_ATOM; }

    //
    HK_INLINE int       msb() const;

    //
    HK_INLINE bool      isZero() const { return getSize() == 0; }

    //
    HK_INLINE void      setZero() { m_atoms.clear(); }

    //
    HK_INLINE hkMpUint& operator=( const hkMpUint& other ) { if ( this != &other ) { m_atoms.clear(); m_atoms.append( other.m_atoms ); } return *this; }

    //
    HK_INLINE Atom&     operator[](int index) { return m_atoms[index]; }

    //
    HK_INLINE Atom      operator[](int index) const { return m_atoms[index]; }

    //
    HK_INLINE void      stripLeadingZeros() { while(m_atoms.getSize() && 0 == m_atoms.back()) m_atoms.popBack(); }

    //
    HK_INLINE Atom      getAtom(int index, int shift) const;

    //
    HK_EXPORT_COMMON friend inline void             HK_CALL set(hkMpUint& bi, hkUint64 value);

    //
    HK_EXPORT_COMMON friend int                     HK_CALL compare(const hkMpUint& bi0, const hkMpUint& bi1);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL pow(hkMpUint& bi, unsigned exp);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL add(const hkMpUint& bi0, const hkMpUint& bi1, hkMpUint& biOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL sub(const hkMpUint& bi0, const hkMpUint& bi1, hkMpUint& biOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL mul(const hkMpUint& bi0, const hkMpUint& bi1, hkMpUint& biOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL div(const hkMpUint& bi0, const hkMpUint& bi1, hkMpUint& qOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL mod(const hkMpUint& bi0, const hkMpUint& bi1, hkMpUint& rOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL div(const hkMpUint& bi0, const hkMpUint& bi1, hkMpUint& qOut, hkMpUint& rOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL gcd(const hkMpUint& x, const hkMpUint& y, hkMpUint& output);

    // x << count if count > 0 else x >> -count
    HK_EXPORT_COMMON friend void                    HK_CALL shift(hkMpUint& x, int count);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL swap( hkMpUint& bi0, hkMpUint& bi1 ) { bi0.m_atoms.swap( bi1.m_atoms ); }

    //
    HK_EXPORT_COMMON friend void                    HK_CALL toString(const hkMpUint& bi, unsigned base, hkStringBuf& stringOut);

    //
    HK_EXPORT_COMMON friend hkOstream&              HK_CALL operator<<(hkOstream& stream, const hkMpUint& bi) { hkStringBuf s; toString(bi,10,s); stream << s; return stream; }

    private:

    AtomArray   m_atoms;
};

///
/// Multi-precision rational.
///
struct HK_EXPORT_COMMON hkMpRational
{
    HK_DECLARE_CLASS( hkMpRational, New, Reflect );
    HK_RECORD_ATTR( hk::ToString( &hkMpRational::toLogString ) );

    //
    HK_INLINE           hkMpRational(hkInt32 num = 0, hkInt32 den = 1) { setNumeratorAndDemominator(num, den); }

    //
    HK_INLINE           hkMpRational(hkUint32 num, hkUint32 den = 1) { setNumeratorAndDemominator(num, den); }

    //
    HK_INLINE           hkMpRational(hkInt64 num, hkInt64 den = 1) { setNumeratorAndDemominator(num, den); }

    //
    HK_INLINE           hkMpRational(hkUint64 num, hkUint64 den = 1) { setNumeratorAndDemominator(num, den); }

    //
    HK_INLINE           hkMpRational(hkReal value) { set(*this, value); }

    //
    HK_INLINE           hkMpRational(hkSimdRealParameter value) { set(*this, value); }

    //
    HK_INLINE bool      isZero() const { return m_num.isZero(); }

    //
    HK_INLINE bool      isLessZero() const { return m_signed; }

    //
    HK_INLINE bool      isLessEqualZero() const { return isLessZero() || isZero(); }

    //
    HK_INLINE bool      isGreaterZero() const { return !isZero() && !m_signed; }

    //
    HK_INLINE bool      isGreaterEqualZero() const { return !m_signed; }

    //
    HK_INLINE void      setZero() { m_num.setZero(); m_den.setZero(); m_signed = false; }

    //
    HK_INLINE bool      isInfinity() const { return m_den.isZero(); }

    //
    HK_INLINE int       getSign() const { return isZero() ? 0 : (m_signed ? -1 : +1); }

    //
    HK_INLINE void      cleanZero() { if(isZero()) m_signed = false; }

    //
    HK_INLINE int       compareTo(const hkMpRational& rat) const { return compare(*this, rat); }

    //
    inline hkMpRational operator+(const hkMpRational& rat) const { hkMpRational r; add(*this, rat, r); return r; }

    //
    inline hkMpRational operator-(const hkMpRational& rat) const { hkMpRational r; sub(*this, rat, r); return r; }

    //
    inline hkMpRational operator*(const hkMpRational& rat) const { hkMpRational r; mul(*this, rat, r); return r; }

    //
    inline hkMpRational operator/(const hkMpRational& rat) const { hkMpRational r; div(*this, rat, r); return r; }

    //
    inline hkMpRational operator^(int power) const { hkMpRational r = *this; pow(r,power); return r; }

    //
    HK_EXPORT_COMMON friend int                     compare(const hkMpRational& rat0, const hkMpRational& rat1);

    //
    HK_EXPORT_COMMON friend int                     compare(const hkMpRational& rat, const hkMpUint& bi);

    //
    HK_EXPORT_COMMON friend int                     HK_CALL compare(const hkMpUint& bi, const hkMpRational& rat) { return compare(rat, bi) * -1; }

    //
    HK_EXPORT_COMMON friend void                    HK_CALL abs(hkMpRational& rat) { rat.m_signed = false; }

    //
    HK_EXPORT_COMMON friend void                    HK_CALL round(hkMpRational& rat) { hkMpUint intVal; get(rat, intVal); rat.m_num = intVal; set(rat.m_den,1); rat.cleanZero(); }

    //
    HK_EXPORT_COMMON friend void                    HK_CALL inv(hkMpRational& rat);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL pow(hkMpRational& rat, int exp);

    //
    HK_EXPORT_COMMON friend inline void             HK_CALL neg(hkMpRational& rat) { if(!rat.isZero()) rat.m_signed = !rat.m_signed; }

    //
    HK_EXPORT_COMMON friend void                    HK_CALL compact(hkMpRational& rat);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL sum(const hkMpRational& rat0, bool sgn0, const hkMpRational& rat1, bool sgn1, hkMpRational& ratOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL add(const hkMpRational& rat0, const hkMpRational& rat1, hkMpRational& ratOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL sub(const hkMpRational& rat0, const hkMpRational& rat1, hkMpRational& ratOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL mul(const hkMpRational& rat0, const hkMpRational& rat1, hkMpRational& ratOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL mul(const hkMpRational& rat, const hkMpUint& bi, hkMpRational& ratOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL div(const hkMpRational& rat0, const hkMpRational& rat1, hkMpRational& ratOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL set(hkMpRational& ratOut, float value) { setIEEE<23>(ratOut,*reinterpret_cast<hkUint32*>(&value)); }

    //
    HK_EXPORT_COMMON friend void                    HK_CALL set(hkMpRational& ratOut, double value) { setIEEE<52>(ratOut,*reinterpret_cast<hkUint64*>(&value)); }

    //
    HK_EXPORT_COMMON friend void                    HK_CALL set(hkMpRational& ratOut, hkSimdRealParameter value) { set(ratOut, value.getReal()); }

    //
    template <typename TN, typename TD>
    inline void                     setNumeratorAndDemominator(TN numerator, TD denominator);
    template <typename TN, typename TD>
    inline void                     setNumeratorAndDemominator(TN numerator, TD denominator, hkTrait::TrueType TNsigned, hkTrait::TrueType TDsigned );
    template <typename TN, typename TD>
    inline void                     setNumeratorAndDemominator(TN numerator, TD denominator, hkTrait::FalseType TNsigned, hkTrait::FalseType TDsigned );

    //
    HK_EXPORT_COMMON friend void                    HK_CALL get(const hkMpRational& rat, hkMpUint& valueOut);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL get(const hkMpRational& rat, float& valueOut) { hkMpRational::getIEEE<23>(rat, *reinterpret_cast<hkUint32*>(&valueOut)); }

    //
    HK_EXPORT_COMMON friend void                    HK_CALL get(const hkMpRational& rat, double& valueOut) { hkMpRational::getIEEE<52>(rat, *reinterpret_cast<hkUint64*>(&valueOut)); }

    //
    HK_EXPORT_COMMON friend void                    HK_CALL get(const hkMpRational& rat, hkSimdReal& valueOut) { hkReal v; get(rat,v); valueOut.setFromFloat(v); }

    //
    HK_EXPORT_COMMON friend void                    HK_CALL swap( hkMpRational& r0, hkMpRational& r1 ) { swap( r0.m_num, r1.m_num ); swap( r0.m_den, r1.m_den ); hkMath::swap( r0.m_signed, r1.m_signed ); }

    //
    template <int MANTISSA_BITS, typename T>
    inline static void  setIEEE(hkMpRational& ratOut, T valueAsUint);

    //
    template <int MANTISSA_BITS, typename T>
    inline static void  getIEEE(const hkMpRational& rat, T& valueAsUint);

    //
    HK_EXPORT_COMMON friend void                    HK_CALL toString(const hkMpRational& rat, unsigned base, hkStringBuf& stringOut);

    //
    HK_EXPORT_COMMON friend hkOstream&              HK_CALL operator<<(hkOstream& stream, const hkMpRational& rat)
    {
        #ifdef HK_MP_PRINT_RATIONAL_AS_DOUBLE
        double value = 0; get(rat, value);
        stream << value;
        #else
        if(m_signed) stream << '-';
        stream << m_num;
        stream << '/';
        stream << m_den;
        #endif
        return stream;
    }

    // Custom formatter for hkLog
    static void HK_CALL toLogString( const hkReflect::Var& var, hkStringBuf& sb, const hkStringView& extra )
    {
        if ( const hkMpRational* rat = var.dynCast<const hkMpRational>() )
        {
            toString( *rat, unsigned(10), sb );
        }
    }

    private:

    hkMpUint    m_num;
    hkMpUint    m_den;
    bool        m_signed;
};

///
/// Multi-precision vector.
///
template <int N>
struct hkMpVector
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_MATH, hkMpVector);

    enum { DIMENSIONS = N };

    //
    HK_INLINE                       hkMpVector() {}

    //
    HK_INLINE                       hkMpVector(hkVector4Parameter value) { set(*this, value); }

    //
    HK_INLINE                       hkMpVector(hkSimdRealParameter value) { set(*this, value); }

    //
    HK_INLINE                       hkMpVector(hkReal value) { set(*this, value); }

    //
    HK_INLINE hkMpRational&         operator[](int i) { HK_ASSERT(0x45FE2134, i >= 0 && i < N, "Index out of range."); return m_components[i]; }

    //
    HK_INLINE const hkMpRational&   operator[](int i) const { HK_ASSERT(0x45FE2135, i >= 0 && i < N, "Index out of range."); return m_components[i]; }

    //
    friend void                     compact(hkMpVector<N>& v)
    {
        for(int i=0; i<N; ++i) compact(v[i]);
    }

    //
    friend void                     neg(hkMpVector<N>& v)
    {
        for(int i=0; i<N; ++i) neg(v[i]);
    }

    //
    friend void                     inv(hkMpVector<N>& v)
    {
        for(int i=0; i<N; ++i) inv(v[i]);
    }

    //
    friend void                     add(const hkMpVector<N>& v0, const hkMpVector<N>& v1, hkMpVector<N>& vOut)
    {
        for(int i=0; i<N; ++i) add(v0[i], v1[i], vOut[i]);
    }

    //
    friend void                     sub(const hkMpVector<N>& v0, const hkMpVector<N>& v1, hkMpVector<N>& vOut)
    {
        for(int i=0; i<N; ++i) sub(v0[i], v1[i], vOut[i]);
    }

    //
    friend void                     mul(const hkMpVector<N>& v0, const hkMpVector<N>& v1, hkMpVector<N>& vOut)
    {
        for(int i=0; i<N; ++i) mul(v0[i], v1[i], vOut[i]);
    }

    //
    friend void                     mul(const hkMpVector<N>& v0, const hkMpRational& r1, hkMpVector<N>& vOut)
    {
        for(int i=0; i<N; ++i) mul(v0[i], r1, vOut[i]);
    }

    //
    friend void                     div(const hkMpVector<N>& v0, const hkMpVector<N>& v1, hkMpVector<N>& vOut)
    {
        for(int i=0; i<N; ++i) div(v0[i], v1[i], vOut[i]);
    }

    //
    friend void                     div(const hkMpVector<N>& v0, const hkMpRational& r1, hkMpVector<N>& vOut)
    {
        for(int i=0; i<N; ++i) div(v0[i], r1, vOut[i]);
    }

    //
    friend void                     mins(const hkMpVector<N>& v0, const hkMpVector<N>& v1, hkMpVector<N>& vOut)
    {
        for(int i=0; i<N; ++i) vOut[i] = compare(v0[i], v1[i]) < 0 ? v0[i] : v1[i];
    }

    //
    friend void                     maxs(const hkMpVector<N>& v0, const hkMpVector<N>& v1, hkMpVector<N>& vOut)
    {
        for(int i=0; i<N; ++i) vOut[i] = compare(v0[i], v1[i]) > 0 ? v0[i] : v1[i];
    }

    //
    friend void                     horizontalMin(const hkMpVector<N>& v, hkMpRational& rOut)
    {
        hkMpRational current = v[0];
        for(int i = 1; i < N; ++i) if(compare(current, v[i]) < 0) current = v[i];
        rOut = current;
    }

    //
    friend void                     horizontalMax(const hkMpVector<N>& v, hkMpRational& rOut)
    {
        hkMpRational current = v[0];
        for(int i = 1; i < N; ++i) if(compare(current, v[i]) > 0) current = v[i];
        rOut = current;
    }

    //
    friend void                     horizontalSum(const hkMpVector<N>& v, hkMpRational& rOut)
    {
        hkMpRational current = v[0];
        for(int i = 1; i < N; ++i) add(current, v[i], current);
        rOut = current;
    }

    //
    friend void                     cross(const hkMpVector<N>& v0, const hkMpVector<N>& v1, hkMpVector<N>& vOut)
    {
        HK_COMPILE_TIME_ASSERT(N >= 3);
        hkMpRational    p0,p1;

        mul(v0[1], v1[2], p0);
        mul(v0[2], v1[1], p1);
        hkMpRational    x; sub(p0, p1, x);

        mul(v0[2], v1[0], p0);
        mul(v0[0], v1[2], p1);
        hkMpRational    y; sub(p0, p1, y);

        mul(v0[0], v1[1], p0);
        mul(v0[1], v1[0], p1);
        hkMpRational    z; sub(p0, p1, z);

        vOut[0] = x;
        vOut[1] = y;
        vOut[2] = z;

        for(int i=3; i<N; ++i) vOut[i].setZero();
    }

    //
    friend void                     dot(const hkMpVector<N>& v0, const hkMpVector<N>& v1, hkMpRational& rOut)
    {
        hkMpVector<N> p; mul(v0, v1, p);
        horizontalSum(p, rOut);
    }

    //
    friend void                     dotAdd(const hkMpVector<N>& v0, const hkMpVector<N>& v1, const hkMpRational& offset, hkMpRational& rOut)
    {
        hkMpVector<N>   p; mul(v0, v1, p);
        hkMpRational    s; horizontalSum(p, s);
        add(s, offset, rOut);
    }

    //
    friend void                     lengthSquared(const hkMpVector<N>& v, hkMpRational& rOut)
    {
        dot(v,v,rOut);
    }

    //
    friend void                     distanceSquared(const hkMpVector<N>& v0, const hkMpVector<N>& v1, hkMpRational& rOut)
    {
        hkMpVector<N> delta; sub(v1,v0,delta);
        lengthSquared(delta, rOut);
    }

    //
    friend void                     interpolate(const hkMpVector<N>& v0, const hkMpVector<N>& v1, const hkMpRational& fraction, hkMpVector<N>& vOut)
    {
        hkMpVector<N>   dv;
        sub(v1,v0,dv);
        mul(dv, fraction, dv);
        add(v0, dv, vOut);
    }

    //
    friend void                     set(hkMpVector<N>& v, hkVector4Parameter value)
    {
        for(int i=0, n = N > 4 ? 4 : N; i<n; ++i)
        {
            set(v[i], value(i));
        }
    }

    //
    friend void                     set(hkMpVector<N>& v, hkSimdRealParameter value)
    {
        set(v, value.getReal());
    }

    //
    friend void                     set(hkMpVector<N>& v, hkReal value)
    {
        hkMpRational rv; set(rv,value);
        for(int i=0; i<N; ++i) v[i] = rv;
    }

    //
    friend void                     get(const hkMpVector<N>& v, hkVector4& value)
    {
        value.setZero();
        for(int i=0, n = N > 4 ? 4 : N; i<n; ++i)
        {
            get(v[i], value(i));
        }
    }

    //
    friend void                     toString(const hkMpVector<N>& v, unsigned base, hkStringBuf& stringOut)
    {
        hkStringBuf t;
        stringOut.append("[");
        for(int i=0; i<N; ++i)
        {
            if(i) stringOut.append(",");
            toString(v[i], base, t);
            stringOut.append(t.cString());
        }
        stringOut.append("]");
    }

    //
    friend hkOstream&               operator<<(hkOstream& stream, const hkMpVector<N>& v)
    {
        stream << '[';
        for(int i=0; i<N; i++)
        {
            if(i) stream << ',';
            stream << v[i];
        }
        stream << ']';
        return stream;
    }

    private:

    hkMpRational    m_components[N];
};

// Typedefs.

typedef hkMpVector<2>   hkMpVector2;
typedef hkMpVector<3>   hkMpVector3;
typedef hkMpVector<4>   hkMpVector4;

// Solve Ax=b using Gauss-Seidel method. Note: No pivoting is performed so the diagonal of A must be non-zero.
// Axb_TYPE must implement the following:
// int                  dims() const;
// const hkMpRational&  A(int row, int column) const;
// const hkMpRational&  b(int column) const;
// hkMpRational&        x(int column);
template <typename Axb_TYPE>
inline void hkMpLinearSolveGaussSeidel(Axb_TYPE& Axb, int iterations);

// Enable unit tests only on win32/64
#ifdef HK_PLATFORM_WIN32
bool    hkMpUnitTests();
#endif

#include <Common/Base/Math/ExtendedMath/hkMpMath.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
