// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Math/ExtendedMath/hkFpMathDetail.h>
#include <Common/Base/Types/hkTrait.h>
#include <Common/Base/Math/Vector/hkVector2f.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>

namespace hkFpMath
{
    /// A constant 1, which supports only multiplication and is used by rational numbers with zero denominator bits.
    class ConstantOne
    {
    };

    template<int BITS> class FpInt;

    template<int BITS_>
    class FpUint
    {
    public:
        HK_DECLARE_CLASS(FpUint, New, Pod);

        enum { BITS = BITS_ };
        typedef typename Detail::UintStorageType<BITS>::Type StorageType;
        StorageType m_storage;

        HK_INLINE FpUint() {}

        template<int SRC_BITS>
        HK_INLINE FpUint(FpUint<SRC_BITS> src)
        {
            HK_COMPILE_TIME_ASSERT(BITS >= SRC_BITS);
            Detail::widenUint(src.m_storage, m_storage);
        }

        HK_INLINE explicit FpUint(ConstantOne)
        {
            setZero();
            Detail::inPlaceSetWordUint(m_storage, 0, 1);
        }

        HK_INLINE explicit FpUint(hkInt32 val)
        {
            setZero();
            // TODO range check
            Detail::inPlaceSetWordUint(m_storage, 0, val);
        }

        FpUint& operator=(ConstantOne)
        {
            setZero();
            Detail::inPlaceSetWordUint(m_storage, 0, 1);
            return *this;
        }


        FpUint& operator=(hkUint32 val)
        {
            setZero();
            Detail::inPlaceSetWordUint(m_storage, 0, val);
            return *this;
        }

        HK_INLINE void setZero()
        {
            Detail::setZeroUint(m_storage);
        }

        HK_INLINE bool equalZero() const
        {
            return Detail::equalZero(m_storage);
        }

        template<int RES_BITS>
        HK_INLINE bool canNarrow() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS <= BITS);
            return Detail::checkWordRangeUint<RES_BITS>(m_storage);
        }

        template<int RES_BITS>
        HK_INLINE FpUint<RES_BITS> narrow() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS <= BITS);
            FpUint<RES_BITS> r;
            Detail::narrowUint(m_storage, r.m_storage);
            return r;
        }

        template<int RES_BITS>
        HK_INLINE FpUint<RES_BITS> widen() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS >= BITS);

            FpUint<RES_BITS> r;
            Detail::widenUint(m_storage, r.m_storage);
            return r;
        }

        HK_INLINE hkUint32 toUint32() const
        {
            HK_COMPILE_TIME_ASSERT(BITS <= 32);
            return hkUint32(m_storage);
        }

        int countTrailingZeroes() const
        {
            return Detail::countTrailingZerosUint(m_storage);
        }

        HK_INLINE FpInt<BITS + 1> toSigned() const
        {
            FpInt<BITS+1> r;
            Detail::convertUintInt(m_storage, r.m_storage);
            return r;
        }

        HK_INLINE hkFloat32 toFloat32() const
        {
            return Detail::convertUintFloat(m_storage);
        }

        template<int SHIFT_BITS>
        HK_INLINE FpUint<BITS-SHIFT_BITS> shr() const
        {
            FpUint<BITS> r = *this;
            r >>= SHIFT_BITS;
            return r.template narrow<BITS-SHIFT_BITS>();
        }

        hkUint32 hash() const
        {
            return Detail::hash(m_storage);
        }
    };

    template<int B1, int B2, int B3, int B4>
    void divMod(FpUint<B1> const& a, FpUint<B2> const& b, FpUint<B3> & qOut, FpUint<B4> & rOut)
    {
        FpUint<B3> q;
        FpUint<B4> r;
        Detail::divModUint(a.m_storage, b.m_storage, q.m_storage, r.m_storage);
        qOut = q;
        rOut = r;
    }

    template<int B1, int B2, int B3>
    void divExact(FpUint<B1> const& a, FpUint<B2> const& b, FpUint<B3> & qOut)
    {
        FpUint<B3> q;
        FpUint<B2> rIgnored;
        Detail::divModUint(a.m_storage, b.m_storage, q.m_storage, rIgnored.m_storage);

#ifdef HKFPMATH_CHECK_WORD_RANGE
        HK_ASSERT(0x5578f332, rIgnored.equalZero());
#endif
        qOut = q;
    }


    template<int B1, int B2>
    FpUint<B1> gcd_aux(FpUint<B1> const& a, FpUint<B2> const& b, hkTrait::TrueType)
    {
        return gcd_aux(b, a, hkTrait::FalseType());
    }

    template<int B1, int B2>
    FpUint<B2> gcd_aux(FpUint<B1> const& aIn, FpUint<B2> const& bIn, hkTrait::FalseType)
    {
        FpUint<B1> a = aIn;
        FpUint<B2> b = bIn;

        int d;
        {
            int az = a.countTrailingZeroes();
            a >>= az;

            int bz = b.countTrailingZeroes();
            b >>= bz;

            d = hkMath::min2(az, bz);
        }

        while(a != b)
        {
            if(a > b)
            {
                a = (a-b).asUintNarrowing();
                int az = a.countTrailingZeroes();
                a >>= az;
            }
            else
            {
                b = (b-a).template asUintNarrowingTo<B2>();
                int bz = b.countTrailingZeroes();
                b >>= bz;
            }
        }

        b <<= d;
        return b;
    }

    /// Determine the GCD of two nonzero integers.
    template<int B1, int B2>
    FpUint<((B1<B2)?B1:B2)> gcd(FpUint<B1> const& a, FpUint<B2> const& b)
    {
        typedef hkTrait::TraitBool<(B1<B2)> SwapArgs;
        return gcd_aux(a, b, SwapArgs());
    }

    template<int B>
    int ceilLog2(FpUint<B> const& x)
    {
        return Detail::ceilLog2Uint(x.m_storage);
    }

    template<int BITS_>
    class FpInt
    {
    public:
        HK_DECLARE_CLASS(FpInt, New, Pod);

        enum { BITS = BITS_ };
        typedef typename Detail::UintStorageType<BITS>::Type StorageType;
        StorageType m_storage;

        HK_INLINE FpInt() {}

        template<int SRC_BITS>
        HK_INLINE FpInt(FpInt<SRC_BITS> src)
        {
            HK_COMPILE_TIME_ASSERT(BITS >= SRC_BITS);
            Detail::widenInt(src.m_storage, m_storage);
        }

        template<int SRC_BITS>
        HK_INLINE FpInt(FpUint<SRC_BITS> src)
        {
            HK_COMPILE_TIME_ASSERT(BITS >= SRC_BITS+1);
            Detail::convertUintInt(src.m_storage, m_storage);
        }

        HK_INLINE explicit FpInt(ConstantOne)
        {
            setZero();
            Detail::inPlaceSetWordUint(m_storage, 0, 1);
        }

        HK_INLINE explicit FpInt(hkInt32 val)
        {
            Detail::convertInt32Int(val, m_storage);
        }

        HK_INLINE void setZero()
        {
            Detail::setZeroUint(m_storage);
        }

        HK_INLINE void setMinVal()
        {
            Detail::setMinValInt<BITS>(m_storage);
        }

        HK_INLINE void setMaxVal()
        {
            Detail::setMaxValInt<BITS>(m_storage);
        }

        void HK_INLINE operator=(hkInt32 val)
        {
            Detail::convertInt32Int(val, m_storage);
        }

        template<int RES_BITS>
        HK_INLINE bool canNarrow() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS <= BITS);
            return Detail::checkWordRangeInt<RES_BITS>(m_storage);
        }

        template<int RES_BITS>
        HK_INLINE FpInt<RES_BITS> narrow() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS <= BITS);

#ifdef HKFPMATH_CHECK_WORD_RANGE
            HK_ASSERT(0x94751ad8, Detail::checkWordRangeInt<RES_BITS>(m_storage));
#endif
            FpInt<RES_BITS> r;
            Detail::narrowInt(m_storage, r.m_storage);

            return r;
        }

        template<int RES_BITS>
        HK_INLINE FpInt<RES_BITS> narrowSaturate() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS <= BITS);

            FpInt<RES_BITS> r;
            if(Detail::checkWordRangeInt<RES_BITS>(m_storage))
            {
                Detail::narrowInt(m_storage, r.m_storage);
            }
            else
            {
                Detail::inPlaceSaturateInt<RES_BITS>(r.m_storage, lessZero());
            }
            return r;
        }

        template<int RES_BITS>
        HK_INLINE FpInt<RES_BITS> widen() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS >= BITS);
            FpInt<RES_BITS> r;
            Detail::widenInt(m_storage, r.m_storage);
            return r;
        }

        HK_INLINE bool lessZero() const
        {
            return Detail::lessZeroInt(m_storage);
        }

        HK_INLINE bool equalZero() const
        {
            return Detail::equalZero(m_storage);
        }

        HK_INLINE bool greaterZero() const
        {
            return (!equalZero()) & (!lessZero());
        }

        HK_INLINE bool greaterEqualZero() const
        {
            return !lessZero();
        }

        HK_INLINE bool lessEqualZero() const
        {
            return !greaterZero();
        }

        HK_INLINE FpUint<BITS-1> abs() const
        {
            FpUint<BITS-1> r;
            Detail::absIntToUint(m_storage, r.m_storage);
            return r;
        }

        HK_INLINE FpUint<BITS-1> asUintNarrowing() const
        {
#ifdef HKFPMATH_CHECK_WORD_RANGE
            HK_ASSERT(0x75bee783, !lessZero());
#endif
            FpUint<BITS-1> r;
            Detail::narrowUint(m_storage, r.m_storage);
            return r;
        }

        template<int RES_BITS>
        HK_INLINE FpUint<RES_BITS> asUintNarrowingTo() const
        {
#ifdef HKFPMATH_CHECK_WORD_RANGE
            HK_ASSERT(0x75bee783, !lessZero());
#endif
            FpUint<RES_BITS> r;
            Detail::narrowUint(m_storage, r.m_storage);
            return r;
        }

        HK_INLINE int sgn() const
        {
            return Detail::sgnInt(m_storage);
        }

        HK_INLINE hkInt32 toInt32() const
        {
            HK_COMPILE_TIME_ASSERT(BITS <= 32);
            return hkInt32(Detail::signExtend(m_storage));
        }

        HK_INLINE hkFloat32 toFloat32() const
        {
            return Detail::convertIntFloat(m_storage);
        }

        void setFromFloat32(hkFloat32 x)
        {
            Detail::convertFloatInt(x, m_storage);
            Detail::checkWordRangeInt<BITS>(m_storage);
        }

        template<int EXPONENT_BIAS>
        void setFromBiasedFloat32(hkFloat32 x)
        {
            Detail::convertBiasedFloatInt<EXPONENT_BIAS>(x, m_storage);
            Detail::checkWordRangeInt<BITS>(m_storage);
        }

        hkUint32 hash() const
        {
            return Detail::hash(m_storage);
        }
    };

    /// given a and b, produce q and r such that q*b + r = a, with the sign of r matching the sign of a.
    template<int B1, int B2, int B3, int B4>
    void divMod(FpInt<B1> const& a, FpInt<B2> const& b, FpInt<B3> & qOut, FpInt<B4> & rOut)
    {
        bool aNeg = a.lessZero();
        bool bNeg = b.lessZero();

        FpUint<B1-1> aAbs = a.abs();
        FpUint<B2-1> bAbs = b.abs();

        Detail::divModUint(aAbs.m_storage, bAbs.m_storage, qOut.m_storage, rOut.m_storage);

        if(aNeg != bNeg)
        {
            qOut = -qOut;
        }
        if(aNeg)
        {
            rOut = -rOut;
        }
    }

    template<int B1, int B2, int B3>
    void divExact(FpInt<B1> const& a, FpInt<B2> const& b, FpInt<B3> & qOut)
    {
        bool aNeg = a.lessZero();
        bool bNeg = b.lessZero();

        FpUint<B1-1> aAbs = a.abs();
        FpUint<B2-1> bAbs = b.abs();
        FpUint<B2-1> rAbsIgnored;

        Detail::divModUint(aAbs.m_storage, bAbs.m_storage, qOut.m_storage, rAbsIgnored.m_storage);

#ifdef HKFPMATH_CHECK_WORD_RANGE
        HK_ASSERT(0x5578f332, rAbsIgnored.equalZero());
#endif

        if(aNeg != bNeg)
        {
            qOut = -qOut;
        }
    }

    /// Find the quotient and remainder of a/b. The sign of q and r will match the sign of a. If the remainder is
    /// discarded, this is effectively round-towards-zero.
    template<int B1, int B2, int B3, int B4>
    void divMod(FpInt<B1> const& a, FpUint<B2> const& b, FpInt<B3> & qOut, FpInt<B4> & rOut)
    {
        bool aNeg = a.lessZero();

        FpUint<B1-1> aAbs = a.abs();

        Detail::divModUint(aAbs.m_storage, b.m_storage, qOut.m_storage, rOut.m_storage);

        if(aNeg)
        {
            qOut = -qOut;
            rOut = -rOut;
        }
    }

    template<int B1, int B2, int B3>
    void divExact(FpInt<B1> const& a, FpUint<B2> const& b, FpInt<B3> & qOut)
    {
        bool aNeg = a.lessZero();

        FpUint<B1-1> aAbs = a.abs();
        FpUint<B2-1> rAbsIgnored;

        FpUint<B1-1> qAbs;

        Detail::divModUint(aAbs.m_storage, b.m_storage, qAbs.m_storage, rAbsIgnored.m_storage);

#ifdef HKFPMATH_CHECK_WORD_RANGE
        HK_ASSERT(0x5578f332, rAbsIgnored.equalZero());
#endif
        qOut = aNeg ? -qAbs : FpInt<B3>(qAbs);
    }

    /// Divide with round-to-infinity or round-to-negative-infinity, based on roundingDir (which must be -1 or 1).
    template<int B1, int B2, int B3>
    void divRounding(FpInt<B1> const& a, FpUint<B2> const& b, int roundingDir, FpInt<B3> & qOut)
    {
        FpInt<B3> q;
        FpInt<B2> r;

        divMod(a, b, q, r);

        // Check whether the remainder is both nonzero, and pointing (along with the quotient) in the rounding
        // direction; i.e. a negative result and rounding down, or a positive result and rounding up. Those are the
        // situations where the desired rounding behavior does not match the behavior of just throwing the remainder
        // away (i.e. round towards zero).
        //
        // This also happens to make roundingDir==0 do round-towards-zero, but not efficiently.
        if(r.sgn() == roundingDir)
        {
            q = (q + FpInt<8>(roundingDir)).template narrow<B3>();
        }

        qOut = q;
    }

    /// Divide with round-to-infinity or round-to-negative-infinity, based on roundingDir (which must be -1 or 1).
    template<int B1, int B2, int B3>
    void divRounding(FpUint<B1> const& a, FpUint<B2> const& b, int roundingDir, FpUint<B3> & qOut)
    {
        FpUint<B3> q;
        FpUint<B2> r;
        Detail::divModUint(a.m_storage, b.m_storage, q.m_storage, r.m_storage);

        if(!r.equalZero())
        {
            q = (q + FpInt<8>(roundingDir)).template narrow<B3>();
        }

        qOut = q;
    }

    /// Divide two signed integers by their unsigned GCD, such that they become coprime.
    /// For the purposes of this function, gcd(a,0)==a for a!=0. If all operands are zero,
    /// they are left as zero.
    /// Signs are unchanged.
    template<int B1, int B2>
    void reduce(FpInt<B1> & a, FpInt<B2> & b)
    {
        if(a.equalZero() | b.equalZero())
        {
            a = FpInt<B1>(a.sgn());
            b = FpInt<B2>(b.sgn());
        }
        else
        {
            FpUint<(B1<B2) ? B1-1 : B2-1> abGcd = gcd(a.abs(), b.abs());
            divExact(a, abGcd, a);
            divExact(b, abGcd, b);
        }
    }

    /// Divide three signed integers by their unsigned GCD, such that they become set-wise coprime.
    /// For the purposes of this function, gcd(a,0)==a for a!=0. If all operands are zero,
    /// they are left as zero.
    /// Signs are unchanged.
    template<int B1, int B2, int B3>
    void reduce(FpInt<B1> & a, FpInt<B2> & b, FpInt<B3> & c)
    {
        if(a.equalZero())
        {
            reduce(b, c);
        }
        else if(b.equalZero())
        {
            reduce(a, c);
        }
        else if(c.equalZero())
        {
            reduce(a, b);
        }
        else
        {
            FpUint<(B1<B2) ? B1-1 : B2-1> abGcd = gcd(a.abs(), b.abs());
            FpUint<((B1<B2) ? ((B1<B3) ? B1 : B3) : ((B2<B3) ? B2 : B3)) - 1> abcGcd = gcd(abGcd, c.abs());

            divExact(a, abcGcd, a);
            divExact(b, abcGcd, b);
            divExact(c, abcGcd, c);
        }
    }

    template<int B1, int B2, int B3, int B4>
    void reduce(FpInt<B1> & a, FpInt<B2> & b, FpInt<B3> & c, FpInt<B4> & d)
    {
        if(a.equalZero())
        {
            reduce(b, c, d);
        }
        else if(b.equalZero())
        {
            reduce(a, c, d);
        }
        else if(c.equalZero())
        {
            reduce(a, b, d);
        }
        else if(d.equalZero())
        {
            reduce(a, b, c);
        }
        else
        {
            auto abGcd = gcd(a.abs(), b.abs());
            auto abcGcd = gcd(abGcd, c.abs());
            auto abcdGcd = gcd(abcGcd, d.abs());

            divExact(a, abcdGcd, a);
            divExact(b, abcdGcd, b);
            divExact(c, abcdGcd, c);
            divExact(d, abcdGcd, d);
        }
    }

    template<int B1, int B2>
    void reduce(FpInt<B1> & a, FpUint<B2> & b)
    {
        if(a.equalZero() | b.equalZero())
        {
            a = FpInt<B1>(a.sgn());
            b = FpUint<B2>(!b.equalZero());
        }
        else
        {
            FpUint<(B1-1<B2) ? B1-1 : B2> abGcd = gcd(a.abs(), b);
            divExact(a, abGcd, a);
            divExact(b, abGcd, b);
        }
    }

    template<int B1, int B2, int B3>
    void reduce(FpInt<B1> & a, FpInt<B2> & b, FpUint<B3> & c)
    {
        if(a.equalZero())
        {
            reduce(b, c);
        }
        else if(b.equalZero())
        {
            reduce(a, c);
        }
        else if(c.equalZero())
        {
            reduce(a, b);
        }
        else
        {
            FpUint<(B1-1<B2-1) ? B1-1 : B2-1> abGcd = gcd(a.abs(), b.abs());
            FpUint<((B1-1<B2-1) ? ((B1-1<B3) ? B1-1 : B3) : ((B2-1<B3) ? B2-1 : B3))> abcGcd = gcd(abGcd, c);

            divExact(a, abcGcd, a);
            divExact(b, abcGcd, b);
            divExact(c, abcGcd, c);
        }
    }

    template<int B>
    int ceilLog2(FpInt<B> const& x)
    {
        return Detail::ceilLog2Int(x.m_storage);
    }

    template<int B1, int B2>
    HK_INLINE FpUint<B1 + B2> operator*(FpUint<B1> a, FpUint<B2> b)
    {
        FpUint<Detail::StorageBits<B1>::VALUE + Detail::StorageBits<B2>::VALUE> r;
        Detail::mulUintUint(a.m_storage, b.m_storage, r.m_storage);
        return r.template narrow<B1 + B2>();
    }

    template<int B1, int B2>
    HK_INLINE FpUint<((B1>B2) ? B1 : B2) + 1> operator+(FpUint<B1> a, FpUint<B2> b)
    {
        FpUint<((B1>B2) ? B1 : B2) + 1 > r;
        Detail::addUintUint(a.m_storage, b.m_storage, r.m_storage);
        return r;
    }

    template<int B1>
    HK_INLINE FpUint<B1> const& operator>>=(FpUint<B1> & a, int bits)
    {
        Detail::inPlaceShrUint(a.m_storage, bits);
        return a;
    }

    template<int B1>
    HK_INLINE FpUint<B1> const& operator<<=(FpUint<B1> & a, int bits)
    {
        Detail::inPlaceShlUint(a.m_storage, bits);
        return a;
    }

    template<int B1, int B2>
    HK_INLINE bool operator==(FpUint<B1> a, FpUint<B2> b)
    {
        return Detail::equalUintUint(a.m_storage, b.m_storage);
    }

    template<int B1>
    HK_INLINE bool operator==(FpUint<B1> a, hkUint32 b)
    {
        return Detail::equalUintUint(a.m_storage, b);
    }

    template<int B1, int B2>
    HK_INLINE bool operator!=(FpUint<B1> a, FpUint<B2> b)
    {
        return !(a==b);
    }

    template<int B1>
    HK_INLINE bool operator!=(FpUint<B1> a, hkUint32 b)
    {
        return !(a==b);
    }

    template<int B1, int B2>
    HK_INLINE bool operator<(FpUint<B1> a, FpUint<B2> b)
    {
        return Detail::lessUintUint(a.m_storage, b.m_storage);
    }

    template<int B1, int B2>
    HK_INLINE bool operator>(FpUint<B1> a, FpUint<B2> b)
    {
        return b<a;
    }

    template<int B1, int B2>
    HK_INLINE bool operator>=(FpUint<B1> a, FpUint<B2> b)
    {
        return !(a<b);
    }


    template<int B1, int B2>
    HK_INLINE bool operator<=(FpUint<B1> a, FpUint<B2> b)
    {
        return !(a>b);
    }

    /// Compare the two operands. Returns 1 if a>b, -1 if a<b, and 0 if a==b.
    template<int B1, int B2>
    HK_INLINE int cmp(FpUint<B1> const& a, FpUint<B2> const& b)
    {
        return (a - b).sgn();
    }

    template<int BITS>
    HK_INLINE FpInt<BITS> operator-(FpInt<BITS> a)
    {
        FpInt<BITS> res;
        Detail::negateInt(a.m_storage, res.m_storage);
        return res;
    }

    template<int BITS>
    HK_INLINE FpInt<BITS+1> operator-(FpUint<BITS> a)
    {
        FpInt<BITS+1> aSigned = a;
        FpInt<BITS+1> res;
        Detail::negateInt(aSigned.m_storage, res.m_storage);
        return res;
    }

    template<int B1, int B2>
    HK_INLINE FpInt<B1 + B2> operator*(FpInt<B1> const& a, FpInt<B2> const& b)
    {
        FpInt<Detail::StorageBits<B1>::VALUE + Detail::StorageBits<B2>::VALUE> r;
        Detail::mulIntInt(a.m_storage, b.m_storage, r.m_storage);
        return r.template narrow<B1+B2>();
    }

    template<int B1, int B2>
    HK_INLINE FpInt<((B1>B2) ? B1 : B2) + 1> operator+(FpInt<B1> const& a, FpInt<B2> const& b)
    {
        FpInt<((B1>B2) ? B1 : B2) + 1> r;
        Detail::addIntInt(a.m_storage, b.m_storage, r.m_storage);
        return r;
    }

    template<int B1, int B2>
    HK_INLINE FpInt<((B1>B2) ? B1 : B2) + 1> operator-(FpInt<B1> const& a, FpInt<B2> const& b)
    {
        FpInt<((B1>B2) ? B1 : B2) + 1> r;
        Detail::subIntInt(a.m_storage, b.m_storage, r.m_storage);
        return r;
    }

    template<int B1, int B2>
    HK_INLINE bool operator==(FpInt<B1> a, FpInt<B2> b)
    {
        return Detail::equalIntInt(a.m_storage, b.m_storage);
    }

    template<int B1, int B2>
    HK_INLINE bool operator!=(FpInt<B1> a, FpInt<B2> b)
    {
        return !(a==b);
    }

    template<int B1, int B2>
    HK_INLINE bool operator<(FpInt<B1> a, FpInt<B2> b)
    {
        return Detail::lessIntInt(a.m_storage, b.m_storage);
    }

    template<int B1, int B2>
    HK_INLINE bool operator>(FpInt<B1> a, FpInt<B2> b)
    {
        return b<a;
    }

    template<int B1, int B2>
    HK_INLINE bool operator<=(FpInt<B1> a, FpInt<B2> b)
    {
        return !(a>b);
    }

    template<int B1, int B2>
    HK_INLINE bool operator>=(FpInt<B1> a, FpInt<B2> b)
    {
        return !(a<b);
    }

    // Mixed uint/int operations
    template<int B1, int B2>
    HK_INLINE FpInt<B1 + B2> operator*(FpUint<B1> const& a, FpInt<B2> const& b)
    {
        FpInt<Detail::StorageBits<B1>::VALUE + Detail::StorageBits<B2>::VALUE> r;
        Detail::mulUintInt(a.m_storage, b.m_storage, r.m_storage);
        return r.template narrow<B1 + B2>();
    }

    template<int B1, int B2>
    HK_INLINE FpInt<B1 + B2> operator*(FpInt<B1> const& a, FpUint<B2> const& b)
    {
        return b*a;
    }

    template<int B1, int B2>
    HK_INLINE FpInt<(((B1+1)>B2) ? (B1+1) : B2) + 1> operator+(FpUint<B1> a, FpInt<B2> b)
    {
        return a.toSigned() + b;
    }

    template<int B1, int B2>
    HK_INLINE FpInt<((B1>(B2+1)) ? B1 : (B2+1)) + 1> operator+(FpInt<B1> a, FpUint<B2> b)
    {
        return a + b.toSigned();
    }

    template<int B1, int B2>
    HK_INLINE FpInt<((B1>B2) ? B1 : B2) + 1> operator-(FpUint<B1> a, FpUint<B2> b)
    {
        return (a.toSigned() - b.toSigned()).template narrow<((B1>B2) ? B1 : B2) + 1>();
    }

    template<int B1, int B2>
    HK_INLINE FpInt<((B1>B2) ? B1 : B2) + 1> operator-(FpInt<B1> a, FpUint<B2> b)
    {
        return (a - b.toSigned()).template narrow<((B1>B2) ? B1 : B2) + 1>();
    }

    template<int B1, int B2>
    HK_INLINE bool operator<(FpInt<B1> a, FpUint<B2> b)
    {
        return a.lessZero() || a.template asUintNarrowingTo<B1-1>() < b;
    }

    template<int B1, int B2>
    HK_INLINE bool operator<(FpUint<B1> a, FpInt<B2> b)
    {
        return !b.lessZero() && a < b.template asUintNarrowingTo<B2-1>();
    }

    template<int B1, int B2>
    HK_INLINE bool operator>=(FpUint<B1> a, FpInt<B2> b)
    {
        return !(a<b);
    }

    template<int B1, int B2>
    HK_INLINE bool operator>=(FpInt<B1> a, FpUint<B2> b)
    {
        return !(a<b);
    }

    /// Compare the two operands. Returns 1 if a>b, -1 if a<b, and 0 if a==b.
    template<int B1, int B2>
    HK_INLINE int cmp(FpInt<B1> const& a, FpInt<B2> const& b)
    {
        return (a - b).sgn();
    }

    namespace
    {
        template<int N, bool IS_NEG>
        struct Const_Aux {};

        template<int N>
        struct Const_Aux<N, false>
        {
            typedef FpUint<hkMath::Log2<N>::ANSWER + 1> Type;
            static HK_INLINE Type get()
            {
                return Type(N);
            }
        };

        template<>
        struct Const_Aux<0, false>
        {
            typedef FpUint<1> Type;
            static HK_INLINE Type get()
            {
                return FpUint<1>(0);
            }
        };

        template<int N>
        struct Const_Aux<N, true>
        {
            typedef FpInt<hkMath::Log2<1-N>::ANSWER + 1> Type;
            static HK_INLINE Type get()
            {
                return Type(-N);
            }
        };
    }

    template<int N>
    HK_INLINE typename Const_Aux<N, (N<0)>::Type constant()
    {
        return Const_Aux<N, (N<0)>::get();
    }

    inline ConstantOne operator*(ConstantOne, ConstantOne)
    {
        return ConstantOne();
    }

    template<int N>
    FpUint<N> operator*(ConstantOne, FpUint<N> b)
    {
        return b;
    }

    template<int N>
    FpUint<N> operator*(FpUint<N> a, ConstantOne)
    {
        return a;
    }

    template<int N>
    FpInt<N> operator*(ConstantOne, FpInt<N> b)
    {
        return b;
    }

    template<int N>
    FpInt<N> operator*(FpInt<N> a, ConstantOne)
    {
        return a;
    }

    template<int NUMERATOR_BITS_, int DENOMINATOR_BITS_>
    class FpRational
    {
    public:
        HK_DECLARE_CLASS(FpRational, New, Pod);

        enum { NUMERATOR_BITS = NUMERATOR_BITS_ };
        enum { DENOMINATOR_BITS = DENOMINATOR_BITS_ };

        typedef FpInt<NUMERATOR_BITS> NumeratorType;
        typedef FpUint<DENOMINATOR_BITS> DenominatorType;

        FpRational() {}

        template<typename T>
        HK_INLINE explicit FpRational(T n) : m_numerator(n), m_denominator(1) { }

        template<int N, int D>
        FpRational(FpRational<N, D> const& other) : m_numerator(other.m_numerator), m_denominator(other.m_denominator) {}

        FpRational(FpInt<NUMERATOR_BITS> n, FpUint<DENOMINATOR_BITS> d) : m_numerator(n), m_denominator(d) { }

        FpRational(FpInt<NUMERATOR_BITS> n, FpInt<DENOMINATOR_BITS+1> d)
        {
            m_denominator = d.abs();
            m_numerator = d.lessZero() ? -n : n;
        }

        int sgn() const { return m_numerator.sgn(); }

        bool equalZero() const { return m_numerator.equalZero(); }

        bool lessZero() const { return m_numerator.lessZero(); }

        bool greaterZero() const { return m_numerator.greaterZero(); }

        bool greaterEqualZero() const { return m_numerator.greaterEqualZero(); }

        FpRational<NUMERATOR_BITS, DENOMINATOR_BITS> abs() const
        {
            return FpRational<NUMERATOR_BITS, DENOMINATOR_BITS>(m_numerator.abs(), m_denominator);
        }

        HK_INLINE void setZero()
        {
            m_numerator.setZero();
            m_denominator = FpUint<DENOMINATOR_BITS>(1);
        }

        HK_INLINE void setOne()
        {
            m_numerator = 1;
            m_denominator = 1;
        }

        void setMinVal()
        {
            m_numerator.setMinVal();
            m_denominator = 1;
        }

        void setMaxVal()
        {
            m_numerator.setMaxVal();
            m_denominator = 1;
        }

        HK_INLINE hkFloat32 toFloat32() const
        {
            return Detail::convertIntUintRatioFloat(m_numerator.m_storage, m_denominator.m_storage);
        }

        FpRational<NUMERATOR_BITS, DENOMINATOR_BITS> reduced() const
        {
            if(m_numerator.equalZero())
            {
                return FpRational(FpInt<NUMERATOR_BITS>(0));
            }

            bool neg = m_numerator.lessZero();
            FpUint<NUMERATOR_BITS-1> numeratorAbs = m_numerator.abs();

            FpUint<((NUMERATOR_BITS-1)<DENOMINATOR_BITS)?(NUMERATOR_BITS-1):DENOMINATOR_BITS> ndGcd = gcd(numeratorAbs, m_denominator);
            FpUint<(NUMERATOR_BITS-1)> reducedNumeratorAbs;
            divExact(numeratorAbs, ndGcd, reducedNumeratorAbs);

            FpInt<NUMERATOR_BITS> reducedNumerator = reducedNumeratorAbs;
            if(neg)
            {
                reducedNumerator = -reducedNumerator;
            }

            FpUint<DENOMINATOR_BITS> reducedDenominator;
            divExact(m_denominator, ndGcd, reducedDenominator);

            return FpRational<NUMERATOR_BITS, DENOMINATOR_BITS>(reducedNumerator, reducedDenominator);
        }

        /// Returns true if the rational number is in the range [0,1].
        bool inClosedUnit() const
        {
            return !m_numerator.lessZero() && m_numerator.asUintNarrowing() <= m_denominator;
        }

        hkUint32 hash() const
        {
            using namespace hkHash;
            return combineHashValues(hkHashValue(m_numerator), hkHashValue(m_denominator));
        }

        FpInt<NUMERATOR_BITS> m_numerator;
        FpUint<DENOMINATOR_BITS> m_denominator;
    };

    template<int NUMERATOR_BITS_>
    class FpRational<NUMERATOR_BITS_, 0>
    {
    public:
        HK_DECLARE_CLASS(FpRational, New, Pod);

        enum { NUMERATOR_BITS = NUMERATOR_BITS_ };

        FpRational() {}
        template<int A_BITS>
        HK_INLINE explicit FpRational(FpInt<A_BITS> n, ConstantOne d = ConstantOne())
            : m_numerator(n)
        {
            HK_COMPILE_TIME_ASSERT(A_BITS <= NUMERATOR_BITS);
        }

        template<int N, int D>
        FpRational(FpRational<N, D> const& other) : m_numerator(other.m_numerator), m_denominator(other.m_denominator) {}

        FpInt<NUMERATOR_BITS> m_numerator;
        ConstantOne m_denominator;
    };

    template<int N, int D>
    HK_INLINE FpRational<N, D> operator-(FpRational<N, D> a)
    {
        return FpRational<N, D>(-a.m_numerator, a.m_denominator);
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE FpRational<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> operator+(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        FpInt<(N1+D2>N2+D1 ? N1+D2 : N2+D1)+1> num = a.m_numerator*b.m_denominator + b.m_numerator*a.m_denominator;
        FpUint<D1 + D2> denom = a.m_denominator * b.m_denominator;

        return FpRational<(N1+D2>N2+D1 ? N1+D2 : N2+D1)+1, D1+D2>(num, denom);
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE FpRational<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> operator-(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        return FpRational<(N1+D2>N2+D1 ? N1+D2 : N2+D1)+1, D1+D2>(
            a.m_numerator*b.m_denominator - b.m_numerator*a.m_denominator,
            a.m_denominator * b.m_denominator);
    }

    template<int N1, int N2, int D2>
    HK_INLINE FpRational<(N1 + D2>N2 ? N1 + D2 : N2) + 1, D2> operator+(FpInt<N1> a, FpRational<N2, D2> b)
    {
        return FpRational<(N1 + D2>N2 ? N1 + D2 : N2) + 1, D2>(a*b.m_denominator + b.m_numerator, b.m_denominator);
    }

    template<int N1, int D1, int N2>
    HK_INLINE FpRational<(N1 > N2 + D1 ? N1 : N2 + D1) + 1, D1> operator+(FpRational<N1, D1> a, FpInt<N2> b)
    {
        return FpRational<(N1 > N2 + D1 ? N1 : N2 + D1) + 1, D1>(a.m_numerator + b*a.m_denominator, a.m_denominator);
    }

    template<int N1, int N2, int D2>
    HK_INLINE FpRational<N1 + N2, D2> operator*(FpInt<N1> a, FpRational<N2, D2> b)
    {
        return FpRational<N1+N2, D2>(a*b.m_numerator, b.m_denominator);
    }

    template<int N1, int D1, int N2>
    HK_INLINE FpRational<N1 + N2, D1> operator*(FpRational<N1, D1> a, FpInt<N2> b)
    {
        return FpRational<N1+N2, D1>(a.m_numerator*b, a.m_denominator);
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE FpRational<N1 + N2, D1 + D2> operator*(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        return FpRational<N1+N2, D1+D2>(a.m_numerator*b.m_numerator, a.m_denominator*b.m_denominator);
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE FpRational<N1 + D2, D1 + N2> operator/(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        return FpRational<N1+D2, D1+N2>(a.m_numerator*b.m_denominator, a.m_denominator*b.m_numerator);
    }

    template<int N, int D>
    HK_INLINE FpRational<N, D-1> operator/(FpInt<N> a, FpInt<D> b)
    {
        return FpRational<N, D-1>(a, b);
    }

    template<int N, int D>
    HK_INLINE FpRational<N, D> operator/(FpInt<N> a, FpUint<D> b)
    {
        FpRational<N, D> res;
        res.m_numerator = a;
        res.m_denominator = b;
        return res;
    }

    template<int N, int D>
    HK_INLINE FpRational<N, D> operator/(FpUint<N> a, FpInt<D> b)
    {
        return FpRational<N+1, D-1>(b.lessZero() ? -a : FpInt<N+1>(a), b.abs());
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE bool operator<(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        return a.m_numerator*b.m_denominator < b.m_numerator*a.m_denominator;
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE bool operator>(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        return b<a;
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE bool operator<=(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        return !(b<a);
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE bool operator>=(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        return !(a<b);
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE bool operator==(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        return a.m_numerator*b.m_denominator == b.m_numerator*a.m_denominator;
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE bool operator!=(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        return !(a==b);
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator<(FpRational<N1, D1> a, FpUint<N2> b)
    {
        return a.m_numerator < b*a.m_denominator;
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator<(FpUint<N1> a, FpRational<N2, D2> b)
    {
        return a*b.m_denominator < b.m_numerator;
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator>(FpRational<N1, D1> a, FpUint<N2> b)
    {
        return b<a;
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator>(FpUint<N1> a, FpRational<N2, D2> b)
    {
        return b<a;
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator<=(FpRational<N1, D1> a, FpUint<N2> b)
    {
        return !(b<a);
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator<=(FpUint<N1> a, FpRational<N2, D2> b)
    {
        return !(b<a);
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator>=(FpRational<N1, D1> a, FpUint<N2> b)
    {
        return !(a<b);
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator>=(FpUint<N1> a, FpRational<N2, D2> b)
    {
        return !(a<b);
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator==(FpRational<N1, D1> a, FpUint<N2> b)
    {
        return a.m_numerator == b*a.m_denominator;
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator==(FpUint<N1> a, FpRational<N2, D2> b)
    {
        return a*b.m_denominator == b.m_numerator;
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator!=(FpRational<N1, D1> a, FpUint<N2> b)
    {
        return !(a==b);
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator!=(FpUint<N1> a, FpRational<N2, D2> b)
    {
        return !(a==b);
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator<(FpRational<N1, D1> a, FpInt<N2> b)
    {
        return a.m_numerator < b*a.m_denominator;
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator<(FpInt<N1> a, FpRational<N2, D2> b)
    {
        return a*b.m_denominator < b.m_numerator;
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator>(FpRational<N1, D1> a, FpInt<N2> b)
    {
        return b<a;
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator>(FpInt<N1> a, FpRational<N2, D2> b)
    {
        return b<a;
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator<=(FpRational<N1, D1> a, FpInt<N2> b)
    {
        return !(b<a);
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator<=(FpInt<N1> a, FpRational<N2, D2> b)
    {
        return !(b<a);
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator>=(FpRational<N1, D1> a, FpInt<N2> b)
    {
        return !(a<b);
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator>=(FpInt<N1> a, FpRational<N2, D2> b)
    {
        return !(a<b);
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator==(FpRational<N1, D1> a, FpInt<N2> b)
    {
        return a.m_numerator == b*a.m_denominator;
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator==(FpInt<N1> a, FpRational<N2, D2> b)
    {
        return a*b.m_denominator == b.m_numerator;
    }

    template<int N1, int D1, int N2>
    HK_INLINE bool operator!=(FpRational<N1, D1> a, FpInt<N2> b)
    {
        return !(a==b);
    }

    template<int N1, int N2, int D2>
    HK_INLINE bool operator!=(FpInt<N1> a, FpRational<N2, D2> b)
    {
        return !(a==b);
    }

    /// Perform a comparison of a and b, assuming that abs(a-b) < (abs(a)+abs(b))*FLT_EPS*2^LOG_DELTA.
    template<int LOG_DELTA, int N1, int D1, int N2, int D2>
    HK_INLINE int cmpClose(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        int anBits = ceilLog2(a.m_numerator);
        int adBits = ceilLog2(a.m_denominator);
        int bnBits = ceilLog2(b.m_numerator);
        int bdBits = ceilLog2(b.m_denominator);

        int prodBits = hkMath::max2(anBits+bdBits, adBits+bnBits);

        int errorBits = prodBits + LOG_DELTA - 22;
        if(errorBits < Detail::MULTI_WORD_UINT_BITS)
        {
            Detail::MultiWordUintType anLow = Detail::getWordUint(a.m_numerator).m_storage;
            Detail::MultiWordUintType adLow = Detail::getWordUint(a.m_denominator).m_storage;
            Detail::MultiWordUintType bnLow = Detail::getWordUint(b.m_numerator).m_storage;
            Detail::MultiWordUintType bdLow = Detail::getWordUint(b.m_denominator).m_storage;

            Detail::MultiWordUintType prodLow = anLow*bdLow - adLow*bnLow;

            return Detail::sgnInt(prodLow);
        }
        else
        {
            return cmp(a, b);
        }
    }

    template<int N1, int D1, int N2, int D2>
    HK_INLINE int cmp(FpRational<N1, D1> a, FpRational<N2, D2> b)
    {
        FpInt<N1+D2> lhs = a.m_numerator*b.m_denominator;
        FpInt<N2+D1> rhs = b.m_numerator*a.m_denominator;

        return cmp(lhs, rhs);
    }


    /// Solve the system ax+by=e, cx+dy=f
    ///
    /// This uses an expanded and factored form of Cramer's Rule, operating directly on numerators and denominators
    template<int N1, int D1, int N2, int D2, int N3, int D3>
    bool solve2x2(
        hkFpMath::FpRational<N1, D1> a,
        hkFpMath::FpRational<N1, D1> b,
        hkFpMath::FpRational<N1, D1> c,
        hkFpMath::FpRational<N1, D1> d,
        hkFpMath::FpRational<N2, D2> e,
        hkFpMath::FpRational<N2, D2> f,
        hkFpMath::FpRational<N3, D3>& x,
        hkFpMath::FpRational<N3, D3>& y)
    {
        HK_COMPILE_TIME_ASSERT(N3 >= N1+N2+3*D1+D2+1);
        HK_COMPILE_TIME_ASSERT(D3 >= 2*N1+2*D1+2*D2+1);

        hkFpMath::FpUint<2*D1> bd_cd = b.m_denominator*c.m_denominator;
        hkFpMath::FpUint<2*D1> ad_dd = a.m_denominator*d.m_denominator;
        hkFpMath::FpInt<N1+D1> dn_fd = d.m_numerator*f.m_denominator;
        hkFpMath::FpInt<N2+D1> bn_ed = b.m_numerator*e.m_denominator;
        hkFpMath::FpInt<N1+2*D1+D2> bd_cd_dn_fd = bd_cd*dn_fd;
        hkFpMath::FpInt<N1+2*D1+D2> ad_bn_dd_ed = ad_dd*bn_ed;
        hkFpMath::FpInt<N1+D2> an_ed = a.m_numerator*e.m_denominator;
        hkFpMath::FpInt<N1+D2> cn_fd = c.m_numerator*f.m_denominator;
        hkFpMath::FpInt<2*N1+2*D1+2*D2+1> denom = an_ed*bd_cd_dn_fd - ad_bn_dd_ed*cn_fd;

        if(denom.equalZero())
        {
            return false;
        }

        hkFpMath::FpInt<N2+D1> ad_en = a.m_denominator*e.m_numerator;
        hkFpMath::FpInt<N2+D1> cd_fn = c.m_denominator*f.m_numerator;
        hkFpMath::FpInt<N1+N2+3*D1+D2+1> xNum = ad_en*bd_cd_dn_fd - ad_bn_dd_ed*cd_fn;

        hkFpMath::FpInt<N2+D1> dd_fn = d.m_denominator*f.m_numerator;
        hkFpMath::FpInt<N2+D1> bd_en = b.m_denominator*e.m_numerator;
        hkFpMath::FpInt<N1+N2+3*D1+D2+1> yNum = an_ed*bd_cd*dd_fn - ad_dd*bd_en*cn_fd;

        if(denom.lessZero())
        {
            x.m_numerator = -xNum;
            y.m_numerator = -yNum;
        }
        else
        {
            x.m_numerator = xNum;
            y.m_numerator = yNum;
        }
        x.m_denominator = y.m_denominator = denom.abs().template narrow<2*N1+2*D1+2*D2+1>();
        return true;
    }

    template<int BITS_>
    struct FpIntVec2
    {
        HK_DECLARE_CLASS(FpIntVec2, New, Pod);

        enum { BITS = BITS_ };
        typedef FpInt<BITS> ScalarType;

        FpIntVec2() {}

        template<typename X, typename Y>
        FpIntVec2(X const& x, Y const& y) : m_x(x), m_y(y) {}

        template<int B2>
        explicit FpIntVec2(FpIntVec2<B2> const& other) : m_x(other.m_x), m_y(other.m_y) {}

        template<int B2>
        FpIntVec2 const& operator=(FpIntVec2<B2> const& other) { m_x = other.m_x; m_y = other.m_y; return *this; }

        template<int I>
        ScalarType const& getComponent() const
        {
            HK_COMPILE_TIME_ASSERT(I >= 0 && I < 2);
            if(I==0) { return m_x; }
            else { return m_y; }
        }

        void setZero()
        {
            m_x.setZero();
            m_y.setZero();
        }

        void setMin()
        {
            m_x.setMinVal();
            m_y.setMinVal();
        }

        template<typename T>
        void set(T const& x, T const& y)
        {
            m_x = x;
            m_y = y;
        }

        FpIntVec2<BITS> perp() const { return FpIntVec2(-m_y, m_x); }

        FpUint<2*BITS> lengthSquared() const
        {
            return (m_x*m_x + m_y*m_y).asUintNarrowing();
        }

        template<int RES_BITS>
        HK_INLINE FpIntVec2<RES_BITS> narrow() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS <= BITS);
            return FpIntVec2<RES_BITS>(m_x.template narrow<RES_BITS>(), m_y.template narrow<RES_BITS>());
        }

        template<int RES_BITS>
        HK_INLINE bool canNarrow() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS <= BITS);
            return m_x.template canNarrow<RES_BITS>() & m_y.template canNarrow<RES_BITS>();
        }

        void toFloat32(hkVector4f& vOut) const
        {
            vOut.set(m_x.toFloat32(), m_y.toFloat32(), 0.0f);
        }

        void toFloat32(hkVector2f& vOut) const
        {
            vOut.set(m_x.toFloat32(), m_y.toFloat32());
        }

        bool equalZero() const { return m_x.equalZero() & m_y.equalZero(); }

        hkUint32 hash() const
        {
            using namespace hkHash;
            return combineHashValues(hkHashValue(m_x), hkHashValue(m_y));
        }

        ScalarType m_x, m_y;
    };

    template<int B1, int B2>
    FpIntVec2<(B1>B2 ? B1 : B2)+1> operator+(FpIntVec2<B1> const& a, FpIntVec2<B2> const& b)
    {
        return FpIntVec2<(B1>B2 ? B1 : B2)+1>(a.m_x + b.m_x, a.m_y + b.m_y);
    }

    template<int B1, int B2>
    FpIntVec2<(B1>B2 ? B1 : B2)+1> operator-(FpIntVec2<B1> const& a, FpIntVec2<B2> const& b)
    {
        return FpIntVec2<(B1>B2 ? B1 : B2)+1>(a.m_x - b.m_x, a.m_y - b.m_y);
    }

    template<int B1>
    FpIntVec2<B1> operator-(FpIntVec2<B1> const& a)
    {
        return FpIntVec2<B1>(-a.m_x, -a.m_y);
    }

    template<int B1, int B2>
    FpInt<B1+B2+1> dot(FpIntVec2<B1> const& a, FpIntVec2<B2> const& b)
    {
        return a.m_x*b.m_x + a.m_y*b.m_y;
    }

    template<int B1, int B2>
    FpInt<B1+B2+1> perpDot(FpIntVec2<B1> const& a, FpIntVec2<B2> const& b)
    {
        return a.m_x*b.m_y - a.m_y*b.m_x;
    }

    template<int B1, int B2>
    bool operator==(FpIntVec2<B1> const& a, FpIntVec2<B2> const& b)
    {
        return a.m_x == b.m_x && a.m_y == b.m_y;
    }

    template<int B1, int B2>
    FpIntVec2<B1+B2> operator*(FpIntVec2<B1> const& a, FpUint<B2> const& b)
    {
        return FpIntVec2<B1+B2>(a.m_x*b, a.m_y*b);
    }

    template<int B1, int B2>
    FpIntVec2<B1+B2> operator*(FpIntVec2<B1> const& a, FpInt<B2> const& b)
    {
        return FpIntVec2<B1+B2>(a.m_x*b, a.m_y*b);
    }

    template<int B1, int B2>
    FpIntVec2<B1+B2> operator*(FpUint<B1> const& a, FpIntVec2<B2> const& b)
    {
        return b*a;
    }

    template<int B1, int B2>
    FpIntVec2<B1+B2> operator*(FpInt<B1> const& a, FpIntVec2<B2> const& b)
    {
        return b*a;
    }

    template<int BITS_>
    struct FpIntVec3
    {
        HK_DECLARE_CLASS(FpIntVec3, New, Pod);

        enum { BITS = BITS_ };
        typedef FpInt<BITS> ScalarType;

        typedef FpIntVec2<BITS> XyType;

        FpIntVec3() {}

        template<typename T>
        FpIntVec3(T const& x, T const& y, T const& z) : m_x(x), m_y(y), m_z(z) {}

        template<typename T>
        FpIntVec3(T const& xyz) : m_x(xyz.m_x), m_y(xyz.m_y), m_z(xyz.m_z) {}

        template<int I>
        ScalarType const& getComponent() const
        {
            HK_COMPILE_TIME_ASSERT(I >= 0 && I < 3);
            if(I==0) { return m_x; }
            else if(I==1) { return m_y; }
            else { return m_z; }
        }

        void setZero()
        {
            m_x.setZero();
            m_y.setZero();
            m_z.setZero();
        }


        void setMinVal()
        {
            m_x.setMinVal();
            m_y.setMinVal();
            m_z.setMinVal();
        }

        void setMaxVal()
        {
            m_x.setMaxVal();
            m_y.setMaxVal();
            m_z.setMaxVal();
        }

        template<int B1, int B2>
        void setMin(FpIntVec3<B1> a, FpIntVec3<B2> b)
        {
            m_x = hkMath::min2(a.m_x, b.m_x);
            m_y = hkMath::min2(a.m_y, b.m_y);
            m_z = hkMath::min2(a.m_z, b.m_z);
        }

        template<int B1, int B2>
        void setMax(FpIntVec3<B1> a, FpIntVec3<B2> b)
        {
            m_x = hkMath::max2(a.m_x, b.m_x);
            m_y = hkMath::max2(a.m_y, b.m_y);
            m_z = hkMath::max2(a.m_z, b.m_z);
        }

        template<typename T>
        void set(T const& x, T const& y, T const& z)
        {
            m_x = x;
            m_y = y;
            m_z = z;
        }

        template<int RES_BITS>
        HK_INLINE bool canNarrow() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS <= BITS);
            return m_x.template canNarrow<RES_BITS>() & m_y.template canNarrow<RES_BITS>() & m_z.template canNarrow<RES_BITS>();
        }

        template<int RES_BITS>
        HK_INLINE FpIntVec3<RES_BITS> narrow() const
        {
            HK_COMPILE_TIME_ASSERT(RES_BITS <= BITS);
            return FpIntVec3<RES_BITS>(m_x.template narrow<RES_BITS>(), m_y.template narrow<RES_BITS>(), m_z.template narrow<RES_BITS>());
        }

        void toFloat32(hkVector4f& vOut) const
        {
            vOut.set(m_x.toFloat32(), m_y.toFloat32(), m_z.toFloat32());
        }

        void setFromFloat32(hkVector4Parameter v)
        {
            m_x.setFromFloat32(v(0));
            m_y.setFromFloat32(v(1));
            m_z.setFromFloat32(v(2));
        }

        bool equalZero() const { return m_x.equalZero() & m_y.equalZero() & m_z.equalZero(); }

        FpIntVec2<BITS> getXY() const
        {
            return FpIntVec2<BITS>(m_x, m_y);
        }

        template<int N>
        void setXY(FpIntVec2<N> const& xy)
        {
            HK_COMPILE_TIME_ASSERT(N <= BITS);
            m_x = xy.m_x;
            m_y = xy.m_y;
        }

        ScalarType getComponent(int i) const
        {
            HK_COMPILE_TIME_ASSERT(HK_OFFSET_EQUALS(FpIntVec3<BITS>, m_y, HK_OFFSET_OF(FpIntVec3<BITS>, m_x)+sizeof(ScalarType)));
            HK_COMPILE_TIME_ASSERT(HK_OFFSET_EQUALS(FpIntVec3<BITS>, m_z, HK_OFFSET_OF(FpIntVec3<BITS>, m_x)+2*sizeof(ScalarType)));
            return *((&m_x)+i);
        }

        template<typename T>
        void setComponent(int i, T const& val) const
        {
            HK_COMPILE_TIME_ASSERT(HK_OFFSET_EQUALS(FpIntVec3<BITS>, m_y, HK_OFFSET_OF(FpIntVec3<BITS>, m_x)+sizeof(ScalarType)));
            HK_COMPILE_TIME_ASSERT(HK_OFFSET_EQUALS(FpIntVec3<BITS>, m_z, HK_OFFSET_OF(FpIntVec3<BITS>, m_x)+2*sizeof(ScalarType)));
            *((&m_x)+i) = val;
        }

        hkUint32 hash() const
        {
            using namespace hkHash;
            return combineHashValues(combineHashValues(hkHashValue(m_x), hkHashValue(m_y)), hkHashValue(m_z));
        }

        ScalarType m_x, m_y, m_z;
    };

    template<int N1, int N2>
    FpIntVec3<(N1>N2 ? N1 : N2)+1> operator+(FpIntVec3<N1> const& a, FpIntVec3<N2> const& b)
    {
        return FpIntVec3<(N1>N2 ? N1 : N2)+1>(a.m_x + b.m_x, a.m_y + b.m_y, a.m_z + b.m_z);
    }

    template<int N1, int N2>
    FpIntVec3<(N1>N2 ? N1 : N2)+1> operator-(FpIntVec3<N1> const& a, FpIntVec3<N2> const& b)
    {
        return FpIntVec3<(N1>N2 ? N1 : N2)+1>(a.m_x - b.m_x, a.m_y - b.m_y, a.m_z - b.m_z);
    }

    template<int B1, int B2>
    FpIntVec3<B1+B2+2> cross(FpIntVec3<B1> const& a, FpIntVec3<B2> const& b)
    {
        return FpIntVec3<B1+B2+2>(
            a.m_y*b.m_z - a.m_z*b.m_y,
            a.m_z*b.m_x - a.m_x*b.m_z,
            a.m_x*b.m_y - a.m_y*b.m_x);
    }

    template<int A_BITS, int B_BITS>
    HK_INLINE FpInt<A_BITS + B_BITS + 2>dot(FpIntVec3<A_BITS> const& a, FpIntVec3<B_BITS> const& b)
    {
        return a.m_x*b.m_x + a.m_y*b.m_y + a.m_z*b.m_z;
    }

    template<int A_BITS, int B_BITS, int C_BITS>
    HK_INLINE FpInt<A_BITS + B_BITS + C_BITS + 2> tripleProduct(FpIntVec3<A_BITS> const& a, FpIntVec3<B_BITS> const& b, FpIntVec3<C_BITS> const& c)
    {
        return
            (a.m_x * (b.m_y*c.m_z - b.m_z*c.m_y)
            + a.m_y * (b.m_z*c.m_x - b.m_x*c.m_z)
            + a.m_z * (b.m_x*c.m_y - b.m_y*c.m_x)).template narrow<A_BITS+B_BITS+C_BITS+2>();
    }

    template<int B1, int B2>
    bool operator==(FpIntVec3<B1> const& a, FpIntVec3<B2> const& b)
    {
        return a.m_x == b.m_x && a.m_y == b.m_y && a.m_z == b.m_z;
    }

    template<int B1, int B2>
    bool operator!=(FpIntVec3<B1> const& a, FpIntVec3<B2> const& b)
    {
        return !(a==b);
    }

    template<int B1, int B2>
    FpIntVec3<B1+B2> operator*(FpIntVec3<B1> const& a, FpUint<B2> const& b)
    {
        return FpIntVec3<B1+B2>(a.m_x*b, a.m_y*b, a.m_z*b);
    }

    template<int B1, int B2>
    FpIntVec3<B1+B2> operator*(FpIntVec3<B1> const& a, FpInt<B2> const& b)
    {
        return FpIntVec3<B1+B2>(a.m_x*b, a.m_y*b, a.m_z*b);
    }

    template<int B1, int B2>
    FpIntVec3<B1+B2> operator*(FpUint<B1> const& a, FpIntVec3<B2> const& b)
    {
        return b*a;
    }

    template<int B1, int B2>
    FpIntVec3<B1+B2> operator*(FpInt<B1> const& a, FpIntVec3<B2> const& b)
    {
        return b*a;
    }

    template<int B>
    FpIntVec3<B> operator-(FpIntVec3<B> const& a)
    {
        return FpIntVec3<B>(-a.m_x, -a.m_y, -a.m_z);
    }

    template<int N1, int N2>
    bool lexLess(hkFpMath::FpIntVec3<N1> const& a, hkFpMath::FpIntVec3<N2> const& b)
    {
        int xSgn = cmp(a.m_x, b.m_x);
        if(xSgn != 0)
        {
            return xSgn < 0;
        }

        int ySgn = cmp(a.m_y, b.m_y);
        if(ySgn != 0)
        {
            return ySgn < 0;
        }

        return a.m_z < b.m_z;
    }

    template<int NUMERATOR_BITS_, int DENOMINATOR_BITS_>
    struct FpRationalVec2
    {
        HK_DECLARE_CLASS(FpRationalVec2, New, Pod);

        enum { NUMERATOR_BITS = NUMERATOR_BITS_ };
        enum { DENOMINATOR_BITS = DENOMINATOR_BITS_ };

        FpRationalVec2() {}

        template<typename X, typename Y>
        FpRationalVec2(X const& x, Y const& y) : m_x(x), m_y(y) {}

        template<int B2>
        FpRationalVec2(FpIntVec2<B2> const& other) : m_x(other.m_x), m_y(other.m_y) {}

        template<int N2, int D2>
        FpRationalVec2 const& operator=(FpRationalVec2<N2, D2> const& other) { m_x = other.m_x; m_y = other.m_y; return *this; }

        template<int B2>
        FpRationalVec2 const& operator=(FpIntVec2<B2> const& other) { m_x = FpRational<NUMERATOR_BITS, DENOMINATOR_BITS>(other.m_x); m_y = FpRational<NUMERATOR_BITS, DENOMINATOR_BITS>(other.m_y); return *this; }

        hkUint32 hash() const
        {
            using namespace hkHash;
            return combineHashValues(hkHashValue(m_x), hkHashValue(m_y));
        }

        FpRational<NUMERATOR_BITS, DENOMINATOR_BITS> m_x, m_y;
    };

    template<int N1, int D1, int N2, int D2>
    FpRationalVec2<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> operator+(FpRationalVec2<N1, D1> const& a, FpRationalVec2<N2, D2> const& b)
    {
        return FpRationalVec2<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2>(a.m_x + b.m_x, a.m_y + b.m_y);
    }

    template<int N1, int D1, int N2, int D2>
    FpRationalVec2<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> operator-(FpRationalVec2<N1, D1> const& a, FpRationalVec2<N2, D2> const& b)
    {
        return FpRationalVec2<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2>(a.m_x - b.m_x, a.m_y - b.m_y);
    }

    template<int N1, int D1, int N2, int D2>
    FpRational<N1+N2+D1+D2+1, 2*D1+2*D2> dot(FpRationalVec2<N1, D1> const& a, FpRationalVec2<N2, D2> const& b)
    {
        return a.m_x*b.m_x + a.m_y*b.m_y;
    }

    template<int N1, int D1, int N2, int D2>
    FpRational<N1+N2+D1+D2+1, 2*D1+2*D2> perpDot(FpRationalVec2<N1, D1> const& a, FpRationalVec2<N2, D2> const& b)
    {
        return a.m_x*b.m_y - a.m_y*b.m_x;
    }

    template<int N1, int D1, int N2, int D2>
    bool operator==(FpRationalVec2<N1, D1> const& a, FpRationalVec2<N2, D2> const& b)
    {
        return a.m_x == b.m_x && a.m_y == b.m_y;
    }

    template<int N1, int D1, int N2, int D2>
    bool operator!=(FpRationalVec2<N1, D1> const& a, FpRationalVec2<N2, D2> const& b)
    {
        return !(a==b);
    }

    template<int N, int D>
    struct FpRationalVec3
    {
        FpRational<N, D> m_x, m_y, m_z;
    };

    /// A vector of two rational numbers which are guaranteed to share the same denominator
    template<int NUMERATOR_BITS_, int DENOMINATOR_BITS_>
    struct FpFusedRationalVec2
    {
        HK_DECLARE_CLASS(FpFusedRationalVec2, New, Pod);

        enum { NUMERATOR_BITS = NUMERATOR_BITS_ };
        enum { DENOMINATOR_BITS = DENOMINATOR_BITS_ };

        typedef FpIntVec2<NUMERATOR_BITS> NumeratorType;
        typedef FpUint<DENOMINATOR_BITS> DenominatorType;
        typedef FpRational<NUMERATOR_BITS, DENOMINATOR_BITS> ScalarType;

        FpFusedRationalVec2() {}

        // Note no two-argument constructor (it would be ambiguous)

        template<typename T, typename D>
        FpFusedRationalVec2(T const& x, T const& y, D const& d) : m_numerator(x, y), m_denominator(d) {}

        template<typename T>
        explicit FpFusedRationalVec2(T const& xy) : m_numerator(xy), m_denominator(ConstantOne()) {}


        template<int N2, int D2>
        FpFusedRationalVec2(FpFusedRationalVec2<N2, D2> const& other)
        {
            m_numerator = other.m_numerator;
            m_denominator = other.m_denominator;
        }

        template<int N2, int D2>
        void operator=(FpFusedRationalVec2<N2, D2> const& other)
        {
            m_numerator = other.m_numerator;
            m_denominator = other.m_denominator;
        }

        template<int N2>
        void operator=(FpIntVec2<N2> const& other)
        {
            m_numerator = other;
            m_denominator = 1;
        }

        NumeratorType m_numerator;
        DenominatorType m_denominator;

        template<int I>
        ScalarType getComponent() const {
            HK_COMPILE_TIME_ASSERT(I >= 0 && I < 2);
            if(I==0) { return getX(); }
            else { return getY(); }
        }

        void setZero()
        {
            m_numerator.setZero();
            m_denominator = ConstantOne();
        }

        void setMinVal()
        {
            m_numerator.setMinVal();
            m_denominator = 1;
        }

        template<typename T>
        void set(T const& x, T const& y)
        {
            m_numerator.set(x, y);
            m_denominator = ConstantOne();
        }

        ScalarType getX() const { return ScalarType(m_numerator.m_x, m_denominator); }
        ScalarType getY() const { return ScalarType(m_numerator.m_y, m_denominator); }

        NumeratorType getAsNum() const {
#ifdef HKFPMATH_CHECK_WORD_RANGE
            HK_ASSERT(0x89457233, m_denominator == ConstantOne());
#endif
            return m_numerator;
        }

        FpRational<2*NUMERATOR_BITS, 2*DENOMINATOR_BITS> lengthSquared() const
        {
            return FpRational<2*NUMERATOR_BITS+1, 2*DENOMINATOR_BITS>(m_numerator.lengthSquared(), m_denominator*m_denominator);
        }

        FpFusedRationalVec2<NUMERATOR_BITS, DENOMINATOR_BITS> reduced() const
        {
            if(m_numerator.m_x.equalZero())
            {
                ScalarType yReduced = getY().reduced(); // Also handles 0,0
                FpFusedRationalVec2<NUMERATOR_BITS, DENOMINATOR_BITS> res;
                res.m_numerator.m_x.setZero();
                res.m_numerator.m_y = yReduced.m_numerator;
                res.m_denominator = yReduced.m_denominator;
                return res;
            }
            else if(m_numerator.m_y.equalZero())
            {
                ScalarType xReduced = getX().reduced();
                FpFusedRationalVec2<NUMERATOR_BITS, DENOMINATOR_BITS> res;
                res.m_numerator.m_x = xReduced.m_numerator;
                res.m_numerator.m_y.setZero();
                res.m_denominator = xReduced.m_denominator;
                return res;
            }
            else
            {
                bool xNeg = m_numerator.m_x.lessZero();
                bool yNeg = m_numerator.m_y.lessZero();
                FpUint<NUMERATOR_BITS-1> xAbs = m_numerator.m_x.abs();
                FpUint<NUMERATOR_BITS-1> yAbs = m_numerator.m_y.abs();
                FpUint<NUMERATOR_BITS-1> xyGcd = gcd(xAbs, yAbs);

                FpUint<((NUMERATOR_BITS-1)<DENOMINATOR_BITS) ? (NUMERATOR_BITS-1) : DENOMINATOR_BITS> xydGcd = gcd(xyGcd, m_denominator);
                FpUint<((NUMERATOR_BITS-1)<DENOMINATOR_BITS) ? (NUMERATOR_BITS-1) : DENOMINATOR_BITS> rIgnored;
                FpUint<(NUMERATOR_BITS-1)> reducedXAbs, reducedYAbs;
                divMod(xAbs, xydGcd, reducedXAbs, rIgnored);
                divMod(yAbs, xydGcd, reducedYAbs, rIgnored);

                FpInt<NUMERATOR_BITS> reducedX = reducedXAbs, reducedY = reducedYAbs;
                if(xNeg)
                {
                    reducedX = -reducedX;
                }
                if(yNeg)
                {
                    reducedY = -reducedY;
                }

                FpUint<DENOMINATOR_BITS> reducedDenominator;
                divMod(m_denominator, xydGcd, reducedDenominator, rIgnored);

                return FpFusedRationalVec2<NUMERATOR_BITS, DENOMINATOR_BITS>(reducedX, reducedY, reducedDenominator);
            }
        }

        template<int N1, int D1>
        HK_INLINE bool canNarrow() const
        {
            return m_numerator.template canNarrow<N1>() & m_denominator.template canNarrow<D1>();
        }

        template<int N1, int D1>
        HK_INLINE FpFusedRationalVec2<N1, D1> narrow() const
        {
            FpFusedRationalVec2<N1, D1> res;
            res.m_numerator = m_numerator.template narrow<N1>();
            res.m_denominator = m_denominator.template narrow<D1>();
            return res;
        }

        void toFloat32(hkVector4f& vOut) const
        {
            hkVector2f v2;
            toFloat32(v2);
            v2.convertToVector4(vOut);
        }

        void toFloat32(hkVector2f& vOut) const
        {
            Detail::convertIntUintRatiosFloat(
                m_numerator.m_x.m_storage, m_numerator.m_y.m_storage,
                m_denominator.m_storage,
                &vOut.x);
        }

        hkUint32 hash() const
        {
            using namespace hkHash;
            return combineHashValues(hkHashValue(m_numerator), hkHashValue(m_denominator));
        }
    };

    template<int N1, int D1, int N2, int D2>
    bool operator==(FpFusedRationalVec2<N1, D1> const& a, FpFusedRationalVec2<N2, D2> const& b)
    {
        return a.getX() == b.getX() && a.getY() == b.getY();
    }

    template<int N1, int D1, int N2>
    bool operator==(FpFusedRationalVec2<N1, D1> const& a, FpIntVec2<N2> const& b)
    {
        return a.getX() == b.m_x && a.getY() == b.m_y;
    }

    template<int N1, int N2, int D2>
    bool operator==(FpIntVec2<N1> const& a, FpFusedRationalVec2<N2, D2> const& b)
    {
        return a.m_x == b.getX() && a.m_y == b.getY();
    }

    template<int N1, int D1, int N2, int D2>
    bool operator!=(FpFusedRationalVec2<N1, D1> const& a, FpFusedRationalVec2<N2, D2> const& b)
    {
        return !(a==b);
    }

    template<int N1, int D1, int N2>
    bool operator!=(FpFusedRationalVec2<N1, D1> const& a, FpIntVec2<N2> const& b)
    {
        return a.getX() != b.m_x || a.getY() != b.m_x;
    }

    template<int N1, int N2, int D2>
    bool operator!=(FpIntVec2<N1> const& a, FpFusedRationalVec2<N2, D2> const& b)
    {
        return a.m_x != b.getX() || a.m_y != b.getY();
    }

    template<int N1, int N2, int D2>
    FpFusedRationalVec2<N1+N2, D2> operator*(FpIntVec2<N1> const& a, FpRational<N2, D2> const& b)
    {
        FpFusedRationalVec2<N1+N2, D2> res;
        res.m_numerator = a*b.m_numerator;
        res.m_denominator = b.m_denominator;
        return res;
    }

    template<int N1, int D1, int N2>
    FpFusedRationalVec2<N1+N2, D1> operator*(FpRational<N1, D1> const& a, FpIntVec2<N2> const& b)
    {
        return b*a;
    }

    template<int N1, int D1, int N2, int D2>
    FpFusedRationalVec2<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> operator+(FpFusedRationalVec2<N1, D1> const& a, FpFusedRationalVec2<N2, D2> const& b)
    {
        FpFusedRationalVec2<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> res;
        res.m_numerator.m_x = a.m_numerator.m_x * b.m_denominator + b.m_numerator.m_x * a.m_denominator;
        res.m_numerator.m_y = a.m_numerator.m_y * b.m_denominator + b.m_numerator.m_y * a.m_denominator;
        res.m_denominator = a.m_denominator * b.m_denominator;
        return res;
    }

    template<int N1, int D1, int N2, int D2>
    FpFusedRationalVec2<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> operator-(FpFusedRationalVec2<N1, D1> const& a, FpFusedRationalVec2<N2, D2> const& b)
    {
        FpFusedRationalVec2<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> res;
        res.m_numerator.m_x = a.m_numerator.m_x * b.m_denominator - b.m_numerator.m_x * a.m_denominator;
        res.m_numerator.m_y = a.m_numerator.m_y * b.m_denominator - b.m_numerator.m_y * a.m_denominator;
        res.m_denominator = a.m_denominator * b.m_denominator;
        return res;
    }

    template<int N1, int N2>
    bool lexLess(hkFpMath::FpIntVec2<N1> const& a, hkFpMath::FpIntVec2<N2> const& b)
    {
        int xSgn = cmp(a.m_x, b.m_x);
        if(xSgn != 0)
        {
            return xSgn < 0;
        }

        return a.m_y < b.m_y;
    }

    template<int N>
    bool lexLessZero(hkFpMath::FpIntVec2<N> const& a)
    {
        if(a.m_x.lessZero())
        {
            return true;
        }
        else if(a.m_x.equalZero())
        {
            return a.m_y.lessZero();
        }
        else
        {
            return false;
        }
    }

    template<int N>
    bool lexGreaterZero(hkFpMath::FpIntVec2<N> const& a)
    {
        if(a.m_x.lessZero())
        {
            return false;
        }
        else if(a.m_x.equalZero())
        {
            return a.m_y.greaterZero();
        }
        else
        {
            return true;
        }
    }

    template<int N>
    int lexCmpZero(hkFpMath::FpIntVec2<N> const& a)
    {
        int xSgn = a.m_x.sgn();
        if(xSgn != 0)
        {
            return xSgn;
        }
        else
        {
            return a.m_y.sgn();
        }
    }

    template<int N1, int D1, int N2, int D2>
    bool lexLess(hkFpMath::FpFusedRationalVec2<N1, D1> const& a, hkFpMath::FpFusedRationalVec2<N2, D2> const& b)
    {
        int xSgn = cmp(a.getX(), b.getX());
        if(xSgn != 0)
        {
            return xSgn < 0;
        }

        return a.getY() < b.getY();
    }

    /// A lexLess implementation which looks for ways to avoid the multiplication. This will not provide much of a
    /// speedup if the two points are known to be nearly equal.
    template<int N1, int D1, int N2, int D2>
    bool lexLessFast(hkFpMath::FpFusedRationalVec2<N1, D1> const& a, hkFpMath::FpFusedRationalVec2<N2, D2> const& b)
    {
        // First look for easy cases
        int aSgn = lexCmpZero(a.m_numerator), bSgn = lexCmpZero(b.m_numerator);
        if(aSgn != bSgn)
        {
            // different halfplanes
            return aSgn < bSgn;
        }
        else if(aSgn == 0)
        {
            // both operands are zero; not sure if testing this case saves us time
            return false;
        }
        else
        {
            // same halfplane; if the denominators and numerators point in opposite directions, we don't have to multiply
            int denSgn = cmp(a.m_denominator, b.m_denominator);
            if(denSgn == 0)
            {
                // equal denominators
                return lexLess(a.m_numerator, b.m_numerator);
            }
            else if((denSgn < 0) == (aSgn < 0))
            {
                if(!lexLess(b.m_numerator, a.m_numerator))
                {
                    return true;
                }
            }
            else // (denSgn < 0) != (aSgn < 0)
            {
                if(!lexLess(a.m_numerator, b.m_numerator))
                {
                    return false;
                }
            }
        }

        return lexLess(a, b);
        // Do general processing
        int xSgn = cmp(a.getX(), b.getX());
        if(xSgn != 0)
        {
            return xSgn < 0;
        }

        return a.getY() < b.getY();
    }

    template<int N1, int N2>
    int lexCmp(hkFpMath::FpIntVec2<N1> const& a, hkFpMath::FpIntVec2<N2> const& b)
    {
        int xSgn = cmp(a.m_x, b.m_x);
        if(xSgn != 0)
        {
            return xSgn;
        }

        return cmp(a.m_y, b.m_y);
    }

    template<int N1, int D1, int N2, int D2>
    int lexCmp(hkFpMath::FpFusedRationalVec2<N1, D1> const& a, hkFpMath::FpFusedRationalVec2<N2, D2> const& b)
    {
        if(a.m_denominator == b.m_denominator)
        {
            return lexCmp(a.m_numerator, b.m_numerator);
        }

        int xSgn = cmp(a.getX(), b.getX());
        if(xSgn != 0)
        {
            return xSgn;
        }

        return cmp(a.getY(), b.getY());
    }


    template<int N1, int D1, int N2, int D2>
    FpRational<N1+N2+1, D1+D2> dot(FpFusedRationalVec2<N1, D1> const& a, FpFusedRationalVec2<N2, D2> const& b)
    {
        return FpRational<N1+N2+1, D1+D2>(dot(a.m_numerator, b.m_numerator), a.m_denominator*b.m_denominator);
    }

    template<int N1, int D1, int N2>
    FpRational<N1+N2+1, D1> dot(FpFusedRationalVec2<N1, D1> const& a, FpIntVec2<N2> const& b)
    {
        return FpRational<N1+N2+1, D1>(dot(a.m_numerator, b), a.m_denominator);
    }

    template<int N1, int N2, int D2>
    FpRational<N1+N2+1, D2> dot(FpIntVec2<N1> const& a, FpFusedRationalVec2<N2, D2> const& b)
    {
        return FpRational<N1+N2+1, D2>(dot(a, b.m_numerator), b.m_denominator);
    }

    template<int N1, int D1, int N2, int D2>
    FpRational<N1+N2+1, D1+D2> perpDot(FpFusedRationalVec2<N1, D1> const& a, FpFusedRationalVec2<N2, D2> const& b)
    {
        return FpRational<N1+N2+1, D1+D2>(perpDot(a.m_numerator, b.m_numerator), a.m_denominator*b.m_denominator);
    }

    template<int N1, int N2, int D2>
    FpRational<N1+N2+1, D2> perpDot(FpIntVec2<N1> const& a, FpFusedRationalVec2<N2, D2> const& b)
    {
        return FpRational<N1+N2+1, D2>(perpDot(a, b.m_numerator), b.m_denominator);
    }

    template<int N1, int D1, int N2>
    FpRational<N1+N2+1, D1> perpDot(FpFusedRationalVec2<N1, D1> const& a, FpIntVec2<N2> const& b)
    {
        return FpRational<N1+N2+1, D1>(perpDot(a.m_numerator, b), a.m_denominator);
    }

    /// Return positive if abc winds CCW, negative if abc winds CW, and zero if the three are parallel
    template<int N1, int N2, int N3>
    int windingSgn(FpIntVec2<N1> const& a, FpIntVec2<N2> const& b, FpIntVec2<N3> const& c)
    {
        return perpDot(b-a, c-a).sgn();
    }

    /// Return positive if abc winds CCW, negative if abc winds CW, and zero if the three are parallel
    template<int N1, int D1, int N2, int D2, int N3, int D3>
    int windingSgn(FpFusedRationalVec2<N1, D1> const& a, FpFusedRationalVec2<N2, D2> const& b, FpFusedRationalVec2<N3, D3> const& c)
    {
        return
            (perpDot(a.m_numerator, b.m_numerator)*c.m_denominator
            + perpDot(b.m_numerator, c.m_numerator)*a.m_denominator
            + perpDot(c.m_numerator, a.m_numerator)*b.m_denominator).sgn();
    }

    /// A vector of three rational numbers which are guaranteed to share the same denominator
    template<int NUMERATOR_BITS_, int DENOMINATOR_BITS_>
    struct FpFusedRationalVec3
    {
        HK_DECLARE_CLASS(FpFusedRationalVec3, New, Pod);

        enum { NUMERATOR_BITS = NUMERATOR_BITS_ };
        enum { DENOMINATOR_BITS = DENOMINATOR_BITS_ };

        typedef FpIntVec3<NUMERATOR_BITS> NumeratorType;
        typedef FpUint<DENOMINATOR_BITS> DenominatorType;
        typedef FpRational<NUMERATOR_BITS, DENOMINATOR_BITS> ScalarType;
        typedef FpFusedRationalVec2<NUMERATOR_BITS, DENOMINATOR_BITS> XyType;

        FpFusedRationalVec3() {}

        template<typename T>
        HK_INLINE explicit FpFusedRationalVec3(T const& n) : m_numerator(n), m_denominator(ConstantOne()) { }

        template<typename T>
        FpFusedRationalVec3(T const& x, T const& y, T const& z) : m_numerator(x, y, z), m_denominator(ConstantOne()) {}

        template<int N2, int D2>
        FpFusedRationalVec3 const& operator=(FpFusedRationalVec3<N2, D2> const& other) { m_numerator = other.m_numerator; m_denominator = other.m_denominator; return *this; }

        NumeratorType m_numerator;
        DenominatorType m_denominator;

        ScalarType getX() const { return ScalarType(m_numerator.m_x, m_denominator); }
        ScalarType getY() const { return ScalarType(m_numerator.m_y, m_denominator); }
        ScalarType getZ() const { return ScalarType(m_numerator.m_z, m_denominator); }

        template<int I>
        ScalarType getComponent() const
        {
            HK_COMPILE_TIME_ASSERT(I >= 0 && I < 3);
            if(I==0) { return getX(); }
            else if(I==1) { return getY(); }
            else { return getZ(); }
        }

        ScalarType getComponent(int i) const
        {
            return ScalarType(m_numerator.getComponent(i), m_denominator);
        }

        void toFloat32(hkVector4f& vOut) const
        {
            Detail::convertIntUintRatiosFloat(
                m_numerator.m_x.m_storage, m_numerator.m_y.m_storage, m_numerator.m_z.m_storage,
                m_denominator.m_storage,
                &vOut(0));
        }

        template<int N1, int D1>
        HK_INLINE bool canNarrow() const
        {
            return m_numerator.template canNarrow<N1>() & m_denominator.template canNarrow<D1>();
        }

        template<int N1, int D1>
        HK_INLINE FpFusedRationalVec3<N1, D1> narrow() const
        {
            HK_COMPILE_TIME_ASSERT(N1 < NUMERATOR_BITS);
            HK_COMPILE_TIME_ASSERT(D1 < DENOMINATOR_BITS);

            FpFusedRationalVec3<N1, D1> res;
            res.m_numerator = m_numerator.template narrow<N1>();
            res.m_denominator = m_denominator.template narrow<D1>();
            return res;
        }

        FpFusedRationalVec3<NUMERATOR_BITS, DENOMINATOR_BITS> reduced() const
        {
            DenominatorType fullGcd = m_denominator;
            if(!m_numerator.m_x.equalZero())
            {
                fullGcd = gcd(fullGcd, m_numerator.m_x.abs());
            }
            if(!m_numerator.m_y.equalZero())
            {
                fullGcd = gcd(fullGcd, m_numerator.m_y.abs());
            }
            if(!m_numerator.m_z.equalZero())
            {
                fullGcd = gcd(fullGcd, m_numerator.m_z.abs());
            }

            FpFusedRationalVec3<NUMERATOR_BITS, DENOMINATOR_BITS> res;

            divExact(m_numerator.m_x, fullGcd, res.m_numerator.m_x);
            divExact(m_numerator.m_y, fullGcd, res.m_numerator.m_y);
            divExact(m_numerator.m_z, fullGcd, res.m_numerator.m_z);
            divExact(m_denominator, fullGcd, res.m_denominator);

            return res;
        }

        XyType getXY() const { return XyType(m_numerator.m_x, m_numerator.m_y, m_denominator); }

        hkUint32 hash() const
        {
            using namespace hkHash;
            return combineHashValues(hkHashValue(m_numerator), hkHashValue(m_denominator));
        }
    };

    template<int N1, int D1, int N2, int D2>
    bool operator==(FpFusedRationalVec3<N1, D1> const& a, FpFusedRationalVec3<N2, D2> const& b)
    {
        return a.getX() == b.getX() && a.getY() == b.getY() && a.getZ() == b.getZ();
    }

    template<int N1, int D1, int N2, int D2>
    bool operator!=(FpFusedRationalVec3<N1, D1> const& a, FpFusedRationalVec3<N2, D2> const& b)
    {
        return !(a==b);
    }

    template<int N1, int D1, int N2>
    bool operator==(FpFusedRationalVec3<N1, D1> const& a, FpIntVec3<N2> const& b)
    {
        return a.getX() == b.m_x && a.getY() == b.m_y && a.getZ() == b.m_z;
    }

    template<int N1, int D1, int N2>
    bool operator!=(FpFusedRationalVec3<N1, D1> const& a, FpIntVec3<N2> const& b)
    {
        return !(a==b);
    }

    template<int N1, int N2, int D2>
    bool operator==(FpIntVec3<N1> const& a, FpFusedRationalVec3<N2, D2> const& b)
    {
        return a.m_x == b.getX() && a.m_y == b.getY() && a.m_z == b.getZ();
    }

    template<int N1, int N2, int D2>
    bool operator!=(FpIntVec3<N1> const& a, FpFusedRationalVec3<N2, D2> const& b)
    {
        return !(a==b);
    }

    template<int N1, int D1, int N2, int D2>
    FpFusedRationalVec3<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> operator+(FpFusedRationalVec3<N1, D1> const& a, FpFusedRationalVec3<N2, D2> const& b)
    {
        FpFusedRationalVec3<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> res;
        res.m_numeratorX = a.m_numeratorX * b.m_denominator + b.m_numeratorX * a.m_denominator;
        res.m_numeratorY = a.m_numeratorY * b.m_denominator + b.m_numeratorY * a.m_denominator;
        res.m_numeratorY = a.m_numeratorZ * b.m_denominator + b.m_numeratorZ * a.m_denominator;
        res.m_denominator = a.m_denominator * b.m_denominator;
        return res;
    }

    template<int N1, int D1, int N2, int D2>
    FpFusedRationalVec3<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> operator-(FpFusedRationalVec3<N1, D1> const& a, FpFusedRationalVec3<N2, D2> const& b)
    {
        FpFusedRationalVec3<(N1 + D2>N2 + D1 ? N1 + D2 : N2 + D1) + 1, D1 + D2> res;
        res.m_numeratorX = a.m_numeratorX * b.m_denominator - b.m_numeratorX * a.m_denominator;
        res.m_numeratorY = a.m_numeratorY * b.m_denominator - b.m_numeratorY * a.m_denominator;
        res.m_numeratorY = a.m_numeratorZ * b.m_denominator - b.m_numeratorZ * a.m_denominator;
        res.m_denominator = a.m_denominator * b.m_denominator;
        return res;
    }

    template<int N1, int D1, int N2, int D2>
    bool lexLess(hkFpMath::FpFusedRationalVec3<N1, D1> const& a, hkFpMath::FpFusedRationalVec3<N2, D2> const& b)
    {
        int xSgn = cmp(a.getX(), b.getX());
        if(xSgn != 0)
        {
            return xSgn < 0;
        }

        int ySgn = cmp(a.getY(), b.getY());
        if(ySgn != 0)
        {
            return ySgn < 0;
        }
        return a.getZ() < b.getZ();
    }

    // Produce the linear interpolation, with regards to a rational interpolant, between two integer endpoints.
    template<int N1, int N2, int D2>
    FpRational<N1+N2+1, D2> interp(FpInt<N1> a, FpInt<N1> b, FpRational<N2, D2> t)
    {
        return FpRational<N1+N2+2, D2>(a + (b-a)*t.m_numerator, t.m_denominator).template narrow<N1+N2+1, D2>();
    }

    // Produce the linear interpolation, with regards to a rational interpolant, between two intvec2 endpoints.
    template<int N1, int N2, int D2>
    FpFusedRationalVec2<((N1+D2>N1+N2+1)?N1+D2:N1+N2+1)+1, D2> interp(FpIntVec2<N1> a, FpIntVec2<N1> b, FpRational<N2, D2> t)
    {
        FpIntVec2<N1+1> ab = b-a;
        FpFusedRationalVec2<((N1+D2>N1+N2+1)?N1+D2:N1+N2+1)+1, D2> res;
        res.m_numerator.m_x = a.m_x * t.m_denominator + ab.m_x * t.m_numerator;
        res.m_numerator.m_y = a.m_y * t.m_denominator + ab.m_y * t.m_numerator;
        res.m_denominator = t.m_denominator;
        return res;
    }

    // Produce the 2D interpolation, with regards to a fused 2D rational interpolant, between three intvec3 endpoints.
    template<int N1, int N2, int D2>
    FpFusedRationalVec3<N1+(N2>D2?N2:D2)+2, D2> interp(FpIntVec3<N1> a, FpIntVec3<N1> b, FpIntVec3<N1> c, FpFusedRationalVec2<N2, D2> uv)
    {
        FpIntVec3<N1+1> ab = b-a;
        FpIntVec3<N1+1> ac = c-a;
        FpFusedRationalVec3<N1+(N2>D2?N2:D2)+2, D2> res;
        res.m_numerator.m_x = (a.m_x * uv.m_denominator + ab.m_x * uv.m_numerator.m_x + ac.m_x * uv.m_numerator.m_y).template narrow<N1+(N2>D2 ? N2 : D2)+2>();
        res.m_numerator.m_y = (a.m_y * uv.m_denominator + ab.m_y * uv.m_numerator.m_x + ac.m_y * uv.m_numerator.m_y).template narrow<N1+(N2>D2 ? N2 : D2)+2>();
        res.m_numerator.m_z = (a.m_z * uv.m_denominator + ab.m_z * uv.m_numerator.m_x + ac.m_z * uv.m_numerator.m_y).template narrow<N1+(N2>D2 ? N2 : D2)+2>();
        res.m_denominator = uv.m_denominator;
        return res;
    }

    template<int N1, int D1, int N2, int D2>
    FpRational<N1+N2+2, D1+D2> dot(FpFusedRationalVec3<N1, D1> const& a, FpFusedRationalVec3<N2, D2> const& b)
    {
        return FpRational<N1+N2+2, D1+D2>(a.m_numerator.m_x*b.m_numerator.m_x + a.m_numerator.m_y*b.m_numerator.m_y + a.m_numerator.m_z*b.m_numerator.m_z, a.m_denominator*b.m_denominator);
    }

    template<int N1, int D1, int N2>
    FpRational<N1+N2+2, D1> dot(FpFusedRationalVec3<N1, D1> const& a, FpIntVec3<N2> const& b)
    {
        return FpRational<N1+N2+2, D1>(a.m_numerator.m_x*b.m_x + a.m_numerator.m_y*b.m_y + a.m_numerator.m_z*b.m_z, a.m_denominator);
    }

    template<int N1, int N2, int D2>
    FpRational<N1+N2+2, D2> dot(FpIntVec3<N1> const& a, FpFusedRationalVec3<N2, D2> const& b)
    {
        return FpRational<N1+N2+2, D2>(a.m_x*b.m_numerator.m_x + a.m_y*b.m_numerator.m_y + a.m_z*b.m_numerator.m_z, b.m_denominator);
    }

    template<int B1, int B2, int N3, int D3>
    bool solve2x2(
        FpInt<B1> a,
        FpInt<B1> b,
        FpInt<B1> c,
        FpInt<B1> d,
        FpInt<B2> e,
        FpInt<B2> f,
        FpFusedRationalVec2<N3, D3>& xy)
    {
        HK_COMPILE_TIME_ASSERT(N3 >= B1+B2+1);
        HK_COMPILE_TIME_ASSERT(D3 >= 2*B1);

        FpInt<2*B1+1> denom = a*d - b*c;

        if(denom.equalZero())
        {
            return false;
        }

        FpInt<B1+B2+1> xNum = d*e-b*f;
        FpInt<B1+B2+1> yNum = a*f-c*e;

        if(denom.lessZero())
        {
            xy.m_numerator.m_x = -xNum;
            xy.m_numerator.m_y = -yNum;
        }
        else
        {
            xy.m_numerator.m_x = xNum;
            xy.m_numerator.m_y = yNum;
        }
        xy.m_denominator = denom.abs();
        return true;
    }

    template<int Bab, int Bcd, int Bef1, int Bef2, int NOut1, int NOut2, int DOut>
    bool solve2x2x2(
        FpIntVec2<Bab> ab,
        FpIntVec2<Bcd> cd,
        FpIntVec2<Bef1> ef1,
        FpIntVec2<Bef2> ef2,
        FpIntVec2<NOut1> & xy1NumOut,
        FpIntVec2<NOut2> & xy2NumOut,
        FpUint<DOut> & denomOut)
    {
        const int Babcd = Bab > Bcd ? Bab : Bcd;
        HK_COMPILE_TIME_ASSERT(NOut1 >= Babcd+Bef1+1);
        HK_COMPILE_TIME_ASSERT(NOut2 >= Babcd+Bef2+1);
        HK_COMPILE_TIME_ASSERT(DOut >= Bab+Bcd);

        FpInt<Bab+Bcd+1> denom = perpDot(ab, cd);

        if(denom.equalZero())
        {
            return false;
        }

        FpIntVec2<Babcd+Bef1+1> xy1Num(cd.m_y*ef1.m_x-ab.m_y*ef1.m_y, ab.m_x*ef1.m_y-cd.m_x*ef1.m_x);
        FpIntVec2<Babcd+Bef2+1> xy2Num(cd.m_y*ef2.m_x-ab.m_y*ef2.m_y, ab.m_x*ef2.m_y-cd.m_x*ef2.m_x);

        if(denom.lessZero())
        {
            xy1Num = -xy1Num;
            xy2Num = -xy2Num;
        }

        xy1NumOut = xy1Num;
        xy2NumOut = xy2Num;
        denomOut = denom.abs();

        return true;
    }

    template<int B>
    struct FpIntAabb
    {
        HK_DECLARE_CLASS(FpIntAabb, New, Pod);

        typedef FpIntVec3<B> ExtentType;
        typedef typename ExtentType::ScalarType ScalarType;

        ExtentType m_min;
        ExtentType m_max;

        void setFromPoint(ExtentType x) { m_min = x; m_max = x; }

        void setFromLine(ExtentType x, ExtentType y) { m_min.setMin(x, y); m_max.setMax(x, y); }

        void setFromTriangle(ExtentType x, ExtentType y, ExtentType z) { setFromLine(x, y); includePoint(z); }

        void setFromPoints(hkArrayView<const ExtentType> points)
        {
            m_min = m_max = points[0];
            for(int i = 1; i<points.getSize(); i++)
            {
                includePoint(points[i]);
            }
        }

        void setFull()
        {
            m_min.setMinVal();
            m_max.setMaxVal();
        }

        void setFromFloat32(hkAabb const& aabb)
        {
            m_min.setFromFloat32(aabb.m_min);
            m_max.setFromFloat32(aabb.m_max);
        }

        void includePoint(ExtentType x) { m_min.setMin(m_min, x); m_max.setMax(m_max, x); }

        void toFloat32(hkAabb & aabbOut) const
        {
            m_min.toFloat32(aabbOut.m_min);
            m_max.toFloat32(aabbOut.m_max);
        }

        template<int B2>
        bool containsPointClosed(FpIntVec3<B2> const& p) const
        {
            for(int i = 0; i<3; i++)
            {
                if(p.getComponent(i) < m_min.getComponent(i) || p.getComponent(i) > m_max.getComponent(i))
                {
                    return false;
                }
            }
            return true;
        }

        template<int N2, int D2>
        bool containsPointClosed(FpFusedRationalVec3<N2, D2> const& p) const
        {
            for(int i = 0; i<3; i++)
            {
                if(p.getComponent(i) < m_min.getComponent(i) || p.getComponent(i) > m_max.getComponent(i))
                {
                    return false;
                }
            }
            return true;
        }
    };

    template<int B1, int B2>
    FpIntAabb<(B1>B2?B1:B2)> intersect(FpIntAabb<B1> const& a, FpIntAabb<B2> const& b)
    {
        FpIntAabb<(B1>B2?B1:B2)> res;
        res.m_min.setMax(a.m_min, b.m_min);
        res.m_max.setMin(a.m_max, b.m_max);
        return res;
    }

    template<int B1, int B2>
    bool overlapOpen(FpIntAabb<B1> const& a, FpIntAabb<B2> const& b)
    {
        for(int i = 0; i<3; i++)
        {
            if(a.m_min.getComponent(i) >= b.m_max.getComponent(i))
            {
                return false;
            }
            if(b.m_min.getComponent(i) >= a.m_max.getComponent(i))
            {
                return false;
            }
        }
        return true;
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
