// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>

/// A complete list of axial rotation constants.
/// Each axial rotation corresponds to an orientation preserving element of the cubic/octahedral symmetry group:
/// http://en.wikipedia.org/wiki/Octahedral_symmetry
enum hkAxialRotationTag
{
    HK_AXIAL_ROT_IDENTITY = 0,

    // Rotations around one of the axes. The 90, 270 degree rotations correspond to S4's 4 cycles, the
    // 180 degree rotations correspond to S4's double transpositions.
    HK_AXIAL_ROT_X90 = 1,
    HK_AXIAL_ROT_X180 = 2,
    HK_AXIAL_ROT_X270 = 3,
    HK_AXIAL_ROT_Y90 = 4,
    HK_AXIAL_ROT_Y180 = 5,
    HK_AXIAL_ROT_Y270 = 6,
    HK_AXIAL_ROT_Z90 = 7,
    HK_AXIAL_ROT_Z180 = 8,
    HK_AXIAL_ROT_Z270 = 9,

    // Rotations of 120 and 240 degrees around one of the diagonals of the cube. These correspond to S4's 3-cycles.
    HK_AXIAL_ROT_DIAG1_ROT120 = 10,
    HK_AXIAL_ROT_DIAG1_ROT240 = 11,
    HK_AXIAL_ROT_DIAG2_ROT120 = 12,
    HK_AXIAL_ROT_DIAG2_ROT240 = 13,
    HK_AXIAL_ROT_DIAG3_ROT120 = 14,
    HK_AXIAL_ROT_DIAG3_ROT240 = 15,
    HK_AXIAL_ROT_DIAG4_ROT120 = 16,
    HK_AXIAL_ROT_DIAG4_ROT240 = 17,

    // Rotations around an axis connecting the mid points of two opposing edges of the cube. These correspond to S4's transpositions.
    HK_AXIAL_ROT_TRANSP1 = 18,
    HK_AXIAL_ROT_TRANSP2 = 19,
    HK_AXIAL_ROT_TRANSP3 = 20,
    HK_AXIAL_ROT_TRANSP4 = 21,
    HK_AXIAL_ROT_TRANSP5 = 22,
    HK_AXIAL_ROT_TRANSP6 = 23,

    HK_NUM_AXIAL_ROTATIONS = 24,
};

/// A rotation that preserves the property of being axis aligned.
class HK_EXPORT_COMMON hkAxialRotation
{
public:
    HK_DECLARE_CLASS(hkAxialRotation, New, Reflect);

    /// Sets this rotation to the identity.
    inline void setIdentity();

    /// Gets the given constant rotation.
    static inline hkAxialRotation getConstant(hkAxialRotationTag tag);

    /// Sets this rotation composition of A and B. The effect of the resulting rotation,
    /// when applied to a vector x, will be the same as the result of a rotation by B,
    /// followed by a rotation by A.
    inline void setComposition(const hkAxialRotation& a, const hkAxialRotation& b);

    /// Sets this rotation to the inverse of \p a, ie. the rotation such that the
    /// composition of a with this inverse rotation gives the identity rotation.
    inline void setInverse(const hkAxialRotation &a);

    /// Appends a 90 degree rotation around the x-axis to this rotation. This is equivalent to
    ///   rot.setComposition(hkAxialRotation::getConstant(HK_AXIAL_ROT_90_DEG_X), rot);
    inline void rotateX90Deg();

    /// Appends a 90 degree rotation around the y-axis to this rotation. This is equivalent to
    ///   rot.setComposition(hkAxialRotation::getConstant(HK_AXIAL_ROT_90_DEG_Y), rot);
    inline void rotateY90Deg();

    /// Appends a 90 degree rotation around the z-axis to this rotation. This is equivalent to
    ///   rot.setComposition(hkAxialRotation::getConstant(HK_AXIAL_ROT_90_DEG_Z), rot);
    inline void rotateZ90Deg();

    /// Gets the hkRotation representation of this rotation.
    inline void getRotation(hkRotation& rot) const;

    /// Sets this rotation to the axial rotation closest to the given rotation.
    void setRotation(const hkRotation& rot);

    /// Applies this rotation to \p v.
    inline void rotateVector(hkVector4& v) const;

    /// Applies the inverse of this rotation to \p v.
    inline void rotateInverseVector(hkVector4& v) const;

    /// Applies this rotation to \p aabb.
    inline void rotateAabb(hkAabb& aabb) const;

    /// Applies the inverse of this rotation to \p aabb.
    inline void rotateInverseAabb(hkAabb& aabb) const;

    inline hkBool32 operator == (const hkAxialRotation& b) const;

private:
    hkUint8 m_tag;

    static const hkUint8 s_multTable[24][24];
    static const hkUint8 s_inverseTable[24];
};

#include <Common/Base/Math/AxialTransform/hkAxialRotation.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
