// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0


#include <Common/Base/hkBase.h>
#include <Common/Base/Types/hkBaseTypes.h>
#include <Common/Base/DebugUtil/DeterminismUtil/hkNetworkedDeterminismUtil.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>

#define DEBUG_LOG_IDENTIFIER "common.base.determinism.networked"
#include <Common/Base/System/Log/hkLog.hxx>

hkNetworkedDeterminismUtil* hkNetworkedDeterminismUtil::s_instance = HK_NULL;

hkNetworkedDeterminismUtil::hkNetworkedDeterminismUtil(_In_z_ const char* hostname, int port)
{
    m_serverAddress = hostname;
    m_serverPort = port;

    m_server = HK_NULL;
    m_client = HK_NULL;
}

hkNetworkedDeterminismUtil::~hkNetworkedDeterminismUtil()
{
    delete m_server; m_server = HK_NULL;
    delete m_client; m_server = HK_NULL;
}

void HK_CALL hkNetworkedDeterminismUtil::create(_In_z_ const char* host, int port)
{
    HK_ASSERT(0xad903091, ! s_instance, "An instance already created.");

#if defined (HK_ENABLE_NETWORKED_DETERMINISM_UTIL)
    s_instance = new hkNetworkedDeterminismUtil(host, port);

    if ( ! s_instance->tryToCreateClient() )
    {
        s_instance->createServer();
    }
    hkCheckDeterminismUtil::createInstance();

#endif
}


void HK_CALL hkNetworkedDeterminismUtil::destroy()
{
#if defined (HK_ENABLE_NETWORKED_DETERMINISM_UTIL)
    HK_ASSERT(0xad903091, s_instance, "An instance does not exist.");

    delete s_instance;
    s_instance = HK_NULL;

    hkCheckDeterminismUtil::destroyInstance();
#endif
}


hkNetworkedDeterminismUtil::Server::Server(int listeningPort, int maxNumClients)
{
    m_listeningSocket = hkSocket::create();
    m_maxNumClients = maxNumClients;

    HK_ASSERT(0xad903092, m_listeningSocket, "Socket not created.");

    if (m_listeningSocket)
    {
        m_listeningSocket->listen(listeningPort);
        Log_Info( "hkNetworkedDeterminismUtil::Server created and will poll for new client(s) on port {} every frame", listeningPort );
    }
    else
    {
        Log_Info( "hkNetworkedDeterminismUtil::Server could not be created, please check that you platform supports sockets with the hkBase library" );
    }
}

hkNetworkedDeterminismUtil::Server::~Server()
{
    delete m_listeningSocket;

    for (int ci = 0; ci < m_clients.getSize(); ci++)
    {
        delete m_clients[ci];
    }
    m_clients.clearAndDeallocate();
}

hkNetworkedDeterminismUtil::Client::Client(hkSocket* socket)
{
    m_socket = socket;
}

hkNetworkedDeterminismUtil::Client::~Client()
{
    delete m_socket;
}


void hkNetworkedDeterminismUtil::Server::pollForNewClients()
{
    if (m_listeningSocket && m_clients.getSize() < m_maxNumClients)
    {
        hkSocket* newClient = m_listeningSocket->pollForNewClient();
        if (newClient)
        {
            m_clients.pushBack(newClient);
        }
    }
}

void hkNetworkedDeterminismUtil::Server::sendCommand(const Command& command)
{
    for (int i = 0; i < m_clients.getSize(); i++)
    {
        hkSocket* client = m_clients[i];

        client->getWriter().write(&command.m_type, sizeof(command.m_type));
        const int size = command.m_data.getSize();
        client->getWriter().write(&size, sizeof(size));
        client->getWriter().write(command.m_data.begin(), size);
        client->getWriter().flush();
    }
}

void hkNetworkedDeterminismUtil::Server::synchronizeWithClients()
{
    // Wait for 4 bytes from each client.
    //
    // Todo: timeouts.

    for (int i = 0; i < m_clients.getSize(); i++)
    {
        // use time out ...
        char buffer[4];
        if (readFromSocket(m_clients[i], buffer, 4).isFailure())
        {
            m_clients[i]->close();
            m_clients.removeAtAndCopy(i);
            i--;
        }
    }
}

void hkNetworkedDeterminismUtil::Client::sendSynchronizationBytes()
{
    char buffer[4];
    writeToSocket(m_socket, buffer, 4);

    // handle socket closing ..


}

void hkNetworkedDeterminismUtil::Client::processCommands(hkNetworkedDeterminismUtil::Command::Type expectedCommandType)
{
    hkArray<hkUint8> buffer;
    buffer.setSize(4);

    // Read command type
    /* hkResult socketResult =*/ readFromSocket(m_socket, buffer.begin(), 4);
    HK_ON_DEBUG( int type = *(int*)(buffer.begin()) );

    HK_ASSERT(0xad903131, expectedCommandType == type, "Unexpected command type.");

    // Read data size
    HK_ON_DEBUG( hkResult socketResult = ) readFromSocket(m_socket, buffer.begin(), 4);
    int dataSize = *(int*)(buffer.begin());

    // Read actual data
    //
    buffer.setSize(dataSize);
    HK_ON_DEBUG( socketResult = ) readFromSocket(m_socket, buffer.begin(), dataSize);
    HK_ON_DEBUG( socketResult = socketResult ); // silence compiler warning
}



hkNetworkedDeterminismUtil::ControlCommand::ControlCommand(_In_reads_bytes_(size) const void* buffer, int size)
{
    m_type = TYPE_CONTROL;
    m_data.setSize( size );
    hkString::memCpy(m_data.begin(), buffer, size);
}


hkNetworkedDeterminismUtil::DeterminismDataCommand::DeterminismDataCommand(_In_reads_bytes_(bufferSize) const char* buffer, int bufferSize)
{
    m_type = TYPE_DETERMINISM_DATA;

    m_data.setSize(bufferSize);
    hkString::memCpy(m_data.begin(), buffer, bufferSize);
}


void hkNetworkedDeterminismUtil::startStepDemoImpl(ControlCommand& controlCommand)
{
    hkCheckDeterminismUtil& util = hkCheckDeterminismUtil::getInstance();
    if (m_server)
    {
        // Cache control command
        m_controlCommand = controlCommand;

        util.startWriteMode();
    }
    if (m_client)
    {
        // Receive control command
        char buffer[2048];
        readFromSocket(m_client->m_socket, buffer, 8);
        HK_ASSERT(0xad903252, ((int*)buffer)[0] == Command::TYPE_CONTROL, "Unexpected command.");
        int dataSize = ((int*)buffer)[1];
        controlCommand.m_data.setSize(dataSize);
        readFromSocket(m_client->m_socket, controlCommand.m_data.begin(), dataSize);

        // Receive determinism data command
        readFromSocket(m_client->m_socket, buffer, 8);
        HK_ASSERT(0xad903252, ((int*)buffer)[0] == Command::TYPE_DETERMINISM_DATA, "Unexpected command.");
        dataSize = ((int*)buffer)[1];

        // Populate hkCheckDeterminismUtil in write mode.
        util.startWriteMode();
        while (dataSize > 0)
        {
            int thisSize = hkMath::min2( dataSize, 2048 );
            readFromSocket( m_client->m_socket, buffer, thisSize );
            util.m_memoryTrack->write( buffer, thisSize );
            dataSize -= thisSize;

        }

        // Set member m_memoryTrack to null it so that it doesn't get deleted in finish().
        hkMemoryTrack* memoryTrack = util.m_memoryTrack;
        util.m_memoryTrack = HK_NULL;
        util.finish();

        // Start comparing mode.
        util.m_memoryTrack = memoryTrack;
        util.startCheckMode(HK_NULL);
    }

//  const bool isPrimaryThread = true;
//  hkCheckDeterminismUtil::getInstance().workerThreadStartFrame(isPrimaryThread);
}

void hkNetworkedDeterminismUtil::endStepDemoImpl()
{
//  hkCheckDeterminismUtil::workerThreadFinishFrame();

    if (m_server)
    {
        // check for new clients
        m_server->pollForNewClients();

        // send cached control to all clients
        m_server->sendCommand(m_controlCommand);

        // Send determinism data
        DeterminismDataCommand determinismDataCommand(0, 0);
        int dataSize = hkCheckDeterminismUtil::getInstance().m_memoryTrack->getSize();
        determinismDataCommand.m_data.setSize(dataSize);

        HK_ASSERT(0xad903244, hkCheckDeterminismUtil::getInstance().m_memoryTrack != HK_NULL, "The hkNetworkedDeterminismUtil assumes the determinsm check util to use a hkMemoryTrack.");
        int offset = 0;

        hkArray<hkUint8*>& sectors = hkCheckDeterminismUtil::getInstance().m_memoryTrack->m_sectors;
        int sectorSize = hkCheckDeterminismUtil::getInstance().m_memoryTrack->m_numBytesPerSector;
        for (int si = 0; si < sectors.getSize()-1; si++)
        {
            hkString::memCpy(determinismDataCommand.m_data.begin()+offset, sectors[si], sectorSize);
            offset += sectorSize;
        }
        if (sectors.getSize())
        {
            hkString::memCpy(determinismDataCommand.m_data.begin()+offset, sectors.back(), hkCheckDeterminismUtil::getInstance().m_memoryTrack->m_numBytesLastSector);
            offset += hkCheckDeterminismUtil::getInstance().m_memoryTrack->m_numBytesLastSector;
        }
        HK_ASSERT(0xad903246, offset == dataSize, "Data written doesn't match hkMemoryTrack's size.");

        hkCheckDeterminismUtil::getInstance().finish();

        m_server->sendCommand(determinismDataCommand);

    }

    if (m_client)
    {
        hkCheckDeterminismUtil::getInstance().finish();
    }
}

bool hkNetworkedDeterminismUtil::tryToCreateClient()
{
    HK_ASSERT(0xad903093, ! m_client, "Client already created. To reconnect to a server destroy the existing client first.");

    hkSocket* socket = hkSocket::create();

    if (socket)
    {
        if (socket->connect(m_serverAddress.cString(), m_serverPort).isFailure())
        {
            delete socket;
            return false;
        }

        // Create client with the socket.
        m_client = new Client(socket);
        return true;
    }

    return false;
}

void hkNetworkedDeterminismUtil::createServer()
{
    int maxNumClients = 1;
    m_server = new Server(m_serverPort, maxNumClients);
}

hkResult hkNetworkedDeterminismUtil::readFromSocket(_Inout_ hkSocket* socket, _Out_writes_bytes_(size) void* buffer, int size)
{
    const int fullSize = size;
    while(size)
    {
        if (!socket->isOk()) { break; }
        size -= socket->getReader().read(hkAddByteOffset(buffer, fullSize-size), size);
    }

    hkResult result = size == 0 ? HK_SUCCESS : HK_FAILURE;
    HK_ASSERT(0xad930241, result.isSuccess(), "Socket broke.");
    return result;
}

hkResult hkNetworkedDeterminismUtil::writeToSocket(_Inout_ hkSocket* socket, _In_reads_bytes_(size) const void* buffer, int size)
{
    int sizeWritten = socket->getWriter().write(buffer, size);
    HK_ASSERT(0xad90324a, size == sizeWritten, "Not all data written to socket. Communication will now hang.");

    return size == sizeWritten ? HK_SUCCESS : HK_FAILURE;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
