// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/hkBaseTypes.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/DebugUtil/DeterminismUtil/hkCheckDeterminismUtil.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/System/Io/Writer/Crc/hkCrcStreamWriter.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Thread/JobQueue/hkJobQueue.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#ifdef HK_PLATFORM_WIN32
#include <stdio.h>
#include <time.h>
#include <Common/Base/Fwd/hkwindows.h>
#endif

#ifdef LODEPNG_H
#include <vector>
#endif

#define DEBUG_LOG_IDENTIFIER "common.base.determinism.default"
#include <Common/Base/System/Log/hkLog.hxx>


// Markers used in the shared stream to indicate the beginning and end of the data collected from all the jobs in the frame
#define CDU_JOBS_START_MARKER   0xadadadad
#define CDU_JOBS_END_MARKER     0xbdbdbdbd


HK_EXPORT_COMMON HK_THREAD_LOCAL( hkCheckDeterminismUtil::ThreadData* ) hkCheckDeterminismUtil_ThreadData;
hkCheckDeterminismUtil* hkCheckDeterminismUtil::s_instance;

int hkCheckDeterminismUtil_id;
hkReal* hkCheckDeterminismUtil_reference;
hkReal* hkCheckDeterminismUtil_object;
int hkCheckDeterminismUtil_size;
hkReal* hkCheckDeterminismUtil_crcObject;

hkUint32 hkCheckDeterminismUtil::s_randomSeed = 5;

hkUint32 hkCheckDeterminismUtil::s_heapAllocScrubValueWrite = 0x7ffa110c; // ALLOC
hkUint32 hkCheckDeterminismUtil::s_heapFreeScrubValueWrite = 0x7fffefef; // FREE
hkUint32 hkCheckDeterminismUtil::s_heapAllocScrubValueCheck = 0x0; // ALLOC
hkUint32 hkCheckDeterminismUtil::s_heapFreeScrubValueCheck = 0x0; // FREE
hkUint32 hkCheckDeterminismUtil::s_stackScrubValueWrite = 0x7ffdadad;
hkUint32 hkCheckDeterminismUtil::s_stackScrubValueCheck = 0x0;


hkCheckDeterminismUtil::hkCheckDeterminismUtil()
{
    m_inSingleThreadedCode = true;
    m_shared = new hkCriticalSection();
    m_threadDataLock = new hkCriticalSection();

    m_sharedInputStream = HK_NULL;
    m_sharedOutputStream = HK_NULL;
    m_primaryWorkerThreadInputStream = HK_NULL;
    m_primaryWorkerThreadOutputStream = HK_NULL;
    m_memoryTrack = HK_NULL;

    // Assign a thread data to this thread
    allocateThreadData();

    m_frame = 0;

    // Debug flag
    m_writingStFromWorker = false;
    m_mode = MODE_IDLE;
    m_checkFailMode = CHECK_FAIL_MODE_BREAK;
    m_duplicateFuidMode = DUPLICATE_FUID_MODE_BREAK;

#ifdef HK_PLATFORM_WIN32
    // Disable output buffering for determinism error reports.
    setbuf(stdout, NULL);
#endif

    // Job delay
    m_delayJobs = false;
    m_delayCounter = 1024 * 256;
    if (m_delayJobs)
    {
#ifdef LODEPNG_H
        {
            std::vector<unsigned char> image;
            unsigned width, height;
            lodepng::decode(image, width, height, "pattern.png");

            m_delayJobSeed.setSize(image.size());
            for (unsigned int i = 0; i < image.size(); ++i)
            {
                m_delayJobSeed[i] = image[i];
            }
        }
#endif

        m_delayJobs = !m_delayJobSeed.isEmpty();
    }

    // Thread tracking
    m_enableThreadTracker = false;
    m_threadTracker.setSize(32);
    m_maxTheadId = 0;
}


hkCheckDeterminismUtil::~hkCheckDeterminismUtil()
{
    ThreadData& threadData = getThreadData();
    HK_ASSERT(0xad876dda, m_sharedInputStream == threadData.m_inputStream, "Upon destruction, the thread-local streams are expected to be set to the shared streams (i.e. working in single-threaded mode).");
    HK_ASSERT(0xad876dda, m_sharedOutputStream == threadData.m_outputStream, "Upon destruction, the thread-local streams are expected to be set to the shared streams (i.e. working in single-threaded mode).");

    finish();

    delete m_shared;
    threadData.reset();

    for( int i=0; i<m_threadData.getSize(); i++ )
    {
        delete m_threadData[i];
    }
    delete m_threadDataLock;
    HK_THREAD_LOCAL_SET( hkCheckDeterminismUtil_ThreadData, HK_NULL );

    delete m_memoryTrack;
}

hkCheckDeterminismUtil::Fuid& hkCheckDeterminismUtil::Fuid::getZeroFuid()
{
    static Fuid fuid;
    fuid.m_0 = hkUint32(-1);
    fuid.m_jobPackedId = hkUint16(-1);
    fuid.m_2 = 0;
    fuid.m_3 = 0;
    
    return fuid;
}

hkCheckDeterminismUtil::Fuid& hkCheckDeterminismUtil::Fuid::getCanceledFuid()
{
    static Fuid fuid;
    fuid.m_0 = hkUint32(-2);
    fuid.m_jobPackedId = hkUint16(-1);
    fuid.m_2 = 0;
    fuid.m_3 = 0;
    
    return fuid;
}

namespace
{
#if defined (HK_ENABLE_DETERMINISM_CHECKS)
    static void makeFilename(_In_z_ const char* filename, bool stampFilename, hkStringPtr& ptrOut)
    {
        hkStringBuf buf;
#if defined(HK_PLATFORM_WIN32) && !defined(HK_PLATFORM_WINRT)
        if (stampFilename)
        {
            char szFileName[MAX_PATH];
            GetModuleFileNameA(NULL, szFileName, MAX_PATH);
            buf.set(szFileName);
            buf.pathNormalize();
            buf.pathBasename();
            buf.appendPrintf("_%d_%d.bin", hkInt32(GetCurrentProcessId()), hkUint32(::time(0)));
        }
        else
#endif
        {
            buf.printf("%s", filename);
        }

        ptrOut = buf.cString();
    }
#endif // defined (HK_ENABLE_DETERMINISM_CHECKS)
}

void hkCheckDeterminismUtil::startWriteMode(bool stampFilename, _In_opt_z_ const char* filename)
{
#if defined (HK_ENABLE_DETERMINISM_CHECKS)
    HK_ASSERT(0xaf36affe, !m_sharedInputStream, "You cannot switch to WRITE mode without calling finish() first.");

    // set scrub values for write
    hkMemorySystem& memorySystem = hkMemorySystem::getInstance();
    memorySystem.setHeapScrubValues(s_heapAllocScrubValueWrite, s_heapFreeScrubValueWrite);

    if (!filename)
    {
        HK_ASSERT_NO_MSG(0xf0ed3dfe, !m_memoryTrack);
        m_memoryTrack = new hkMemoryTrack;
        m_sharedOutputStream = new hkOstream(m_memoryTrack);
    }
    else
    {
        makeFilename(filename, stampFilename, m_filename);
        Log_Info(m_filename);
        m_sharedOutputStream = new hkOstream(m_filename.cString());
    }

    HK_ASSERT(0xaf36affd, m_sharedOutputStream->isOk(), "Output file could not be opened.");

    m_mode = MODE_WRITE;

    ThreadData& threadData = getThreadData();
    threadData.m_inputStream = m_sharedInputStream;
    threadData.m_outputStream = m_sharedOutputStream;

    m_frame = 0;
#endif
}


void hkCheckDeterminismUtil::startCheckMode(_In_opt_z_ const char* filename)
{
#if defined (HK_ENABLE_DETERMINISM_CHECKS)
    HK_ASSERT(0xaf36affe, !m_sharedOutputStream, "You cannot switch to READ mode without calling finish() first.");

    // set scrub values for check
    hkMemorySystem& memorySystem = hkMemorySystem::getInstance();
    memorySystem.setHeapScrubValues(s_heapAllocScrubValueCheck, s_heapFreeScrubValueCheck);

    if (m_memoryTrack)
    {
        m_sharedInputStream = new hkIstream(m_memoryTrack);
    }
    else
    {
        m_sharedInputStream = new hkIstream(filename ? filename : m_filename.cString());
    }


    if (!m_sharedInputStream->isOk())
    {
        HK_ASSERT(0xaf36affe, false, "Input file not found.");
        finish();
    }
    else
    {
        m_mode = MODE_COMPARE;
    }

    ThreadData& threadData = getThreadData();
    threadData.m_inputStream = m_sharedInputStream;
    threadData.m_outputStream = m_sharedOutputStream;

    m_frame = 0;
#endif
}

struct HK_EXPORT_COMMON IntPair
{
    int m_id;
    int m_size;
    inline hkBool operator<(const IntPair& other) const { return m_size < other.m_size; }
};

void hkCheckDeterminismUtil::finish()
{
#if defined (HK_ENABLE_DETERMINISM_CHECKS)
    if (m_mode == MODE_WRITE)
    {
        HK_ASSERT_NO_MSG(0x11583010, m_sharedOutputStream != nullptr );

        if( m_sharedOutputStream )
        {
            m_sharedOutputStream->flush();
        }

        delete m_sharedOutputStream;
        m_sharedOutputStream = HK_NULL;

        // do not touch m_memoryTrack

        #if defined(HK_DETERMINISM_CHECK_SIZES)
        {
            hkArray<IntPair> pairs;
            for( hkPointerMap<int, int>::Iterator i = m_sizePerId.getIterator(); m_sizePerId.isValid( i ); i = m_sizePerId.getNext( i ) )
            {
                IntPair& p = pairs.expandOne();
                p.m_id = m_sizePerId.getKey( i );
                p.m_size = m_sizePerId.getValue( i );
            }
            hkSort( pairs.begin(), pairs.getSize() );
            for( int k = pairs.getSize() - 1; k >= 0; k-- )
            {
                IntPair& p = pairs[k];
                Log_Info( "hkCheckDeterminismUtil::id {:x} uses {} bytes", p.m_id, p.m_size );
            }

        }
        #endif
    }
    else if (m_mode == MODE_COMPARE)
    {
        HK_ASSERT_NO_MSG(0x3f18e015, m_sharedInputStream != nullptr );

        // check whether we reached the end of the file
        {
            char tmpBuffer[4];  m_sharedInputStream->read( tmpBuffer, 1 );
            HK_ASSERT_NO_MSG( 0xad87b754, !m_sharedInputStream->isOk() );
        }

        delete m_sharedInputStream;
        m_sharedInputStream = HK_NULL;

        if( m_memoryTrack != HK_NULL )
        {
            delete m_memoryTrack;
            m_memoryTrack = HK_NULL;
        }

        #ifdef HK_PLATFORM_WIN32
        hkFileSystem::getInstance().remove( m_filename.cString() );
        #endif
    }

    ThreadData& threadData = getThreadData();
    threadData.m_inputStream = HK_NULL;
    threadData.m_outputStream = HK_NULL;

    m_mode = MODE_IDLE;
#endif
}


namespace
{

    static void initializeBreakpoint(struct Breakpoint& b);

    struct Breakpoint
    {
        hkUint64                        frame;
        hkCheckDeterminismUtil::Fuid    fuid;
        hkUint64                        checkIndex;
        int                             size;
        int                             dataErrorIndex;
        hkUint8                         d[1024];
        hkUint8                         od[1024];

        Breakpoint() {}

        static Breakpoint s_b;
        static bool s_initialized;

        static const Breakpoint& get()
        {
            if (!s_initialized)
            {
                hkString::memSet(&s_b, 0, sizeof(s_b));
                s_b.fuid = hkCheckDeterminismUtil::Fuid::getZeroFuid();

                initializeBreakpoint(s_b);
                //s_initialized = true; // Should this be set??
            }

            return s_b;
        }

        static void print(hkStringBuf& str, hkUint64 frame, _In_reads_(size) const char* sourceObject, _In_reads_(size) const char* oldObject, int size, int index, _In_reads_(numThreads + 1) const hkCheckDeterminismUtil::Fuid* threadFuids, int numThreads)
        {
            hkCheckDeterminismUtil::Fuid fuid = hkCheckDeterminismUtil::getCurrentJobFuid();
            hkUint64 checkIndex = hkCheckDeterminismUtil::getCurrentCheckIndex();

            str.appendPrintf("#if 1\n{\n");
            str.appendPrintf("\n\n////// Determinism Breakpoint START //////\n");
            str.appendPrintf("// place this code into hkCheckDeterminismUtil.cpp, initializeBreakpoint()\n");
            str.appendPrintf("#define HK_DETERMINISM_ENABLE_BREAKPOINT\n");
            str.appendPrintf("b.frame = %llu;\n", frame);
            str.appendPrintf("b.fuid.m_0= %u; b.fuid.m_jobPackedId= %u; b.fuid.m_2= %u; b.fuid.m_3= %u; b.fuid.m_4= %u; // <%u, %u, %u, %u, %u>\n",
                (hkUint32)fuid.m_0, (hkUint32)fuid.getPackedJobId(), (hkUint32)fuid.m_2, (hkUint32)fuid.m_3, (hkUint32)fuid.m_4,
                (hkUint32)fuid.m_0, (hkUint32)fuid.getPackedJobId(), (hkUint32)fuid.m_2, (hkUint32)fuid.m_3, (hkUint32)fuid.m_4);
            str.appendPrintf("b.checkIndex = %u;\n", hkUint32(checkIndex));
            str.appendPrintf("b.size = %d;\n", size);
            str.appendPrintf("b.dataErrorIndex = %d;\n", index);

            str.appendPrintf("// New data\n");
            for (int i = 0; i < size; ++i)
            {
                str.appendPrintf("b.d[%d]=%d;", i, hkUint8(sourceObject[i]));
                if ((i > 0) && ((i % 32) == 0)) str.appendPrintf("\n");
            }
            str.appendPrintf("\n");

            str.appendPrintf("// Old data\n");
            for (int i = 0; i < size; ++i)
            {
                str.appendPrintf("b.od[%d]=%d;", i, hkUint8(oldObject[i]));
                if ((i > 0) && ((i % 32) == 0)) str.appendPrintf("\n");
            }
            str.appendPrintf("\n");

        if (numThreads > 0)
            {
                hkCheckDeterminismUtil::Fuid invalidFuid = hkCheckDeterminismUtil::Fuid::getZeroFuid();
                invalidFuid.m_0 = 0;

            const hkCheckDeterminismUtil::ThreadData* td = HK_THREAD_LOCAL_GET( hkCheckDeterminismUtil_ThreadData );
                str.appendPrintf("/* Thread Fuids\n");
            for (int i=0; i<=numThreads; ++i)
                {
                    if (threadFuids[i] != invalidFuid)
                    {
                    str.appendPrintf("[ %d%s] <%u, %u, %u, %u, %u>\n", i, td->m_index == i ? "*": " ",
                            (hkUint32)threadFuids[i].m_0, (hkUint32)threadFuids[i].getPackedJobId(), (hkUint32)threadFuids[i].m_2, (hkUint32)threadFuids[i].m_3, (hkUint32)threadFuids[i].m_4);
                    }
                }
                str.appendPrintf("*/\n");
            }

            str.appendPrintf("////// Determinism Breakpoint END //////\n");
            str.appendPrintf("#endif\n\n");
        }
    };

    Breakpoint Breakpoint::s_b;
    bool Breakpoint::s_initialized = false;

    // When there is a set breakpoint, its generated could should go into this function. Otherwise, it should be totally empty.
    static void initializeBreakpoint(struct Breakpoint& b)
    {
    }


    static void checkBreakpoint(const hkUint64& frame, _In_ const void* object)
    {
#ifdef HK_DETERMINISM_ENABLE_BREAKPOINT
        const Breakpoint& bkpt = Breakpoint::get();

        hkCheckDeterminismUtil::Fuid fuid = hkCheckDeterminismUtil::getCurrentJobFuid();
        hkUint32 checkIndex = hkCheckDeterminismUtil::getCurrentCheckIndex();

        //printf("frame:%d ", frame);
        //printf("fuid:%u, %u, %u, %u, %u ", hkUint32(fuid.m_0), hkUint32(fuid.m_1), hkUint32(fuid.m_2), hkUint32(fuid.m_3), hkUint32(fuid.m_4));
        //printf("ci:%u\n", hkUint32(checkIndex));

        if (frame == bkpt.frame)
        {
            if (fuid == bkpt.fuid)
            {
                if (checkIndex == bkpt.checkIndex)
                {
                    // some variables helping debugging
                    const hkReal* of = (const hkReal*)object; (void)of;
                    const hkReal* cf = (const hkReal*)bkpt.d; (void)cf;
                    const void*const* oh = (const void*const*)of; (void)oh;
                    const void*const* ch = (const void*const*)cf; (void)ch;

                    Log_Info("\nDeterminism breakpoint reached");
                }
                else
                {
                    if (checkIndex + 1 == bkpt.checkIndex)
                    {
                        Log_Info("\nDeterminism pre-breakpoint reached");
                    }
                }
            }
        }
#endif
    }


} // anonymous namespace


bool hkCheckDeterminismUtil::isNearBreakpoint(hkUint64 offset)
{
#ifdef HK_DETERMINISM_ENABLE_BREAKPOINT
    const Breakpoint& bkpt = Breakpoint::get();
    hkCheckDeterminismUtil::Fuid fuid = getCurrentJobFuid();
    hkUint32 checkIndex = hkCheckDeterminismUtil::getCurrentCheckIndex();

    if (m_frame == bkpt.frame)
    {
        if (fuid == bkpt.fuid)
        {
            if (checkIndex <= bkpt.checkIndex && checkIndex + offset >= bkpt.checkIndex)
            {
                return true;
            }
        }
    }

    return false;
#else
    return false;
#endif
}

void hkCheckDeterminismUtil::randomizeMemoryImpl(_Inout_updates_bytes_(numBytes) void* data, int numBytes)
{
    // this function is not deterministic when used with multiple threads, however the
    // goal is to randomize memory, so this is OK
    int seed = hkAtomic::exchangeAdd32(&s_randomSeed, 101);
    hkPseudoRandomGenerator random(seed);
    random.randomizeMemory(data, numBytes);
}

void hkCheckDeterminismUtil::checkImpl(int id, _In_reads_bytes_(size) const void* object, int size, _In_reads_opt_(_Inexpressible_()) const int* excluded, _Outptr_opt_ const void** referenceObject)
{
    if (!size)
    {
        return;
    }

    const char* sourceObject = (const char*)object;
    ThreadData& threadData = getThreadData();
    hkOstream* outputStream = threadData.m_outputStream;
    if (outputStream)
    {
#if defined(HK_DETERMINISM_CHECK_SIZES)
        int currentSize = m_sizePerId.getWithDefault(id, 0);
        currentSize += size;
        m_sizePerId.insert(id, currentSize);
#endif
        outputStream->write(sourceObject, size);

        HK_ASSERT_NO_MSG(0xf0323446, outputStream->isOk());

        checkBreakpoint(m_frame, object);

        bumpCurrentCheckIndex();
        return;
    }

    // Setup the exclusion variable.
    int nextExludedOffsetIndex = (excluded == HK_NULL ? -1 : 0);
    int nextExcludedOffset = (excluded == HK_NULL ? -1 : excluded[0]);

    hkIstream* inputStream = threadData.m_inputStream;
    if (inputStream == HK_NULL)
    {
        HK_ASSERT(0xad7655dd, false, "Neither stream exists. Likely a missing call to registerAndStartJob()");
        bumpCurrentCheckIndex();
        return;
    }

    hkLocalBuffer<char> readBuffer(size);
    HK_ON_DEBUG(int debug_numRead = )inputStream->read(readBuffer.begin(), size);
    // Note that we don't check ::isOk() here, since we might be at the end of the file.
    HK_ASSERT_NO_MSG(0xf0323445, debug_numRead == size);

    // some variables helping debugging
    static const hkReal* of; of = (const hkReal*)object;    
    static const hkReal* cf; cf = (const hkReal*)readBuffer.begin();
    const void*const* oh = (const void*const*)object; (void)oh;
    const void*const* ch = (const void*const*)readBuffer.begin();
    if (referenceObject) *referenceObject = ch; // export the memory to my caller, this is only useful if one of the errors trigger later
    for (int i = 0; i < size; i++)
    {
        // Compare the bytes.
        if (i != nextExcludedOffset)
        {
            if ((sourceObject[i] != readBuffer[i]))
            {
                hkCheckDeterminismUtil_id = id;
                hkCheckDeterminismUtil_object = (hkReal*)object;
                hkCheckDeterminismUtil_reference = (hkReal*)(ch);
                hkCheckDeterminismUtil_size = size;

                hkStringBuf text;
                text.printf("\nDeterminism check failed: size %d, i %d, obj 0x%p, ref 0x%p, obj[i] 0x%02X, ref[i] 0x%02X",
                    size, i, (void*)sourceObject, (void*)readBuffer.begin(), (hkUint8)sourceObject[i], (hkUint8)readBuffer[i]);

                if (m_checkFailMode == CHECK_FAIL_MODE_BREAK)
                {
                    // Look ahead in the input stream. Can be useful if you use a string as a marker.
                    hkLocalArray<char> futureBuffer(1024);
                    int numRead = inputStream->read(futureBuffer.begin(), 1024);
                    futureBuffer.setSize(numRead);

                    hkStringBuf bkptStr;
                    Breakpoint::print(bkptStr, m_frame, sourceObject, readBuffer.begin(), size, i, m_threadTracker.begin(), m_maxTheadId);
                    hkError::messageError(id, bkptStr.cString(), HK_CURRENT_FILE, __LINE__);

                    HK_ERROR(id, text.cString());
                }
                else
                {
                    Log_Info(text);

                    // Copy reference data read into a per-thread buffer so it can be accessed later via
                    // getLastReferenceData()
                    threadData.m_referenceData.setSize(size);
                    hkString::memCpy(threadData.m_referenceData.begin(), readBuffer.begin(), size);
                    threadData.m_didLastCheckFail = true;

                    break;
                }
            }
        }
        else
        {
            // Skip the required number of excluded bytes.
            i += excluded[nextExludedOffsetIndex + 1] - 1;
            // Advance to the next exclusion.
            nextExludedOffsetIndex += 2;
            // Set the next excluded offset.
            nextExcludedOffset = excluded[nextExludedOffsetIndex];
            // Make sure the excluded array is sorted by offset.
            HK_ASSERT(0xef6e13a6, (nextExcludedOffset == -1) || (nextExcludedOffset >= i), "the 'excluded' array has to be sorted by offset.");
        }
    }

    bumpCurrentCheckIndex();
}

void hkCheckDeterminismUtil::checkCrcImpl(int id, _In_reads_bytes_(size) const void* object, int size, _In_reads_opt_(_Inexpressible_()) const int* excluded)
{
    hkCheckDeterminismUtil_crcObject = (hkReal*)object;

    hkCrc32StreamWriter crcWriter;
    if (!excluded)
    {
        crcWriter.write(object, size);
    }
    else
    {
        // Copy the original object and zero the excluded parts before calculating the CRC
        hkUint8* objectCopy = hkAllocateStack<hkUint8>(size, "checkMtCrc");
        hkString::memCpy(objectCopy, object, size);
        for (int i = 0; excluded[i] != -1; i += 2)
        {
            hkString::memSet(objectCopy + excluded[i], 0, excluded[i + 1]);
        }

        crcWriter.write(objectCopy, size);
        hkDeallocateStack(objectCopy, size);
    }

    hkUint32 crc = crcWriter.getCrc();
    checkImpl(id, &crc, sizeof(hkUint32));
}


//////////////////////////////////////////////////////////////////////////
//
// Registration functions used at the beginning and end of each hkpDynamicsJob, and multi-threading registration functions.
//
//////////////////////////////////////////////////////////////////////////

hkCheckDeterminismUtil::ThreadData* hkCheckDeterminismUtil::allocateThreadData()
{
    HK_ASSERT_NO_MSG(0x74790c98, HK_THREAD_LOCAL_GET( hkCheckDeterminismUtil_ThreadData ) == HK_NULL );

    hkCriticalSectionLock lock(m_threadDataLock);

    ThreadData* td = HK_NULL;
    for( int i = 0; i < m_threadData.getSize(); ++i )
    {
        if( !m_threadData[i]->m_inUse )
        {
            td = m_threadData[i];
            break;
        }
    }
    if( !td )
    {
        td = new ThreadData();
        td->m_index = m_threadData.getSize();
        m_threadData.pushBack( td );
    }

    td->reset();
    td->m_inUse = true;

    HK_THREAD_LOCAL_SET( hkCheckDeterminismUtil_ThreadData, td );

    return td;
}

void hkCheckDeterminismUtil::freeThreadData()
{
    hkCheckDeterminismUtil::ThreadData* td = HK_THREAD_LOCAL_GET( hkCheckDeterminismUtil_ThreadData );
    HK_ASSERT_NO_MSG( 0x3dbbc60e, td );

    hkCriticalSectionLock lock( m_threadDataLock );

    td->reset();
    td->m_inUse = false;

    HK_THREAD_LOCAL_SET( hkCheckDeterminismUtil_ThreadData, HK_NULL );
}

void hkCheckDeterminismUtil::workerThreadStartFrameImpl(hkBool isPrimaryWorkerThread)
{
    ThreadData& threadData = getThreadData();
    {
        hkUlong tmp = isPrimaryWorkerThread;
        threadData.m_isPrimaryWorkerThread = reinterpret_cast<void*&>(tmp);
    }

    if (isPrimaryWorkerThread)
    {
        m_frame++;

        HK_ASSERT_NO_MSG(0XAD876716, !m_writingStFromWorker);

        HK_ASSERT_NO_MSG(0xad8766dd, m_primaryWorkerThreadInputStream == HK_NULL);
        HK_ASSERT_NO_MSG(0xad8766dd, m_primaryWorkerThreadOutputStream == HK_NULL);

        if (getInstance().m_mode == hkCheckDeterminismUtil::MODE_COMPARE)
        {
            extractRegisteredJobsImpl();
        }

        // Create streams for single-threaded sections.
        registerAndStartJobImpl(Fuid::getZeroFuid());

        m_primaryWorkerThreadInputStream = threadData.m_inputStream;
        m_primaryWorkerThreadOutputStream = threadData.m_outputStream;

    }
    else
    {
        threadData.m_inputStream = HK_NULL;
        threadData.m_outputStream = HK_NULL;
    }
}

void hkCheckDeterminismUtil::workerThreadFinishFrameImpl()
{
    ThreadData& threadData = getThreadData();
    if (threadData.m_isPrimaryWorkerThread)
    {
        finishJob(Fuid::getZeroFuid(), false);

        m_primaryWorkerThreadInputStream = HK_NULL;
        m_primaryWorkerThreadOutputStream = HK_NULL;

        HK_ASSERT_NO_MSG(0XAD876, !m_writingStFromWorker);

        if (hkCheckDeterminismUtil::getInstance().m_mode == hkCheckDeterminismUtil::MODE_WRITE)
        {
            hkCheckDeterminismUtil::getInstance().combineRegisteredJobs();
        }
        if (hkCheckDeterminismUtil::getInstance().m_mode == hkCheckDeterminismUtil::MODE_COMPARE)
        {
            hkCheckDeterminismUtil::getInstance().clearRegisteredJobs();
        }
    }
}


void hkCheckDeterminismUtil::delayJob(const Fuid& id, bool start) const
{
    if (m_delayJobs)
    {
        hkInt32 seed = hkInt32(m_frame) + hkInt32(start) + hkInt32(id.m_0) + hkInt32(id.getPackedJobId()) + hkInt32(id.m_2) + hkInt32(id.m_3) + hkInt32(id.m_4);
        unsigned int pixel = (seed * 8) % (m_delayJobSeed.getSize() / 4);
        const unsigned char* data = &m_delayJobSeed[pixel * 4];
        hkUint32 cnt = data[0] ? m_delayCounter : 0;

        if (cnt)
        {
            //printf("delay fuid:%u, %u, %u, %u, %u \n", hkUint32(id.m_0), hkUint32(id.m_1), hkUint32(id.m_2), hkUint32(id.m_3), hkUint32(id.m_4));
            for (hkUint32 i = 0; i < cnt; ++i)
            {
                setCurrentCheckIndex(i == 0 ? 0 : -1);
            }
            setCurrentCheckIndex(0);
        }
        else
        {
            //printf("no delay fuid:%u, %u, %u, %u, %u \n", hkUint32(id.m_0), hkUint32(id.m_1), hkUint32(id.m_2), hkUint32(id.m_3), hkUint32(id.m_4));
        }
    }
}


void hkCheckDeterminismUtil::registerAndStartJobImpl(Fuid& jobFuid)
{
#if defined(HK_ENABLE_DETERMINISM_CHECKS)
    ThreadData& threadData = getThreadData();

    m_shared->enter();

    if (threadData.m_isPrimaryWorkerThread)
    {
        if (jobFuid == Fuid::getZeroFuid())
        {
            HK_ASSERT_NO_MSG(0XAD98666D, threadData.m_inputStream == m_sharedInputStream);
            HK_ASSERT_NO_MSG(0XAD98666D, threadData.m_outputStream == m_sharedOutputStream);

            HK_ASSERT_NO_MSG(0XAD876, m_writingStFromWorker == false);
            m_writingStFromWorker = true;

        }
        else
        {
            HK_ASSERT(0XAD98666D, threadData.m_inputStream == m_primaryWorkerThreadInputStream, "Did you forget to call workerThreadStartFrame()?");
            HK_ASSERT(0XAD98666D, threadData.m_outputStream == m_primaryWorkerThreadOutputStream, "Did you forget to call workerThreadStartFrame()?");

            HK_ASSERT_NO_MSG(0XAD876, m_writingStFromWorker);
            m_writingStFromWorker = false;
        }
    }
    else
    {
        HK_ASSERT(0XAD98666D, threadData.m_inputStream == HK_NULL, "Stream still allocated. Did you miss a call to finishJob() workerThreadFinishFrame()?");
        HK_ASSERT(0XAD98666D, threadData.m_outputStream == HK_NULL, "Stream still allocated. Did you miss a call to finishJob() or workerThreadFinishFrame()?");
    }

    setCurrentJobFuid(jobFuid);
    setCurrentCheckIndex(0);

    // add/find entry to/in the shared list
    hkHashMap<Fuid, JobInfo>::Iterator iter = m_registeredJobs.findKey(jobFuid);

    // create local stream
    if (m_mode == MODE_WRITE)
    {
        // make sure it is unique
        hkInt64 openCount = 0;
        if (m_registeredJobs.isValid(iter))
        {
            if (m_duplicateFuidMode == DUPLICATE_FUID_MODE_BREAK)
            {
                HK_BREAKPOINT(0xad9877da); //"Internal error: Fuid is not frame-unique"
            }
            else
            {
                // Free the memory track of the existing job
                JobInfo& existingInfo = m_registeredJobs.getValue(iter);
                openCount = existingInfo.m_openCount;
                delete existingInfo.m_data;
            }
        }
        JobInfo info;
        info.m_jobFuid = jobFuid;
        info.m_openCount = openCount + 1;
        info.m_data = new hkMemoryTrack;
        threadData.m_outputStream = new hkOstream(info.m_data);
        m_registeredJobs.insert(jobFuid, info);
    }
    else
    {
        HK_ASSERT(0xad9877da, m_registeredJobs.isValid(iter), "Internal error: Fuid is not found");
        JobInfo& info = m_registeredJobs.getValue(iter);
        
        HK_ASSERT(0xad9877da, info.m_openCount == 0 || m_duplicateFuidMode == DUPLICATE_FUID_MODE_REPLAY, "Internal error: Fuid is not frame-unique");
        info.m_openCount += 1;
        if (info.m_jobFuid == Fuid::getZeroFuid())
        {
            // This is the special startFrame/endFrame FUID. This only occurs once per frame, and needs to read the real input stream
            threadData.m_inputStream = new hkIstream(info.m_data);
        }
        else
        {
            // Make a copy of the input track, in case DUPLICATE_FUID_MODE_REPLAY causes two threads to process an FUID at the same time.
            threadData.m_uniqueInputTrack.m_numBytesPerSector = info.m_data->m_numBytesPerSector;
            threadData.m_uniqueInputTrack.m_numBytesLastSector = info.m_data->m_numBytesLastSector;
            threadData.m_uniqueInputTrack.m_numBytesRead = 0;
            threadData.m_uniqueInputTrack.m_numSectorsUnloaded = info.m_data->m_numSectorsUnloaded;
            threadData.m_uniqueInputTrack.m_sectors.setDataUserFree(info.m_data->m_sectors.begin(), info.m_data->m_sectors.getSize(), info.m_data->m_sectors.getCapacity());

            threadData.m_inputStream = new hkIstream(&threadData.m_uniqueInputTrack);
        }
    }

    hkCheckDeterminismUtil::checkMt(0xf0000000, 0xacacacacul);

    m_shared->leave();

    delayJob(jobFuid, true);
#endif
}

void hkCheckDeterminismUtil::finishJobImpl(Fuid& jobFuid, hkBool skipCheck)
{
#if defined(HK_ENABLE_DETERMINISM_CHECKS)
    delayJob(jobFuid, false);
    ThreadData& threadData = getThreadData();

    m_shared->enter();
    // destroy local stream

    if (!skipCheck)
    {
        hkCheckDeterminismUtil::checkMt(0xf0000001, 0xbcbcbcbcul);
    }

    hkHashMap<Fuid, JobInfo>::Iterator iter = m_registeredJobs.findKey(jobFuid);
    HK_ASSERT(0xad986dd9, m_registeredJobs.isValid(iter), "Fuid not found");
    JobInfo& info = m_registeredJobs.getValue(iter);
    HK_ASSERT(0xad986dda, jobFuid == info.m_jobFuid, "Fuid inconsistency.");
    HK_ASSERT(0xad9877db, info.m_openCount == 1 || m_duplicateFuidMode == DUPLICATE_FUID_MODE_REPLAY, "Internal error: Fuid is not frame-unique");
    info.m_openCount -= 1;
    HK_ASSERT_NO_MSG(0xad9877dc, info.m_openCount >= 0);

    delete threadData.m_inputStream;
    delete threadData.m_outputStream;

    if (threadData.m_isPrimaryWorkerThread)
    {
        hkIstream*  inputStream = threadData.m_inputStream;
        hkOstream* outputStream = threadData.m_outputStream;
        if ((inputStream != m_primaryWorkerThreadInputStream) || (outputStream != m_primaryWorkerThreadOutputStream))
        {
            HK_ASSERT(0xad836433, getCurrentJobFuid() == jobFuid, "Job Fuid mismatch when finishing job.");
            HK_ASSERT(0xad876333, (inputStream != m_sharedInputStream) || (outputStream != m_sharedOutputStream), "Trying to finish a job, but all the jobs are finished");

            threadData.m_inputStream = m_primaryWorkerThreadInputStream;
            threadData.m_outputStream = m_primaryWorkerThreadOutputStream;

            HK_ASSERT_NO_MSG(0XAD876, m_writingStFromWorker == false);
            m_writingStFromWorker = true;

            setCurrentJobFuid(Fuid::getZeroFuid());
        }
        else
        {
            HK_ASSERT(0xad836433, jobFuid == Fuid::getZeroFuid(), "Special singleThreaded job expected.");
            HK_ASSERT(0xad836433, getCurrentJobFuid() == Fuid::getZeroFuid(), "Special singleThreaded job expected.");
            threadData.m_inputStream = m_sharedInputStream;
            threadData.m_outputStream = m_sharedOutputStream;

            HK_ASSERT_NO_MSG(0XAD876, m_writingStFromWorker);
            m_writingStFromWorker = false;

            Fuid invalidFuid = Fuid::getZeroFuid();
            invalidFuid.m_0 = 0;
            setCurrentJobFuid(invalidFuid);
        }
    }
    else
    {
        threadData.m_inputStream = HK_NULL;
        threadData.m_outputStream = HK_NULL;

        Fuid invalidFuid = Fuid::getZeroFuid();
        invalidFuid.m_0 = 0;
        setCurrentJobFuid(invalidFuid);
    }

    m_shared->leave();
#endif
}

void hkCheckDeterminismUtil::cancelJobImpl(Fuid& jobFuid)
{
#if defined(HK_ENABLE_DETERMINISM_CHECKS)
    m_shared->enter();

    hkHashMap<Fuid, JobInfo>::Iterator iter = m_registeredJobs.findKey(jobFuid);
    HK_ASSERT(0xe0173428, m_registeredJobs.isValid(iter), "Could not find job in registered job queue");

    JobInfo& info = m_registeredJobs.getValue(iter);

    if (m_mode == MODE_WRITE)
    {
        delete info.m_data;
        m_registeredJobs.remove(jobFuid);
    }
    else
    {
        info.m_data->reset();
    }

    m_shared->leave();
#endif
}

void hkCheckDeterminismUtil::combineRegisteredJobsImpl() // only used on write
{
#if defined(HK_ENABLE_DETERMINISM_CHECKS)
    HK_TIMER_BEGIN("hkCheckDeterminismUtil::combineRegisteredJobsImpl", HK_NULL);

    m_shared->enter();

    HK_ASSERT_NO_MSG(0XAD876, !m_writingStFromWorker);
    HK_ASSERT_NO_MSG(0XAD876655, m_primaryWorkerThreadInputStream == HK_NULL);
    HK_ASSERT_NO_MSG(0XAD876655, m_primaryWorkerThreadOutputStream == HK_NULL);

    hkUint32 check = hkUint32(CDU_JOBS_START_MARKER);
    m_sharedOutputStream->write((char*)&check, sizeof(check));

    // header; jobs count
    int numRegisteredJobs = m_registeredJobs.getSize();
    m_sharedOutputStream->write((char*)&numRegisteredJobs, sizeof(numRegisteredJobs));

    // combine streams from registered jobs
    {
        for (hkHashMap<Fuid, JobInfo>::Iterator iter = m_registeredJobs.getIterator();
            m_registeredJobs.isValid(iter); iter = m_registeredJobs.getNext(iter))
        {
            JobInfo& info = m_registeredJobs.getValue(iter);
            Fuid fuid = info.m_jobFuid;
            m_sharedOutputStream->write((char*)&fuid, sizeof(fuid));
            int dataSize = info.m_data->getSize();
            m_sharedOutputStream->write((char*)&dataSize, sizeof(dataSize));

            if (m_memoryTrack)
            {
                m_memoryTrack->appendByMove(info.m_data);
            }
            else
            {
                char buffer[2048];
                while (dataSize > 0)
                {
                    int numBytes = hkMath::min2(dataSize, 2048);
                    info.m_data->read(buffer, numBytes);
                    m_sharedOutputStream->write(buffer, numBytes);
                    dataSize -= numBytes;
                }
            }
            HK_ASSERT(0XAD8765dd, info.m_openCount == 0, "Job not finished.");
            delete info.m_data;
        }
        m_registeredJobs.clear();
    }

    {
        check = hkUint32(CDU_JOBS_END_MARKER);
        m_sharedOutputStream->write((char*)&check, sizeof(check));
    }

    m_shared->leave();
    HK_TIMER_END();
#endif
}

void hkCheckDeterminismUtil::extractRegisteredJobsImpl() // only used for read
{
#if defined(HK_ENABLE_DETERMINISM_CHECKS)
    HK_TIMER_BEGIN("hkCheckDeterminismUtil::extractRegisteredJobsImpl", HK_NULL);
    m_shared->enter();

    HK_ASSERT_NO_MSG(0XAD876, !m_writingStFromWorker);

    HK_ASSERT_NO_MSG(0XAD876655, m_primaryWorkerThreadInputStream == HK_NULL);
    HK_ASSERT_NO_MSG(0XAD876655, m_primaryWorkerThreadOutputStream == HK_NULL);

    hkUint32 check;
    m_sharedInputStream->read((char*)&check, sizeof(check));
    HK_ASSERT(0xad8655dd, check == hkUint32(CDU_JOBS_START_MARKER), "Stream inconsistent.");

    HK_ASSERT(0xad87656d, m_registeredJobs.getSize() == 0, "Internal inconsistency.");

    int numRegisteredJobs;
    m_sharedInputStream->read((char*)&numRegisteredJobs, sizeof(numRegisteredJobs));

    for (int i = 0; i < numRegisteredJobs; i++)
    {
        JobInfo info;
        info.m_openCount = 0;
        info.m_data = new hkMemoryTrack;

        m_sharedInputStream->read((char*)&info.m_jobFuid, sizeof(Fuid));

        int dataSize;
        m_sharedInputStream->read((char*)&dataSize, sizeof(dataSize));

        // read the data
        while (dataSize > 0)
        {
            int numBytes = hkMath::min2(dataSize, 2048);
            char buffer[2048];
            m_sharedInputStream->read(buffer, numBytes);
            info.m_data->write(buffer, numBytes);
            dataSize -= numBytes;
        }

        m_registeredJobs.insert(info.m_jobFuid, info);
    }

    m_sharedInputStream->read((char*)&check, sizeof(check));
    HK_ASSERT(0xad8655d1, check == hkUint32(CDU_JOBS_END_MARKER), "Stream inconsistent.");

    HK_ASSERT_NO_MSG(0XAD876, !m_writingStFromWorker);

    m_shared->leave();
    HK_TIMER_END();
#endif
}

void hkCheckDeterminismUtil::clearRegisteredJobsImpl()
{
#if defined(HK_ENABLE_DETERMINISM_CHECKS)
    m_shared->enter();

    HK_ASSERT_NO_MSG(0XAD876, !m_writingStFromWorker);
    HK_ASSERT_NO_MSG(0XAD876655, m_primaryWorkerThreadInputStream == HK_NULL);
    HK_ASSERT_NO_MSG(0XAD876655, m_primaryWorkerThreadOutputStream == HK_NULL);

    // destroy streams from individual jobs

    for (hkHashMap<Fuid, JobInfo>::Iterator iter = m_registeredJobs.getIterator();
        m_registeredJobs.isValid(iter); iter = m_registeredJobs.getNext(iter))
    {
        JobInfo& info = m_registeredJobs.getValue(iter);
        HK_ASSERT(0XAD8765dd, info.m_openCount == 0, "Job not finished.");
        delete info.m_data;
    }
    m_registeredJobs.clear();

    m_shared->leave();
#endif
}

void hkCheckDeterminismUtil::setCurrentJobFuid(hkCheckDeterminismUtil::Fuid jobFuid)
{
    HK_ASSERT(0xad342843, sizeof(jobFuid) == 16, "Job Fuid is greater than 3 bytes. Adjust the code.");
    hkUint32* fuidPtr = (hkUint32*)&jobFuid;
    ThreadData& threadData = getThreadData();
    threadData.m_currentJobFuid0 = fuidPtr[0];
    threadData.m_currentJobFuid1 = fuidPtr[1];
    threadData.m_currentJobFuid2 = fuidPtr[2];
    threadData.m_currentJobFuid3 = fuidPtr[3];
    setCurrentCheckIndex(hkUint32(-1));

    if (m_enableThreadTracker)
    {
        const int ti = HK_THREAD_LOCAL_GET(hkThreadNumber);
        if (ti >= m_threadTracker.getSize())
            m_threadTracker.setSize(ti + 1);
        m_threadTracker[ti] = jobFuid;
        m_maxTheadId = ti > m_maxTheadId ? ti : m_maxTheadId;
    }
}

hkUint32 hkCheckDeterminismUtil::getCurrentCheckIndex()
{
    hkUint32 ci = getInstance().getThreadData().m_currentCheckIndex;
    return ci;
}


void hkCheckDeterminismUtil::setCurrentCheckIndex(hkUint32 checkIndex)
{
    getInstance().getThreadData().m_currentCheckIndex = checkIndex;
}

void hkCheckDeterminismUtil::bumpCurrentCheckIndex()
{
    ThreadData& threadData = getInstance().getThreadData();
    threadData.m_currentCheckIndex++;
}

hkCheckDeterminismUtil::Fuid hkCheckDeterminismUtil::getCurrentJobFuid()
{
    Fuid result;
    hkUint32* fuidPtr = (hkUint32*)&result;
    ThreadData& threadData = getInstance().getThreadData();
    fuidPtr[0] = threadData.m_currentJobFuid0;
    fuidPtr[1] = threadData.m_currentJobFuid1;
    fuidPtr[2] = threadData.m_currentJobFuid2;
    fuidPtr[3] = threadData.m_currentJobFuid3;
    return result;
}

void HK_CALL hkCheckDeterminismUtil::wipeStackImpl(hkUint32 value)
{
    // 512 bytes are wiped
    const int wipeSize = 128;
    hkUint32 array[wipeSize];
    hkString::memSet4(array, value, wipeSize);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
