// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

/// Sorted tree base, declare default comparator.
struct HK_EXPORT_COMMON hkSortedTreeBase
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_TREE, hkSortedTreeBase);

    // Default comparators.

    /// Values.
    struct CompareValues
    {
        template <typename T>
        static HK_INLINE int    compare(const T& a, const T& b) { return a < b ? -1 : (b < a ? 1 : 0); }
    };

    /// Pointers.
    struct ComparePointers
    {
        template <typename T>
        static HK_INLINE int    compare(const T& a, const T& b) { return (*a) < (*b) ? -1 : ((*b) < (*a) ? 1 : 0); }
    };

    /// PRNG.
    struct Prng
    {
        // KISS Based PRNG (http://www.cs.ucl.ac.uk/staff/d.jones/GoodPracticeRNG.pdf)
        HK_INLINE               Prng() { reset(); }
        HK_INLINE void      reset() { m_x=123456789; m_y=234567891; m_z=345678912; m_w=456789123; m_c=0; }
        HK_INLINE hkUint32  nextUint32();
        HK_INLINE hkInt32       nextInt32() { return hkInt32(nextUint32() >> 1); }

        hkUint32    m_x, m_y, m_z, m_w, m_c;
    };
};

/// Sorted tree template.
template <typename T, typename COMP = hkSortedTreeBase::CompareValues, typename INDEX = int, int GROWTH = 16>
struct  hkSortedTree : hkSortedTreeBase
{
    HK_DECLARE_CLASS(hkSortedTree, NewOpaque); 

    /// Tree node.
    struct Node
    {
        HK_DECLARE_CLASS(Node, NewOpaque);

        T       m_value;        ///< Node value.
        INDEX   m_children[2];  ///< Node children.
        INDEX   m_parent;       ///< Node parent.
    };

    /// Constructor.
                                hkSortedTree(int numNodes = 0) { clear(numNodes); }

    /// Reset the tree.
    inline void                 clear(int numNodes = 0);

    /// Returns the number of used nodes.
    HK_INLINE int           getSize() const { return m_size; }

    /// Pre-allocate 'count' nodes.
    inline void                 preAllocateNodes(int count);

    /// Insert a new node.
    inline INDEX                insert(const T& value);

    /// Update an existing node.
    inline void                 update(INDEX nodeIndex);

    /// Update an existing node.
    inline void                 update(INDEX nodeIndex, const T& newValue);

    /// Remove an existing node.
    inline void                 remove(INDEX nodeIndex);

    /// Optimize the tree by re-inserting 'count' nodes 'iterations' times.
    /// Note: If count is zero, all the nodes will be optimized.
    inline void                 optimize(int count = 0, int iterations = 1);

    /// Find a value.
    inline INDEX                find(const T& value) const;

    /// Find a value.
    inline INDEX                find(const T& value, INDEX& closest, int& iterations) const;

    /// Get tree root
    HK_INLINE INDEX     getRoot() const { return m_root; }

    /// Get node from index.
    HK_INLINE   const Node& getNode(INDEX index) const { return m_nodes[index]; }

    /// Get node value from index.
    HK_INLINE   const T&    getValue(INDEX index) const { return getNode(index).m_value; }

    /// Get node value from index.
    HK_INLINE   T&          getValue(INDEX index) { return m_nodes[index].m_value; }

    /// Get the node with the minimum value.
    inline INDEX                getMin() const;

    /// Get the node with the maximum value.
    inline INDEX                getMax() const;

    /// Get the first node.
    inline INDEX                getFirst() const;

    /// Get the next node.
    inline INDEX                getNext(INDEX nodeIndex) const;

    /// Traverse all node from min to max.
    template <typename FUNCTOR>
    inline void                 traverseMinMax(FUNCTOR& functor, INDEX root = 0) const;

    /// Traverse all node from max to min.
    template <typename FUNCTOR>
    inline void                 traverseMaxMin(FUNCTOR& functor, INDEX root = 0) const;

    //
    // Internals.
    //

    //
    inline void                 insertNode(INDEX nodeIndex);

    //
    inline void                 removeNode(INDEX nodeIndex);

    //
    inline bool                 checkIntegrity() const;

    hkArray<Node>   m_nodes;    ///< Nodes storage.
    int             m_size;     ///< Number of used nodes.
    INDEX           m_root;     ///< Tree root node.
    INDEX           m_free;     ///< First free node.
    Prng            m_prng;     ///< PRNG.
};

#include <Common/Base/Container/Tree/hkSortedTree.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
