// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

// Tallest allowable tree, ie. 2^64 branches
#ifndef HK_AA_TREE_HEIGHT_LIMIT
#define HK_AA_TREE_HEIGHT_LIMIT 64
#endif

/// An Arne Andersson tree (AA tree) is a balanced red-black tree with one additional rule.
/// Unlike red-black trees, RED nodes on an AA tree can only be added as a right subchild. In other words,
/// no RED node can be a left subchild. This results in the simulation of a 2-3 tree instead of a 2-3-4 tree,
/// which greatly simplifies the maintenance operations. The maintenance algorithms for a red-black tree
/// need to consider seven different shapes to properly balance the tree. An AA tree on the other hand only
/// needs to consider two shapes due to the strict requirement that only right links can be red.
/// See http://en.wikipedia.org/wiki/Aa_tree
class HK_EXPORT_COMMON hkAATree
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_TREE, hkAATree);

        // user-defined item handling
        typedef int   (HK_CALL *compareFunc) ( _In_ const void *p1, _In_ const void *p2 );
        typedef void* (HK_CALL *cloneFunc)   ( _In_ void *p );
        typedef void  (HK_CALL *destroyFunc) ( _In_ void *p );

        // default item handling
        static int   HK_CALL defaultCompare(_In_ const void* p1, _In_ const void* p2);
        static void* HK_CALL defaultClone(_In_ void* p);
        static void  HK_CALL defaultDestroy(_In_ void* p);

        hkAATree( _In_ compareFunc cmp=defaultCompare, _In_ cloneFunc dup=defaultClone, _In_ destroyFunc rel=defaultDestroy );
        ~hkAATree();

        // tree functions

        void* find( _In_ const void* data ) const;
        hkBool insert( _In_ void* data );
        hkBool erase( _In_ const void *data );
        HK_INLINE hkUint32 getSize() const;
        void clear();

    protected:

        struct HK_EXPORT_COMMON Node
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_TREE, hkAATree::Node);

            int          m_level;   // Horizontal level for balance
            void *       m_data;    // User-defined content
            struct Node* m_link[2]; // Left (0) and right (1) links
        };

        // tree traversal

    public:

        struct HK_EXPORT_COMMON Iterator
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_TREE, hkAATree::Iterator);

            hkAATree* m_tree;                          // Paired tree
            Node*     m_it;                            // Current node
            Node*     m_path[HK_AA_TREE_HEIGHT_LIMIT]; // Traversal path
            hkUint32  m_top;                           // Top of stack

            void* start( _In_ hkAATree* tree, int dir );
            void* move( int dir );
            HK_INLINE void* first( _In_ hkAATree* tree );
            HK_INLINE void* last( _In_ hkAATree* tree );
            HK_INLINE void* next();
            HK_INLINE void* prev();
        };


        Node *       m_root; // Top of the tree
        Node *       m_nil;  // End of tree sentinel

    protected:

        HK_INLINE Node* newNode( _In_ void *data );
        HK_INLINE static void skew(_Inout_ Node** t);
        HK_INLINE static void split(_Inout_ Node** t);

        compareFunc  m_cmp;  // Compare two items (user-defined)
        cloneFunc    m_dup;  // Clone an item (user-defined)
        destroyFunc  m_rel;  // Destroy an item (user-defined)
        hkUint32     m_size; // Number of items in the tree

};

#include <Common/Base/Container/Tree/hkAnderssonTree.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
