// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

namespace hkString
{
    _Ret_range_(== , _String_length_(src))
    HK_EXPORT_COMMON int HK_CALL strLen(_In_z_ const char* src);
}

    /// A view on a range of an existing buffer
    /// When extending its interface please consider keeping it in sync with http://www.open-std.org/jtc1/sc22/wg21/docs/papers/2013/n3609.html
class HK_EXPORT_COMMON hkStringView
{
    public:

        HK_DECLARE_CLASS(hkStringView, New, Reflect);
        HK_REFLECT_AS_STRING_UTF8_IMMUTABLE(&hkStringView::s_impl);
        HK_RECORD_ATTR( hk::Serialize(false), hk::ToString(&hkStringView::toString) );
        static void HK_CALL toString(const hkReflect::Var& var, hkStringBuf& sb, const hkStringView& extra);

            /// Invalid value returned by find() and rfind() to indicate not found.
        static const int npos = -1;

            /// Create an empty view.
        HK_INLINE hkStringView() : m_begin(HK_NULL), m_end(HK_NULL) {}

            /// Creates a view of a c-string. This assumes that it is zero terminated and will call strlen to get its length.
            /// If you have a string literal you should use the fromLiteral() static method.
        HK_INLINE hkStringView(_In_opt_z_ const char* _begin) : m_begin(_begin), m_end(HK_NULL) { if(_begin) { m_end = _begin + hkString::strLen(_begin); } }
            /// Create a view of num elements from begin.
        HK_INLINE hkStringView(_In_count_(num) const char* _begin, hkUlong num) : m_begin(_begin), m_end(_begin+num) { }
            /// Create a view from begin and end pointers.
        HK_INLINE hkStringView(_In_ptrdiff_count_(end) const char* _begin, _Maybevalid_ const char* _end) : m_begin(_begin), m_end(_end) { HK_ASSERT_NO_MSG(0x578f890a, m_begin<=m_end); }

            /// Create a view from a literal array. (T[N])
        template<unsigned N>
        HK_INLINE static hkStringView fromLiteral(_In_ const char (&t)[N]) { return hkStringView(&t[0], N-1); }

            /// Return the number of elements in this view.
        HK_INLINE int getSize() const { return hkLosslessCast<int>(m_end-m_begin); }

            /// Access the i'th element.
        HK_INLINE char operator[](int i) const { HK_ASSERT_NO_MSG(0x410c196d, m_begin+i < m_end); return m_begin[i]; }

            /// Return true if the string is non-empty
        operator hkBoolOperator::BoolType() const { return hkBoolOperator::cast(m_begin!=m_end); }

            /// Get the data pointer (which may not be a null-terminated string).
        HK_INLINE _Ret_maybenull_ const char* data() const { return m_begin; }

        // Iterator support
        typedef const char* iterator;
        typedef const char* const_iterator;
        HK_INLINE const_iterator begin() const { return m_begin; }
        HK_INLINE const_iterator end() const { return m_end; }

        class reverse_iterator
        {
            public:

                reverse_iterator() : m_it(iterator()) {}

                reverse_iterator& operator++() { --m_it; return *this; }
                reverse_iterator operator++(int) { reverse_iterator self = *this; --m_it; return self; }

                char operator*() const { return *(m_it - 1); }

                bool operator!=(const reverse_iterator& rhs) const { return m_it != rhs.m_it; }
                bool operator==(const reverse_iterator& rhs) const { return m_it == rhs.m_it; }

                reverse_iterator& operator+=(int idx) { m_it -= idx; return *this; }

                iterator getForward() const { return m_it - 1; }

            private:

                friend class hkStringView;
                explicit reverse_iterator(iterator it) : m_it(it) {}

                iterator m_it;
        };

        HK_INLINE reverse_iterator rbegin() const { return reverse_iterator(m_end); }
        HK_INLINE reverse_iterator rend() const { return reverse_iterator(m_begin); }

            /// Return a view with n elements removed from the start.
        HK_INLINE hkStringView ltrim(_In_range_(>, 0) int n) const { HK_ASSERT_NO_MSG(0x502f65bc, n >= 0); return hkStringView(m_begin+n, m_end); }
            /// Return a view from the provided iterator to the end of this view.
        HK_INLINE hkStringView ltrim(_In_ iterator _begin) const { HK_ASSERT_NO_MSG(0x6477c823, _begin >= m_begin); return hkStringView(_begin, m_end); }

            /// Return a view with n elements removed from the end.
        HK_INLINE hkStringView rtrim(_In_range_(>, 0) int n) const { return hkStringView(m_begin, m_end-n); }
            /// Return a view from the beginning of this view to the provided iterator.
        HK_INLINE hkStringView rtrim(_Maybevalid_ iterator _end) const { HK_ASSERT_NO_MSG(0x6ab93c7e, _end <= m_end); return hkStringView(m_begin, _end); }

            /// This slice is specified by a start index and count. Not start and end indices.
        HK_INLINE hkStringView slice(int idx, _In_range_(>=, 0) int num) const { HK_ASSERT_NO_MSG(0x14edc8cf, m_begin+idx+num <= m_end); return hkStringView(m_begin+idx, m_begin+idx+num); }

            /// slices like the slice method, but sanitize arguments first
        HK_INLINE hkStringView safeSlice(int idx, int num = -1) const { idx = idx < getSize() ? idx : getSize(); num = (num != -1 && idx + num < getSize()) ? num : getSize() - idx; return slice(idx, num); }

            /// Searches for needle starting at the beginning and returns its index if found, npos otherwise.
        int find(char needle) const;

            /// Searches for any of the needles starting at the beginning and returns the index if found, npos otherwise.
        int find(_In_z_ const char* needles) const;

            /// Searches for needle starting at the end and returns its index if found, npos otherwise.
        int rfind(char needle) const;

            /// Searches for any of the needles starting at the end and returns the index if found, npos otherwise.
        int rfind(_In_z_ const char* needles) const;

        int compare(_In_z_ const char* s) const;
        int compare(const hkStringView& s) const;

        int compareLower(_In_z_ const char* s) const;
        int compareLower(const hkStringView& s) const;

        bool beginsWith(_In_z_ const char* s) const;
        bool beginsWith(const hkStringView& s) const;

        bool endsWith(_In_z_ const char* s) const;
        bool endsWith(const hkStringView& s) const;

            /// Copy this on dst, writing at most n bytes. Return HK_FAILURE if the string length is larger than n.
            /// Always writes a null-terminated string.
        hkResult copy(_Out_writes_z_(n) _Out_bytecap_(dstBytes) char* dst, _In_range_(>, n) int dstBytes, _In_range_(>, 0) int n) const;
        template<int _size>
        HK_INLINE hkResult copy(_Out_writes_z_(n) char (&dst)[_size], _In_range_(>, 0) int n) const;

            /// Return a copy of this on the heap. This copy should be deallocated using hkString::strFree().
        _Ret_z_ const char* dup() const;

            /// Return a copy of this using the provided allocator. This copy should be deallocated using hkString::strFree().
        _Ret_z_ const char* dup(hkMemoryAllocator& alloc) const;

            /// For reflection.
        struct StringImpl;
        static const StringImpl s_impl;

            /// Return the first occurrence of the string needle in this string.
        hkStringView strStr(const hkStringView& needle);

            /// Return the first occurrence of the character needle in this string.
            /// The character at m_end is not considered for matching even though the standard strchr function does consider the terminating null.
        _Ret_maybenull_ const char* strChr( int needle ) const;

    private:

        const char* m_begin;
        const char* m_end;
};

namespace hkHash
{
    HK_INLINE hkUint32 hkHashValue(const hkStringView& str)
    {
        return computeFNV1Hash32(str.begin(), str.getSize());
    }
}


inline bool operator==(const hkStringView& lhs, const hkStringView& rhs) { return lhs.compare(rhs) == 0; }
inline bool operator==(const hkStringView& lhs, _In_z_ const char* rhs) { return lhs.compare(rhs) == 0; }
inline bool operator==(_In_z_ const char* lhs, const hkStringView& rhs) { return rhs.compare(lhs) == 0; }

inline bool operator!=(const hkStringView& lhs, const hkStringView& rhs) { return lhs.compare(rhs) != 0; }
inline bool operator!=(const hkStringView& lhs, _In_z_ const char* rhs) { return lhs.compare(rhs) != 0; }
inline bool operator!=(_In_z_ const char* lhs, const hkStringView& rhs) { return rhs.compare(lhs) != 0; }

inline bool operator<(const hkStringView& lhs, const hkStringView& rhs) { return lhs.compare(rhs) < 0; }
inline bool operator<(const hkStringView& lhs, _In_z_ const char* rhs) { return lhs.compare(rhs) < 0; }
inline bool operator<(_In_z_ const char* lhs, const hkStringView& rhs) { return rhs.compare(lhs) < 0; }

inline bool operator<=(const hkStringView& lhs, const hkStringView& rhs) { return lhs.compare(rhs) <= 0; }
inline bool operator<=(const hkStringView& lhs, _In_z_ const char* rhs) { return lhs.compare(rhs) <= 0; }
inline bool operator<=(_In_z_ const char* lhs, const hkStringView& rhs) { return rhs.compare(lhs) <= 0; }

inline bool operator>(const hkStringView& lhs, const hkStringView& rhs) { return lhs.compare(rhs) > 0; }
inline bool operator>(const hkStringView& lhs, _In_z_ const char* rhs) { return lhs.compare(rhs) > 0; }
inline bool operator>(_In_z_ const char* lhs, const hkStringView& rhs) { return rhs.compare(lhs) > 0; }

inline bool operator>=(const hkStringView& lhs, const hkStringView& rhs) { return lhs.compare(rhs) >= 0; }
inline bool operator>=(const hkStringView& lhs, _In_z_ const char* rhs) { return lhs.compare(rhs) >= 0; }
inline bool operator>=(_In_z_ const char* lhs, const hkStringView& rhs) { return rhs.compare(lhs) >= 0; }

inline hkStringView::reverse_iterator operator+(hkStringView::reverse_iterator it, int idx) { hkStringView::reverse_iterator res(it); res += idx; return res; }
inline hkStringView::reverse_iterator operator+(int idx, hkStringView::reverse_iterator it) { hkStringView::reverse_iterator res(it); res += idx; return res; }

template<int _size>
HK_INLINE hkResult hkStringView::copy(_Out_writes_z_(n) char(&dst)[_size], _In_range_(>, 0) int n) const { return hkStringView::copy(dst, _size, n); }

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
