// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Container/StringMap/hkCachedHashMap.h>
#include <Common/Base/Algorithm/Hash/hkHash.h>

struct HK_EXPORT_COMMON hkStringMapOperations
{
    HK_INLINE hkBool32 equal(hkUlong k1, hkUlong k2) const
    {
        return hkString::strCmp( (const char*)k1, (const char*)k2 ) == 0;
    }

    HK_INLINE hkUlong hash(hkUlong key) const
    {
        // We use the 32bit hash here because the map won't ever grow past 32bit indexes.
        const hkUlong h = hkHash::computeFNV1Hash32((const char*) key);
        return h & (hkUlong(-1)>>1);
    }
};

/// Map strings to integers or pointers.
/// Note that neither the keys nor values are copied so the values
/// must exist for the lifetime of this object.
template <typename V, typename Allocator=hkContainerHeapAllocator>
class hkStringMap
{
    HK_COMPILE_TIME_ASSERT(hkSizeOf(V) <= hkSizeOf(hkUlong));

    public:
        friend class hkTrackerStringMapLayoutHandler;

        HK_DECLARE_CLASS(hkStringMap, NewTemplate, NonCopyable);
        HK_RECORD_ATTR(hk::MemoryTracker(opaque=true, handler=&hkReflect::Tracker::hkStringMapHandler<V, Allocator, false>::func));

            /// Iterator class.
            /// All iterators are invalidated after a mutating operation. i.e., insertion,removal
        typedef typename hkCachedHashMap<hkStringMapOperations, Allocator>::Iterator Iterator;

        typedef hkStringMap<V, hkContainerTempAllocator> Temp;
        typedef hkStringMap<V, hkContainerDebugAllocator> Debug;
        typedef hkStringMap<V, hkContainerHeapAllocator> Heap;

        hkStringMap()
        {
        }

            /// Insert key with associated value val.
            /// If key already exists it is overwritten. The key string is not
            /// copied and must exist for the lifetime of the entry.
            /// If key/val was newly inserted, return true. Otherwise return false.
        HK_INLINE hkBool32 insert( _In_z_ const char* key, _In_ V val )
        {
            return m_map.insert( hkUlong(key), hkUlong(val) );
        }

            /// Return the iterator associated with key. Check with isValid().
        HK_INLINE Iterator findKey( _In_z_ const char* key ) const
        {
            return m_map.findKey( hkUlong(key) );
        }

            /// If key is present return its iterator, else insert (key,val) and return the new iterator.
            /// Thus the returned iterator is always valid.
        HK_INLINE Iterator findOrInsertKey( _In_z_ const char* key, _In_ V val )
        {
            return m_map.findOrInsertKey( hkUlong(key), hkUlong(val) );
        }

            /// Return if this map contains the given key.
        HK_INLINE hkBool32 hasKey( _In_z_ const char* key ) const
        {
            return m_map.hasKey( hkUlong(key) );
        }

            /// Return the value associated with key or if not present, insert and return 'ifNotFound'.
        HK_INLINE V getOrInsert( _In_z_ const char* key, _In_ V ifNotFound )
        {
            return (V)m_map.getOrInsert( hkUlong(key), hkUlong(ifNotFound) );
        }

            /// Return the value associated with key or def if not present.
        HK_INLINE V getWithDefault( _In_z_ const char* key, _In_ V def ) const
        {
            return (V)m_map.getWithDefault( hkUlong(key), hkUlong(def) );
        }

            /// If key present, write value into out and return HK_SUCCESS. Else return HK_FAILURE.
        hkResult get( _In_z_ const char* key, _Out_ V* out ) const
        {
            hkUlong tmp;
            if( m_map.get( hkUlong(key), &tmp ).isSuccess() )
            {
                *out = V(tmp);
                return HK_SUCCESS;
            }
            return HK_FAILURE;
        }

            /// Remove pair at "it".
        void remove( _In_ Iterator it )
        {
            m_map.remove( it );
        }

            /// If key present, remove it and return HK_SUCCESS. Otherwise return HK_FAILURE.
        hkResult remove( _In_z_ const char* key )
        {
            return m_map.remove( hkUlong(key) );
        }

            /// Return the number of elements in this map.
        int getSize() const
        {
            return m_map.getSize();
        }

            /// Perform internal consistency check.
        hkBool isOk() const
        {
            return m_map.isOk();
        }

            /// Get an iterator over the keys of this map.
        Iterator getIterator() const
        {
            return m_map.getIterator();
        }

            /// Get the key at iterator i.
            /// Do not modify this key directly. If you must change a key, remove and re-add it.
        const char* getKey( _In_ Iterator i ) const
        {
            return reinterpret_cast<const char*>(m_map.getKey(i));
        }

            /// Get the value at iterator i.
        V getValue( _In_ Iterator i ) const
        {
            return (V)m_map.getValue(i);
        }

            /// Overwrite the value at iterator i.
        void setValue( _In_ Iterator i, _In_ V v )
        {
            m_map.setValue( i, hkUlong(v) );
        }

            /// Get the next iterator after i.
        Iterator getNext( _In_ Iterator i ) const
        {
            return m_map.getNext(i);
        }

            /// Return if the iterator has reached the end.
        hkBool32 isValid( _In_ Iterator i ) const
        {
            return m_map.isValid(i);
        }

            /// Remove all keys from the map.
        void clear()
        {
            m_map.clear();
        }

        void reset()
        {
            // Discard the contents of m_map
            hkCachedHashMap<hkStringMapOperations, Allocator> m_newMap;
            m_map.swap(m_newMap);
        }

            /// Swap all data with another map.
        void swap( hkStringMap& other )
        {
            m_map.swap(other.m_map);
        }

            /// Insert keys from other into this, overwriting duplicates.
        void merge( const hkStringMap& other )
        {
            m_map.merge(other.m_map);
        }

        int getCapacity() const
        {
            return m_map.getCapacity();
        }

            /// Reserve space for at least numElements;
        void reserve( int numElements )
        {
            m_map.reserve(numElements);
        }

    protected:

        hkCachedHashMap<hkStringMapOperations, Allocator> m_map;
};

#include <Common/Base/Container/StringMap/hkStringMap.inl>
#include <Common/Base/_Auto/TemplateTypes/hkStringMap_Types.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
