// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

// For convenience we allow direct conversion from managed System::String on some platform combinations
#if defined(_MANAGED)
#   define HK_UTF8_SUPPORT_MANAGED_STRING
#endif

// For convenience we allow direct conversion from Windows HSTRING on some platform combinations
#if defined(HK_PLATFORM_WINRT) || defined(HK_SUPPORT_DWM)
#   define HK_UTF8_SUPPORT_HSTRING
#endif

// For convenience we allow direct conversion from c++/cx Platform::String on some platform combinations
#if defined(HK_PLATFORM_WINRT) && defined(__cplusplus_winrt)
#   define HK_UTF8_SUPPORT_CX_STRING
#endif

#if defined(HK_BUILDING_WITH_ENGINE)
#   if defined(HK_PLATFORM_WIN32)
#       define HK_UTF8_SUPPORT_HSTRING
#   endif
#endif

#if defined(HK_UTF8_SUPPORT_HSTRING)
    struct HSTRING__;
    typedef HSTRING__* HSTRING;
    typedef long HRESULT;
#endif

/// UTF-8 string utilities.
///
/// Convert between UTF-8 encoded 'char' strings and "wide encoded" 'wchar_t' strings.
/// The meaning of "wide" depends on the platform:
/// - If sizeof(wchar_t) == 2, then "wide encoding" is assumed to be UTF-16.
/// - If sizeof(wchar_t) == 4, then "wide encoding" is assumed to be UTF-32.
///
/// For most utility functions, unless specified otherwise:
/// - The source string may be null terminated and/or a limit on the number of source
///   code units may be supplied. A value of -1 means no limit; in that case the source
///   string must be null-terminated. The source string is read until the limit is reached,
///   or a null terminator is found, whichever comes first.
/// - The destination limit size is the number of code units in the string including the
///   mandatory null terminator; that is, this is the size of the underlying storage buffer.
///   The destination string is always null terminated, even if the source string was not,
///   unless the limit is zero (which means the storage buffer has no space at all).
namespace hkUtf8
{
    /// Unicode code point in [0:0x10FFFF] (valid Unicode range).
    typedef hkUint32 CodePoint;

    /// Return the number of *codepoints* in the UTF-8 encoded input string.
    /// This function is most useful to count the actual glyphs in the string for displaying it,
    /// or for other UI interactions (caret movement, line wrapping, etc.).
    /// Do not use this function to compute buffer sizes, because some codepoints
    /// (glyphs) are encoded in UTF-8 using multiple code units (char), so the
    /// codepoints and code units are not equivalent; there is at most the same
    /// number of codepoints than the number of code units, but sometimes less.
    /// This function uses a fast and simple method (checking if the upper two bits
    /// are not '10') to determine the start of UTF-8 sequences. Thus the input string
    /// must be valid UTF-8 sequence to give a meaningful result.
    /// To get the size of the buffer needed to hold the UTF-8 string, use (hkString::strLen()+1).
    HK_EXPORT_COMMON int HK_CALL strLen( _In_z_ const char* utf8 );

    /// Convert a single codepoint to UTF-8. Returns the number of bytes needed
    /// to encode this codepoint.
    HK_EXPORT_COMMON int HK_CALL utf8FromCodePoint( _Out_writes_(6) char dst[6], CodePoint src );

    /// Convert a wide string to a UTF-8 encoded string.
    HK_EXPORT_COMMON int HK_CALL utf8FromWide( _Out_writes_(dstCap) char* dst, int dstCap, _In_reads_z_(srcCount) const wchar_t* src, int srcCount=-1 );

    /// Convert a UTF-8 encoded string to a wide string.
    HK_EXPORT_COMMON int HK_CALL wideFromUtf8( _Out_writes_(dstCap) wchar_t* dst, int dstCap, _In_reads_z_(srcCount) const char* src, int srcCount=-1 );

    /// Returns the number of UTF-8 code units (char) required to store the UTF-8
    /// encoding of the given wide string. The length always includes one code unit
    /// (char) for the null terminator, so can be used directly as a storage buffer size.
    HK_EXPORT_COMMON int HK_CALL utf8LengthForWide( _In_reads_z_(srcCount) const wchar_t* str, int srcCount = -1 );

    /// Returns the number of wide char (wchar_t) required to store the "wide encoding"
    /// of the given UTF-8 encoded string. The length always includes one code unit
    /// (wchar_t) for the null terminator, so can be used directly as a storage buffer size.
    HK_EXPORT_COMMON int HK_CALL wideLengthForUtf8( _In_reads_z_(srcCount) const char* str, int srcCount = -1 );

    /// Helper to hold a temporary UTF-8 encoded version of a wide string.
    /// The temporary string is deleted in the destructor so normally you will use this object
    /// as an unnamed temporary, e.g. fopen( Utf8FromWide(L"A wide string") );
    /// If you need keep the string, copy it, e.g. by assigning to a hkStringPtr
    /// hkStringPtr utf8 = Utf8FromWide( L"A wide string" );
    class HK_EXPORT_COMMON Utf8FromWide
    {
        public:

            HK_DECLARE_CLASS(Utf8FromWide, NoNew);

                /// Create a converter.
            explicit Utf8FromWide(_In_z_ const wchar_t* s);

            #if defined(HK_UTF8_SUPPORT_MANAGED_STRING)
            explicit Utf8FromWide(System::String^ s);
            #endif

            #if defined(HK_UTF8_SUPPORT_HSTRING)
            explicit Utf8FromWide(_In_ HSTRING h);
            #endif

            #if defined(HK_UTF8_SUPPORT_CX_STRING)
            explicit Utf8FromWide(Platform::String^ s);
            #endif

                /// Access the null or null terminated UTF-8 string.
            _Ret_z_ const char* cString() const { return m_utf8.begin(); }

                /// Access the null or null terminated UTF-8 string.
            _Ret_z_ operator const char*() const { return m_utf8.begin(); }

                /// Get the length of the UTF-8 string, in number of code units (char).
                /// This excludes the null terminator.
            int getLength() const { return ( m_utf8.isEmpty() ? 0 : m_utf8.getSize() - 1 ); }

        private:

            void init(_In_z_ const wchar_t* s);

                // Invariant: m_utf8.getSize() is the number of code units (char) in the
                // UTF-8 storage, including the null terminator; this is the buffer size.
                // This means that for ASCII strings (single code unit per character) the
                // "length" of the string as usully defined (e.g. by strlen()) is (getSize()-1).
                // In general however for UTF-8 strings where a single character (glyph) may be
                // encoded on multiple code units (char), the number of characters (glyphs) is
                // not related with getSize().
            hkInplaceArray<char, 256, hkContainerTempAllocator> m_utf8;
    };


    /// Helper to hold a temporary "wide encoded" version of a UTF-8 encoded string.
    /// The temporary string is deleted in the destructor so normally you will use this object
    /// as an unnamed temporary, e.g. wfopen( WideFromUtf8("A UTF-8 encoded string") );
    /// If you need keep the string, copy it, e.g. by assigning to a std::wstring
    /// std::wstring wide = WideFromUtf8( "A UTF-8 encoded string" );
    class HK_EXPORT_COMMON WideFromUtf8
    {
        public:

            HK_DECLARE_CLASS(WideFromUtf8, NoNew);

                /// Create a converter.
            explicit WideFromUtf8(_In_z_ const char* utf8);

                /// Access the null or null terminated wide string.
            _Ret_z_ const wchar_t* cString() const { return m_wide.begin(); }

                /// Access the null or null terminated wide string.
            _Ret_z_ operator const wchar_t*() const { return m_wide.begin(); }

                /// Get the length of the wide string, in number of code units (wchar_t).
                /// This excludes the null terminator.
            int getLength() const { return ( m_wide.isEmpty() ? 0 : m_wide.getSize() - 1 ); }

            #if defined(HK_UTF8_SUPPORT_MANAGED_STRING)
                /// Convert to managed String
            operator System::String^() const;
            #endif

            #if defined(HK_UTF8_SUPPORT_HSTRING)
                /// Copy the content of this wide string into a newly-allocator HSTRING.
                /// It is the user's responsibility to deallocate the returned HSTRING
                /// via WindowsDeleteString(). So it is recommended to use it with a HString
                /// wrapper which automatically manages the deallocation in its dtor, e.g.:
                ///   HString hstr;
                ///   wide.dupHSTRING( hstr.GetAddressOf() );
                /// In general if you should prefer using hkTmpHstring when a copy is not
                /// necessary, as hkTmpHstring creates a string reference around an existing
                /// buffer (e.g. the current WideFromUtf8 object) without extra allocation.
            HRESULT dupHSTRING(_Outptr_ HSTRING* ret) const;
            #endif

            #if defined(HK_UTF8_SUPPORT_CX_STRING)
                /// Convert to ref String
            operator Platform::String^() const;
            #endif

        private:

                // Invariant: m_wide.getSize() is the number of code units (wchar_t) in the
                // storage, including the null terminator; this is the buffer size.
                // This means that for UCS-2 strings (single code unit per character) the
                // "length" of the string is (getSize()-1).
                // In general however for UTF-16 or UTF-32 strings (depending on what wchar_t
                // maps to) where a single character (glyph) may be encoded on multiple code
                // units (wchar_t), the number of characters (glyphs) is not related with getSize().
            hkInplaceArray<wchar_t, 256, hkContainerTempAllocator> m_wide;
    };

    /// Walk over the codepoints of a UTF-8 encoded string.
    /// Note that there is no isValid() method; the iterator starts in an invalid state
    /// and advance() must be called before current().
    class HK_EXPORT_COMMON Iterator
    {
        public:
            HK_DECLARE_CLASS(Iterator, NoNew);

                /// Create an iterator.
            Iterator(_In_z_ const char* s) : m_utf8( reinterpret_cast<const hkUint8*>(s) ), m_current( CodePoint(-1) ) {}

                /// Return the current codepoint.
            CodePoint current() const { return m_current; }

                /// Move to the next codepoint, returning false if there are no more codpoints and true otherwise.
                /// If non-null the number of bytes consumed is written into lenOut.
                /// If there is an encoding error we still return true and U+FFFD (REPLACEMENT CHARACTER) will be returned from current().
            bool advance(_Out_opt_ int* lenOut=HK_NULL);

        private:

            const hkUint8* m_utf8;
            CodePoint m_current;
    };
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
