// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : COMMON
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkBaseTypes.h>
#include <Common/Base/Container/String/hkString.h>

/// This class is used to store a char* c-string.
/// It automatically handles allocations and deallocation of
/// stored c-string. The string memory may be externally or internally
/// allocated, the lowest bit of the pointer being used as a
/// should-deallocate flag. This simplifies destruction of objects
/// which may be allocated in a packfile for instance.
class HK_EXPORT_COMMON hkStringPtr
{
public:
    HK_DECLARE_CLASS(hkStringPtr, New, Reflect);
    HK_REFLECT_AS_STRING_UTF8_IMMUTABLE(&hkStringPtr::s_impl);
    HK_RECORD_ATTR(hk::IncludeInMgd(false));
    HK_RECORD_ATTR(hk::MemoryTracker(opaque = true, handler = &hkReflect::Tracker::hkStringPtrHandler));

    typedef HK_ALIGN(char AlignedCharBuf[], 2);
    HK_RECORD_ATTR(hk::DefaultStringType(hkStringPtr::AlignedCharBuf));
    typedef const char* ReflectDefaultType;

    ///
    enum StringFlags
    {
        /// If we own it, we free it.
        OWNED_FLAG = 0x1,
    };

    /// Default constructor.
    inline hkStringPtr();

    /// Copy from a const char*.
    hkStringPtr(_In_z_ const char* string);

    /// Nullptr constructor.
    hkStringPtr(nullptr_t);

    /// Copy from a const char*, null terminated after len characters.
    hkStringPtr(_In_reads_z_(len) const char* string, int len);

    /// Copy from a string view.
    explicit hkStringPtr(const hkStringView strView);

    /// Copy from a string buffer.
    explicit hkStringPtr(const hkStringBuf& strBuf);

    /// Copy constructor.
    hkStringPtr(const hkStringPtr& strRef);

    /// Destructor.
    ~hkStringPtr();

    /// Copy from c string.
    hkStringPtr& operator=(_In_z_ const char* s);

    /// Copy from a string view.
    hkStringPtr& operator=(const hkStringView s);

    /// Copy from a string buffer.
    hkStringPtr& operator=(const hkStringBuf& s);

    /// Copy from hkStringPtr.
    hkStringPtr& operator=(const hkStringPtr& s);

    /// Return const pointer to stored string.
    inline const char* cString() const;

    /// Implicit conversion to c string.
    inline operator const char*() const;

    /// Number of characters in this string.
    int getLength() const;

    /// Const index operator.
    inline char operator[](int idx) const;

    /// Replace this strings value with the formatted string.
    void printf(_Printf_format_string_ const char* fmt, ...);

    /// As for printf, but takes a va_list.
    void vprintf(_Printf_format_string_ const char* fmt, va_list args);

    /// Replace this strings value with the given string.
    void set(_In_reads_z_(len)const char* s, int len = -1);

    /// Initializes the internal buffer with the given pointer, without copies.
    /// Instead of copying the string argument contents in a new buffer
    /// as all the other methods, this one is used to set the pointer
    /// directly without allocation and copies. The string is marked as
    /// not owned. The user should make sure that the new pointed buffer
    /// will remain valid at least until the end of the hkStringPtr
    /// object life.
    /// The given pointer must be 2 bytes aligned, this is because the
    /// least significant bit in the stored m_stringAndFlag pointer
    /// is used as a flag (and not as part of the pointer).
    void setPointerAligned(_In_z_ const char* s);

    /// Comparison.
    inline hkBool32 operator==(_In_opt_z_ const char* s) const;
    /// Comparison.
    inline hkBool32 operator!=(_In_opt_z_ const char* s) const;
    /// Less than comparison.
    inline hkBool32 operator<(_In_opt_z_ const char* s) const;
    /// Less than or equal comparison.
    inline hkBool32 operator<=(_In_opt_z_ const char* s) const;
    /// Greater than comparison.
    inline hkBool32 operator>(_In_opt_z_ const char* s) const;
    /// Greater than or equal comparison.
    inline hkBool32 operator>=(_In_opt_z_ const char* s) const;

    /// Return true if s is a prefix of this string
    inline hkBool32 startsWith(_In_z_ const char* s) const;
    /// Return true if s is a suffix of this string
    inline hkBool32 endsWith(_In_z_ const char* s) const;
    /// Return true if s is contained in this string
    inline hkBool32 contains(_In_z_ const char* s) const;
    /// Forward to hkString::strCmp
    inline int compareTo(_In_opt_z_ const char* s) const;

    /// Returns true if the string is NULL or "" (empty string).
    bool isNullOrEmpty() const;

    /// Swap the content of this string with another string.
    void swap(hkStringPtr& other);

    /// For manual reflection.
    struct StringImpl;
    static const StringImpl s_impl;
public:

#ifdef HK_MOVE_SEMANTICS
    inline hkStringPtr(hkStringPtr&& other);
    hkStringPtr& operator= (hkStringPtr&& other);
#endif


private:
    /// m_stringAndFlag&~1 is the real pointer, m_stringAndFlag&1 is the deallocate flag.
    char* m_stringAndFlag;
};

HK_ALWAYS_INLINE hkUint32 hkHashValue(const hkStringPtr& string)
{
    using namespace hkHash;
    return hkHashValue(string.cString());
}

hkBool32 operator==(_In_opt_z_ const char* l, const hkStringPtr& r);
hkBool32 operator!=(_In_opt_z_ const char* l, const hkStringPtr& r);
hkBool32 operator< (_In_opt_z_ const char* l, const hkStringPtr& r);
hkBool32 operator<=(_In_opt_z_ const char* l, const hkStringPtr& r);
hkBool32 operator> (_In_opt_z_ const char* l, const hkStringPtr& r);
hkBool32 operator>=(_In_opt_z_ const char* l, const hkStringPtr& r);

hkBool32 operator==(const hkStringPtr& l, const hkStringPtr& r);
hkBool32 operator!=(const hkStringPtr& l, const hkStringPtr& r);
hkBool32 operator< (const hkStringPtr& l, const hkStringPtr& r);
hkBool32 operator<=(const hkStringPtr& l, const hkStringPtr& r);
hkBool32 operator> (const hkStringPtr& l, const hkStringPtr& r);
hkBool32 operator>=(const hkStringPtr& l, const hkStringPtr& r);

namespace hkTrait
{
    template<>
    struct CanBeReinterpreted<const char*, const hkStringPtr> : public TrueType {};
    template<>
    struct CanBeReinterpreted<char*, const hkStringPtr> : public TrueType {};
}

#include <Common/Base/Container/String/hkStringPtr.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
