// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Container/String/hkStringPtr.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Fwd/hkcstdarg.h>
#include <Common/Base/Memory/Tracker/hkMemoryTrackerSnapshot.h>

static void assign( _Inout_ char** ptr, _In_reads_z_(len) const char* src, int len=-1 )
{
    if( (hkClearBits(*ptr, hkStringPtr::OWNED_FLAG) == src) && (len == -1) )
    {
        return;
    }

    // Don't deallocate the original until we have copied it, supporting truncation by self-assignment
    const char* origPtr = *ptr;

    if( src )
    {
        char* p = ( len == -1 )
            ? hkString::strDup(src)
            : hkString::strNdup(src, len);
        HK_ASSERT_NO_MSG(0x3d02cfc8, (hkUlong(p) & 1) == 0 );
        *ptr = p + 1; // set the owned flag
    }
    else
    {
        *ptr = HK_NULL;
    }

    if( (hkUlong(origPtr) & hkStringPtr::OWNED_FLAG) == hkStringPtr::OWNED_FLAG )
    {
        hkDeallocate<char>( const_cast<char*>(origPtr) - 1);
    }

}

hkStringPtr::hkStringPtr(_In_z_ const char* string)
    : m_stringAndFlag(HK_NULL)
{
    assign(&m_stringAndFlag, string);
}

hkStringPtr::hkStringPtr(_In_reads_z_(len) const char* s, int len)
    : m_stringAndFlag(HK_NULL)
{
    assign(&m_stringAndFlag, s, len);
}

hkStringPtr::hkStringPtr(const hkStringPtr& strRef)
    : m_stringAndFlag(HK_NULL)
{
    assign(&m_stringAndFlag, strRef);
}

hkStringPtr::hkStringPtr(const hkStringView strView)
    : m_stringAndFlag(HK_NULL)
{
    assign(&m_stringAndFlag, strView.begin(), strView.getSize());
}

hkStringPtr::hkStringPtr(const hkStringBuf & strBuf)
    : m_stringAndFlag(HK_NULL)
{
    assign(&m_stringAndFlag, strBuf.cString(), strBuf.getLength());
}

hkStringPtr::~hkStringPtr()
{
    assign(&m_stringAndFlag, HK_NULL);
}

hkStringPtr& hkStringPtr::operator=(_In_z_ const char* string)
{
    assign(&m_stringAndFlag, string);
    return *this;
}

hkStringPtr& hkStringPtr::operator=(const hkStringView s)
{
    assign(&m_stringAndFlag, s.begin(), s.getSize());
    return *this;
}

hkStringPtr& hkStringPtr::operator=(const hkStringBuf& s)
{
    assign(&m_stringAndFlag, s.cString(), s.getLength());
    return *this;
}

hkStringPtr& hkStringPtr::operator=(const hkStringPtr& string)
{
    assign(&m_stringAndFlag, string);
    return *this;
}

int hkStringPtr::getLength() const
{
    const char* s = cString();
    return s ? hkString::strLen(s) : 0;
}

void hkStringPtr::printf(_Printf_format_string_ const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    vprintf(fmt, args);
    va_end(args);
}

void hkStringPtr::vprintf(_Printf_format_string_ const char* fmt, va_list args)
{
    hkStringBuf buf;
    buf.vprintf(fmt, args);
    *this = buf.cString();
}

void hkStringPtr::set(_In_reads_z_(len) const char* s, int len)
{
    assign(&m_stringAndFlag, s, len);
}

void hkStringPtr::setPointerAligned(_In_z_ const char* s)
{
    // make sure that the pointer is at least 2-aligned
    HK_ASSERT(0x3d02cfc8, (hkUlong(s) & 1) == 0, "The pointer must refer to a 2-aligned storage");
    // set the pointer
    if(cString() != s) // do not reassign the same pointer
    {
        // deallocate the previous string (if owned) and set the internal pointer to NULL
        assign(&m_stringAndFlag, HK_NULL);
        // copy the given pointer (force not owned)
        m_stringAndFlag = reinterpret_cast<char*>(hkUlong(s));
    }
}


#ifdef HK_MOVE_SEMANTICS

hkStringPtr& hkStringPtr::operator= ( hkStringPtr&& other )
{
    if (&other!=this)
    {
        assign(&m_stringAndFlag, HK_NULL);

        m_stringAndFlag = other.m_stringAndFlag;
        other.m_stringAndFlag = HK_NULL;
    }
    return (*this);
}

#endif


struct hkStringPtr::StringImpl : public hkReflect::Detail::StringImpl
{
    StringImpl() { /*keep clang happy*/ }

    static hkStringPtr* cast(_In_ void* ptr)
    {
        return static_cast<hkStringPtr*>(ptr);
    }

    static const hkStringPtr* cast(_In_ const void* ptr)
    {
        return static_cast<const hkStringPtr*>(ptr);
    }

    virtual hkResult setValue(_In_ void* addr, _In_ const hkReflect::StringType*, hkReflect::StringValue newValue) const HK_OVERRIDE
    {
        *cast(addr) = newValue;
        return HK_SUCCESS;
    }

    virtual hkResult getValue(_In_ const void* addr, _In_ const hkReflect::StringType* type, _Out_ hkReflect::StringValue* val) const HK_OVERRIDE
    {
        *val = cast(addr)->cString();
        return HK_SUCCESS;
    }

    virtual hkResult inplaceFixup( _In_ void* self, _In_ const hkReflect::Type* selfType, _In_ void* target, _In_ const hkReflect::Type* targetType, int count) const HK_OVERRIDE
    {
        HK_ASSERT_NO_MSG(0x537d43db, (hkUlong(target)&1) == 0 );
        *(void**)self = target;
        return HK_SUCCESS;
    }

    virtual void clearAllocs(_Inout_ void* string, _In_ const hkReflect::StringType* type) const HK_OVERRIDE
    {
        cast(string)->~hkStringPtr();
    }
};

const hkStringPtr::StringImpl hkStringPtr::s_impl;

void hkReflect::Tracker::hkStringPtrHandler(const hkReflect::Var& var, hkMemoryTrackerSnapshot& snapshot)
{
    const char* stringPtr = *static_cast<const char**>(var.getAddress());

    if (hkUlong(stringPtr) & hkStringPtr::OWNED_FLAG)
    {
        stringPtr = hkClearBits(stringPtr, hkStringPtr::OWNED_FLAG);
        snapshot.addLinkedBlock(hkMemoryTrackerSnapshot::Block(
            "buffer_hkStringPtr",
            stringPtr,
            hkLosslessCast<int>(hkString::strLen(stringPtr)) + 1), true);
        snapshot.addLinkedEasyOverhead(stringPtr);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
