// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#ifndef HKBASE_HKSTRINGBUF_H
#define HKBASE_HKSTRINGBUF_H

#include <Common/Base/Types/hkVarArgs.h>
#include <Common/Base/Types/hkVarArgsV.h>

class hkStringView;

    /// This class is useful for short-term manipulation of a string. It's most commonly used as a local variable.
    /// If you want an object to point to a string that is seldom or never changed it's better to use hkStringPtr.
    /// Note that since this object contains an inplace array, it is not memcpy-able and cannot be placed in an hkArray.
class HK_EXPORT_COMMON hkStringBuf
{
    public:
        HK_DECLARE_CLASS(hkStringBuf, New, Reflect, ReflectDetails(fields=false));
        HK_REFLECT_AS_STRING_UTF8_MUTABLE(&hkStringBuf::s_impl);
        HK_RECORD_ATTR(hk::Serialize(false), hk::IncludeInMgd(false));

            /// Creates an empty string.
        hkStringBuf();

            /// Creates a new string as a copy from a null terminated character array.
        hkStringBuf(_In_opt_z_ const char* s);

            /// Construct a string as the concatenation of several strings.
            /// The list is terminated by the first null pointer.
        hkStringBuf(_In_z_ const char* s0,
            _In_z_ const char* s1,
            _In_opt_z_ const char* s2=HK_NULL,
            _In_opt_z_ const char* s3=HK_NULL,
            _In_opt_z_ const char* s4=HK_NULL,
            _In_opt_z_ const char* s5=HK_NULL);

            /// Creates a new string as a copy from a buffer of length len.
            /// The copied string will automatically be null terminated.
        hkStringBuf(_In_reads_(len) const char* b, int len);

            /// Creates a new string as a copy of an existing one.
        hkStringBuf(const hkStringBuf& s);

            /// Creates a new string as a copy of an existing one.
        hkStringBuf(const hkStringView& s);

            /// Copy of an existing string.
        hkStringBuf& operator=(const hkStringBuf& s);

            /// Copy of an existing c string.
        hkStringBuf& operator=(_In_opt_z_ const char* s);

            /// Copy of an existing string view.
        hkStringBuf& operator=(const hkStringView& s);

    private:

            /// For internal use only.
            /// Noncopying constructor, the buffer pointed by ptr is used.
        hkStringBuf(_In_z_count_(capacity) char* ptr, _In_range_(0, capacity) int size, _In_range_(>=, 0) int capacity);

    public:

            /// sets the size as length+1
        void setLength( _In_range_(>=, 0) int length );

    public:

            /// Read only access the internal buffer.
        _Ret_maybenull_z_
        operator const char*() const;

            /// Read only access the internal buffer.
        operator hkStringView() const;

            /// Read only access the internal buffer.
        _Ret_z_
        const char* cString() const
            HK_ATTR( hk::Reflect );

            /// Returns the number of characters in this string excluding the trailing NULL
        int getLength() const;

            /// Read only access the i'th character.
        char operator[] (int i) const;

            /// Returns the first index of c given range, or -1 if not found.
            /// The index returned is from the start of the string, not from startIndex.
        int indexOf (char c, _In_range_(0, endIndex) int startIndex=0, _In_range_(startIndex, HK_INT32_MAX) int endIndex=HK_INT32_MAX) const;

            /// Returns the first index of c given range, or -1 if not found.
            /// The index returned is from the start of the string, not from startIndex.
        int indexOf( _In_z_ const char* s, _In_range_(0, endIndex) int startIndex=0, _In_range_(startIndex, HK_INT32_MAX) int endIndex=HK_INT32_MAX ) const;

            /// Returns the first index of c given range ignoring case, or -1 if not found.
        int indexOfCase( _In_z_ const char* ) const;

            /// Returns the last index of c, or -1 if not found.
            /// The index returned is from the start of the string, not from startIndex.
        int lastIndexOf(char c, _In_range_(0, endIndex) int startIndex=0, _In_range_(startIndex, HK_INT32_MAX) int endIndex=HK_INT32_MAX) const;

            /// Returns the last index of the string s, or -1 if not found.
            /// The index returned is from the start of the string, not from startIndex.
        int lastIndexOf(_In_z_ const char*, _In_range_(0, endIndex) int startIndex=0, _In_range_(startIndex, HK_INT32_MAX) int endIndex=HK_INT32_MAX) const;

            /// Returns true if the string equals "". hkStringBuf cannot store HK_NULL strings, so that does not need to be checked.
        bool isEmpty() const;

            /// Returns <0,0,>0 if *this is lexicographically less than, equal to or greater than s.
        int compareTo(_In_z_ const char*) const;

            /// Returns <0,0,>0 if *this is lexicographically less than, equal to or greater than s, ignoring case.
        int compareToIgnoreCase(_In_z_ const char*) const;

            /// Returns <0,0,>0 if *this is lexicographically less than, equal to or greater than s.
        int compareTo(const hkStringBuf& s) const;

            /// Returns compareTo(s)<0
            /// Note: we use hkBool here to work with map<>
        bool operator< (_In_z_ const char*) const;
            /// Returns compareTo(s)>0
        bool operator> (_In_z_ const char*) const;
            /// Returns compareTo(s)==0
        bool operator==(_In_z_ const char*) const;
            /// Returns compareTo(s)!=0
        bool operator!=(_In_z_ const char*) const;

            /// Return true if this string starts with s.
        bool startsWith( _In_z_ const char* ) const;
            /// Return true if this string starts with s, ignoring case.
        bool startsWithCase( _In_z_ const char* ) const;

            /// Return true if this string ends with s.
        bool endsWith( _In_z_ const char* ) const;
            /// Return true if this string ends with s, ignoring case.
        bool endsWithCase( _In_z_ const char* s ) const;

            /// Splits this string at the separator and returns the sub strings.
            /// This method destroys the original string as it replaces occurrences
            /// of the separator with null bytes. Returns the number of splits.
        int split( int sep, hkArray<hkCString>::Temp& bits );


            /// Set this to the empty string ("")
        void clear();

            /// Overwrite the current value. Arguments are the same as for ::sprintf.
        hkStringBuf& printf(_Printf_format_string_ const char* format, ...); // member function, not HK_CALL
            /// Overwrite the current value. Arguments are the same as for ::sprintf.
        hkStringBuf& vsprintf(_Printf_format_string_ const char* format, va_list hkargs);
            /// Append a formatted string. Arguments are the same as for ::sprintf.
        hkStringBuf& appendPrintf(_Printf_format_string_ const char* format, ...); // member function, not HK_CALL

            /// Overwrite the current value. Arguments are the same as for ::vsprintf.
        hkStringBuf& vprintf(_Printf_format_string_ const char* format, va_list args); // member function, not HK_CALL
          /// Append a formatted string. Arguments are the same as for ::vsprintf.
        hkStringBuf& appendVprintf(_Printf_format_string_ const char* format, va_list args); // member function, not HK_CALL

            /// Sets *this as the concatenation of *this and s.
        hkStringBuf& operator+= (_In_opt_z_ const char* s);

            /// Parameter to hkStringBuf::replace functions
        enum ReplaceType
        {
            REPLACE_ONE,
            REPLACE_ALL
        };

            /// Append the concatenated strings to this string.
        hkStringBuf& appendJoin(_In_z_ const char* s0,
            _In_opt_z_ const char* s1=HK_NULL,
            _In_opt_z_ const char* s2=HK_NULL,
            _In_opt_z_ const char* s3=HK_NULL,
            _In_opt_z_ const char* s4=HK_NULL,
            _In_opt_z_ const char* s5=HK_NULL);
            /// Replace this with the concatenated strings.
        hkStringBuf& setJoin(_In_z_ const char* s0,
            _In_opt_z_ const char* s1=HK_NULL,
            _In_opt_z_ const char* s2=HK_NULL,
            _In_opt_z_ const char* s3=HK_NULL,
            _In_opt_z_ const char* s4=HK_NULL,
            _In_opt_z_ const char* s5=HK_NULL);

            /// Remove n characters from the start of this string.
        hkStringBuf& chompStart(_In_range_(>=, 0) int n);
            /// Remove n characters from the end of this string.
        hkStringBuf& chompEnd(_In_range_(>=, 0) int n);
            /// Set the string as a substring of itself.
        hkStringBuf& slice(int startOffset, int length);
            /// Set the string. If length is not given, everything up to a null is copied.
        hkStringBuf& set(_In_reads_z_(len) const char* s, int len);
        hkStringBuf& set(_In_z_ const char* s)    
            HK_ATTR( hk::Reflect ) { return set(s, -1); }
            /// Append a string. If length is not given, everything up to a null is copied.
        hkStringBuf& append(_In_reads_z_(len) const char* s, int len=-1);
            /// Append the character c, 'repeat' times.
        hkStringBuf& appendRepeat(_In_range_(>, 0) int count, char c);
            /// Prepend a string. If length is not given, everything up to a null is copied.
        hkStringBuf& prepend(_In_reads_z_(len) const char* s, int len=-1);
            /// Insert a string at the given index. If length is not given, everything up to a null is copied.
        hkStringBuf& insert(_In_range_(>=, 0) int pos, _In_reads_opt_z_(len) const char* s, int len=-1);

            /// Extract the filename and extension after the last '/'. e.g. Path/To/File[.ext] -> File[.ext].
            /// If there is no '/', the result is unchanged.
        hkStringBuf& pathBasename();
            /// Extract the value up to the last '/'. e.g. Path/To/File -> Path/To.
            /// If there is no '/', the result is the empty string.
        hkStringBuf& pathDirname();
            /// Extract the value after the first '/'. e.g. Path/To/File -> To/File. In case of network paths
            /// all leading slashes are skipped.
            /// If there is no '/', the result is the empty string.
        hkStringBuf& pathChildname();
            /// Normalize a path by removing instances of '..'. e.g. Path/../To/File -> To/File.
            /// If the path is already normalized the original path is returned. All references
            /// to '\\' are changed to '/' (even if path is already normalized).
        hkStringBuf& pathNormalize();
            /// Extract the extension of the path. e.g. 'Path/To.File' -> '.File'. If an extension
            /// does not exist the result is the empty string.
        hkStringBuf& pathExtension();
            /// Return the filename without the outermost extension e.g. 'Path/To.File.Extension' -> 'Path/To.File'.
            /// If there is no extension, it result is unchanged.
        hkStringBuf& pathWithoutExtension();
            /// Append all of the arguments adding '/' as necessary.
        hkStringBuf& pathAppend(_In_z_ const char* p0, _In_opt_z_ const char* p1=HK_NULL, _In_opt_z_ const char* p2=HK_NULL);
            /// If the current string points to an absolute path, convert the path to be relative to rootPath
            /// Edge cases: c:\file.txt relative to c:\file.txt will return ""
            /// This is currently case insensitive, see [COM-3606]
        hkResult pathRelativeTo(_In_z_ const char* rootPath);

            /// Returns a new string where occurrences of 'from' have been replaced with 'to'.
            /// If ReplaceType==REPLACE_ONE only the first occurrence is replaced.
        bool replace( char from, char to, ReplaceType = REPLACE_ALL );
        bool replace( _In_z_ const char* from, _In_z_ const char* to, ReplaceType = REPLACE_ALL );

            /// Make all characters lower case.
        hkStringBuf& lowerCase();
            /// Make all characters upper case.
        hkStringBuf& upperCase();

            /// Remove trailing whitespace
        hkStringBuf& trimEnd();
            /// Remove leading whitespace
        hkStringBuf& trimStart();

        /// Remove trailing whitespace and the first instance of c. Error if anything other than whitespace is seen before c
        hkResult trimEndFromLast(char c);
        /// Remove leading whitespace and the last instance of c. Error if anything other than whitespace is seen after c
        hkResult trimStartToFirst(char c);

            // Advanced usage.
        hkArray<char>::Temp& getArray();

            /// Internal
        // This function uses C# style formating with {} or {0} as place-holders.
        int formatL( _In_opt_z_ const char* fmt, hkVarArgs::VaTypes argTypes, ... );
        // This function uses C# style formating with {} or {0} as place-holders.
        int appendFormatL(_In_opt_z_ const char* fmt, hkVarArgs::VaTypes argTypes, ... );
        // This function uses C# style formating with {} or {0} as place-holders.
        int formatV(_In_opt_z_ const char* fmt, const hkVarArgs::Vector& args );
        // This function uses C# style formating with {} or {0} as place-holders.
        int appendFormatV(_In_opt_z_ const char* fmt, const hkVarArgs::Vector& args );

            /// String formatting via interpolation. e.g. sb.format("second:{1} first right aligned {0,30}", 44.1, "xxx");
            /// The first argument must be a format string.
            /// Formatting follows the similar rules to CLR String.Format. Each placeholder item for interpolation has
            /// the form "{[INDEX][,WIDTH][:EXTRA]}". E.g. "{}", "{,-10}", "{,8:x}", "{3:*}".
            /// If an INDEX is not supplied, the index is inferred. i.e. "{} {4} {}" == "{0} {4} {5}"
            /// A integer WIDTH may be specified after a comma. Positive right aligns, negative width left aligns.
            /// EXTRA formatting may be specified after a colon. E.g. "{:x}" to show an integer as hexadecimal. "{:*}" will
            /// dereference a pointer argument if it is not null. The exact meaning depends on the argument it is applied to.
            /// If the argument has a hk::ToString attribute, EXTRA will be forwarded to it.
        HK_VARARGS_FUNCTION_FORMAT(int, format, /*forwards to*/ formatV);

            /// String formatting, same es format but appends the result
        HK_VARARGS_FUNCTION_FORMAT(int, appendFormat, /*forwards to*/ appendFormatV);

        struct StringImpl;
            /// For manual reflection.
        static const StringImpl s_impl;

    private:

            /// The string storage.
            /// The storage is always null terminated and the null terminator is included in the array
            /// size so the length of the string is always m_string.getSize()-1.
        hkInplaceArray<char, 128, hkContainerTempAllocator> m_string;
};

bool operator>(_In_z_ const char* str, const hkStringBuf& buf);
bool operator>(const hkStringBuf& lhs, const hkStringBuf& rhs);
bool operator<(_In_z_ const char* str, const hkStringBuf& buf);
bool operator<(const hkStringBuf& lhs, const hkStringBuf& rhs);
bool operator==(_In_z_ const char* str, const hkStringBuf& buf);
HK_EXPORT_COMMON bool operator==(const hkStringBuf& lhs, const hkStringBuf& rhs);

bool operator!=(_In_z_ const char* str, const hkStringBuf& buf);
bool operator!=(const hkStringBuf& lhs, const hkStringBuf& rhs);

    /// Arrays of hkStringBuf don't work.
    /// hkArray relies on realloc and memmove, but hkStringBuf has a pointer to internal storage
    /// which breaks this requirement.
template<> class hkArray<hkStringBuf>;


#include <Common/Base/Container/String/hkStringBuf.inl>

#endif // HKBASE_HKSTRING_H

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
