// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Container/StringView/hkStringView.h>
#include <Common/Base/Fwd/hkcstdarg.h>

static inline char tolower(char i)
{
    return i >= 'A' && i <= 'Z'
        ? i + ('a'-'A')
        : i;
}

static inline char toupper(char i)
{
    return i >= 'a' && i <= 'z'
        ? i - ('a'-'A')
        : i;
}

static inline bool isWhiteSpace(char i)
{
    return (i == ' ') || (i == '\t') || (i == '\r') || (i == '\n');
}

#ifdef HK_DEBUG_SLOW
static bool aliasSafe(_In_z_ const char* p, _In_z_ const char* a, int na)
{
    hkUlong up = hkUlong(p);
    hkUlong start = hkUlong(a);
    hkUlong end = hkUlong(a+na);
    return up < start || up >= end;
}
#endif

static int getInt(const hkStringView& s)
{
    hkStringBuf buf(s);
    return hkString::atoi(buf.cString(), 10);
}

hkStringBuf::hkStringBuf(_In_opt_z_ const char* s)
{
    if( s != HK_NULL )
    {
        int len = hkString::strLen(s);
        setLength( len );
        hkString::memMove(m_string.begin(), s, len );
    }
    else
    {
        setLength(0);
    }
}

hkStringBuf::hkStringBuf(_In_z_ const char* s0, _In_z_ const char* s1, _In_opt_z_ const char* s2, _In_opt_z_ const char* s3, _In_opt_z_ const char* s4, _In_opt_z_ const char* s5)
{
    m_string.pushBackUnchecked(0);
    appendJoin(s0,s1,s2,s3,s4,s5);
}

hkStringBuf::hkStringBuf(_In_reads_(len) const char* b, int len)
{
    setLength(len);
    hkString::memMove(m_string.begin(), b, len);
}

hkStringBuf::hkStringBuf(const hkStringBuf& s)
{
    m_string = s.m_string;
}

hkStringBuf::hkStringBuf(const hkStringView& s)
{
    setLength(s.getSize());
    hkString::memMove(m_string.begin(), s.begin(), s.getSize());
}

hkStringBuf& hkStringBuf::operator=(const hkStringBuf& s)
{
    m_string = s.m_string;
    return *this;
}

hkStringBuf& hkStringBuf::operator=(_In_opt_z_ const char* s)
{
    return (*this) = hkStringView(s);
}

hkStringBuf& hkStringBuf::operator=(const hkStringView& s)
{
    if(s)
    {
        setLength(s.getSize());
        hkString::memMove(m_string.begin(), s.begin(), s.getSize());
    }
    else
    {
        setLength(0);
    }
    return *this;
}

int hkStringBuf::indexOf(char c, _In_range_(0, endIndex) int startIndex, _In_range_(startIndex, HK_INT32_MAX) int endIndex) const
{
    endIndex = hkMath::min2( endIndex, getLength() );
    for( int i = startIndex; i < endIndex; ++i )
    {
        if( m_string[i] == c )
        {
            return i;
        }
    }
    return -1;
}

int hkStringBuf::indexOf( _In_z_ const char* s, _In_range_(0, endIndex) int startIndex, _In_range_(startIndex, HK_INT32_MAX) int endIndex ) const
{
    HK_ASSERT( 0x1fbc29bf, endIndex == HK_INT32_MAX, "Not implemented: hkStringBuf::indexOf(const char*) is only"
        " implemented for the default value endIndex==HK_INT32_MAX (no end limit)." );
    if ( startIndex < m_string.getSize() )
    {
        if ( const char* found = hkString::strStr( m_string.begin() + startIndex, s ) )
        {
            return hkGetByteOffsetInt( m_string.begin(), found );
        }
    }
    return -1;
}

int hkStringBuf::indexOfCase( _In_z_ const char* needle ) const
{
    // naive n^2 implementation
    const char* haystack = m_string.begin();
    for( int hi = 0; haystack[hi]; ++hi )
    {
        int ni;
        for( ni = 0; needle[ni]; ++ni )
        {
            if( tolower(haystack[hi+ni]) != tolower(needle[ni]) )
            {
                break;
            }
        }
        if( needle[ni] == 0 )
        {
            return hi;
        }
    }
//  hkStringBuf haystack = *this; haystack.lowerCase();
//  hkStringBuf needle = s; needle.lowerCase();
//  if( const char* found = hkString::strStr( haystack.cString(), needle.cString() ) )
//  {
//      return int(hkUlong(found - haystack.cString()));
//  }
    return -1;
}

int hkStringBuf::lastIndexOf (char c, _In_range_(0, end) int start, _In_range_(start, HK_INT32_MAX) int end) const
{
    if( end > getLength() )
    {
        end = getLength();
    }
    for(int i = end - 1; i >= start ; --i)
    {
        if( m_string[i] == c )
        {
            return i;
        }
    }
    return -1;
}

int hkStringBuf::lastIndexOf(_In_z_ const char* needle, _In_range_(0, endIndex) int startIndex, _In_range_(startIndex, HK_INT32_MAX) int endIndex) const
{
    
    HK_ASSERT_NO_MSG(0x1fbc29bf, endIndex == HK_INT32_MAX);
    int ret = -1;
    const char* haystack = m_string.begin();
    while( const char* found = hkString::strStr(haystack, needle) )
    {
        ret = int(hkUlong( found - m_string.begin() ));
        haystack = found+1;
    }
    return ret;
}

bool hkStringBuf::isEmpty() const
{
    const char* t = cString();
    return t[0] == '\0';
}

int hkStringBuf::compareTo(_In_z_ const char* s) const
{
    const char* t = cString();
    if (s) // both non null
    {
        return hkString::strCmp(t, s);
    }
    return 1; // null is "less" than all other non null strings
}

int hkStringBuf::compareToIgnoreCase(_In_z_ const char* s) const
{
    const char* t = cString();
    if (s) // both non null
    {
        return hkString::strCasecmp(t, s);
    }
    return 1; // null is "less" than all other non null strings
}

int hkStringBuf::compareTo(const hkStringBuf& s) const
{
    int min = hkMath::min2( getLength(), s.getLength() );
    return hkString::memCmp( cString(), s.cString(), min + 1 );
}

bool hkStringBuf::operator< (_In_z_ const char* s) const
{
    return compareTo(s) < 0;
}

bool hkStringBuf::operator> (_In_z_ const char* s) const
{
    return compareTo(s) > 0;
}

bool hkStringBuf::operator==(_In_z_ const char* s) const
{
    return compareTo(s) == 0;
}

bool hkStringBuf::operator!=(_In_z_ const char* s) const
{
    return compareTo(s) != 0;
}

bool hkStringBuf::startsWith(_In_z_ const char* s) const
{
    int i;
    for( i = 0; m_string[i] && s[i]; ++i )
    {
        if( m_string[i] != s[i] )
        {
            return false;
        }
    }
    return s[i] == '\0';
}

bool hkStringBuf::startsWithCase(_In_z_ const char* s) const
{
    for( int i = 0; m_string[i] && s[i]; ++i )
    {
        if( tolower(m_string[i]) != tolower(s[i]) )
        {
            return false;
        }
    }
    return true;
}

bool hkStringBuf::endsWith(_In_z_ const char* s) const
{
    int sl = hkString::strLen(s);
    if( sl > getLength() )
    {
        return false;
    }
    int start = getLength() - sl;

    for( int i = 0; i < sl; ++i )
    {
        if( m_string[start+i] != s[i] )
        {
            return false;
        }
    }
    return true;
}

bool hkStringBuf::endsWithCase(_In_z_ const char* s) const
{
    int sl = hkString::strLen(s);
    if( sl > getLength() )
    {
        return false;
    }
    int start = getLength() - sl;

    for( int i = 0; i < sl; ++i )
    {
        if( tolower(m_string[start+i]) != tolower(s[i]) )
        {
            return false;
        }
    }
    return true;
}

int hkStringBuf::split( int sep, hkArray<hkCString>::Temp& bits )
{
    int cur = 0;
    bits.pushBack( m_string.begin() );
    while(1)
    {
        int c = indexOf(char(sep), cur);
        if( c >= 0 )
        {
            m_string[c] = '\0';
            cur = c + 1;
            bits.pushBack( m_string.begin()+cur );
        }
        else
        {
            break;
        }
    }
    return bits.getSize();
}

void hkStringBuf::clear()
{
    setLength(0);
}

hkStringBuf& hkStringBuf::printf(_Printf_format_string_ const char* format, ...)
{
    va_list args;
    va_start( args, format );
    vprintf( format, args );
    va_end( args );

    return *this;
}

hkStringBuf& hkStringBuf::vprintf(_Printf_format_string_ const char* format, va_list hkargs)
{
    while(1)
    {
        va_list copy;
        hk_va_copy( copy, hkargs );
        int size = m_string.getCapacity();
        int nchars = hkString::vsnPrintf( m_string.begin(), size, size - 1, format, copy );
        va_end( copy );

        if( nchars >= 0 && nchars < size )
        {
            // usual case, it worked. update length
            setLength( nchars );
            break;
        }
        else if( nchars < 0 )
        {
            // there was not enough room, double capacity
            setLength( size*2 > 255 ? size*2 : 255 );
        }
        else
        {
            // there was not enough room and we were told how much
            // was needed (not including \0)
            setLength( nchars );
        }
    }
    return *this;
}

hkStringBuf& hkStringBuf::appendPrintf(_Printf_format_string_ const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    appendVprintf(fmt, args);
    va_end(args);
    return *this;
}

hkStringBuf& hkStringBuf::appendVprintf(_Printf_format_string_ const char* format, va_list args)
{
    hkStringBuf s;
    s.vprintf(format, args);
    append(s);
    return *this;
}

hkStringBuf& hkStringBuf::operator+= (_In_opt_z_ const char* s)
{
    if ( s != HK_NULL )
    {
        int otherLen = hkString::strLen(s);
        m_string.insertAt( m_string.getSize()-1, s, otherLen);
    }
    return *this;
}

hkStringBuf& hkStringBuf::appendJoin(_In_z_ const char* s0,
    _In_opt_z_ const char* s1,
    _In_opt_z_ const char* s2,
    _In_opt_z_ const char* s3,
    _In_opt_z_ const char* s4,
    _In_opt_z_ const char* s5)
{
    int len[6] = {};
    const char* ptr[6+1] = { s0, s1, s2, s3, s4, s5, HK_NULL }; // last one is sentinel
    int origLen = getLength();
    int totalLen = origLen;
    for( int i = 0; ptr[i] != HK_NULL; ++i )
    {
        len[i] = hkString::strLen(ptr[i]);
        totalLen += len[i];
    }
    setLength( totalLen );
    int dst = origLen;
    for( int i = 0; ptr[i] != HK_NULL; ++i )
    {
        hkString::memMove(m_string.begin() + dst, ptr[i], len[i]);
        dst += len[i];
    }
    return *this;
}

hkStringBuf& hkStringBuf::setJoin(_In_z_ const char* s0,
    _In_opt_z_ const char* s1,
    _In_opt_z_ const char* s2,
    _In_opt_z_ const char* s3,
    _In_opt_z_ const char* s4,
    _In_opt_z_ const char* s5)
{
    clear();
    appendJoin(s0,s1,s2,s3,s4,s5);
    return *this;
}

hkStringBuf& hkStringBuf::chompStart(int n)
{
    n = hkMath::min2(n, getLength());
    if( n > 0 )
    {
        m_string.removeAtAndCopy(0, n);
    }
    return *this;
}

hkStringBuf& hkStringBuf::chompEnd(int n)
{
    if( n > 0 )
    {
        setLength( hkMath::max2(0, getLength()-n) );
    }
    return *this;
}

hkStringBuf& hkStringBuf::slice(int startOffset, int length)
{
    HK_ASSERT_NO_MSG(0x79e7a47c, startOffset+length <= getLength());
    HK_ASSERT_NO_MSG(0x79e7a47d, length >= 0);
    if( startOffset != 0 )
    {
        hkMemUtil::memMove( m_string.begin(), m_string.begin()+startOffset, length );
    }
    setLength(length);
    return *this;
}

hkStringBuf& hkStringBuf::set(_In_reads_z_(len) const char* s, int len)
{
    HK_ASSERT_SLOW_NO_MSG(0x75a34527, aliasSafe(s,m_string.begin(), m_string.getSize()));
    if( len < 0 )
    {
        len = hkString::strLen(s);
    }
    setLength(len);
    hkString::memMove(m_string.begin(), s, len);
    return *this;
}

hkStringBuf& hkStringBuf::append(_In_reads_z_(len) const char* s, int len)
{
    if ( s != HK_NULL )
    {
        HK_ASSERT_SLOW_NO_MSG(0x75a34527, aliasSafe(s,m_string.begin(), m_string.getSize()));
        if( len < 0 )
        {
            len = hkString::strLen(s);
        }

        int oldLen = getLength();
        setLength( oldLen + len);
        hkString::memMove(m_string.begin() + oldLen, s, len);
    }
    return *this;
}

hkStringBuf& hkStringBuf::appendRepeat(int repeat, char c)
{
    if ( repeat >= 1 )
    {
        m_string.popBack();
        m_string.reserve( m_string.getSize()+repeat+1 );
        m_string.setSize( m_string.getSize()+repeat, c);
        m_string.pushBackUnchecked(0);
    }
    return *this;
}

hkStringBuf& hkStringBuf::prepend(_In_reads_z_(len) const char* s, int len)
{
    insert(0, s, len);
    return *this;
}

hkStringBuf& hkStringBuf::insert(_In_range_(>=, 0) int pos, _In_reads_opt_z_(len) const char* s, int len)
{
    if ( s != HK_NULL )
    {
        if(len<0)
        {
            len = hkString::strLen(s);
        }
        m_string.insertAt(pos, s, len);
    }
    return *this;
}

hkStringBuf& hkStringBuf::pathBasename()
{
    int lastSlash = hkMath::max2( lastIndexOf('\\'), lastIndexOf('/') );
    if( lastSlash >= 0 )
    {
        chompStart(lastSlash+1);
    }
    return *this;
}

hkStringBuf& hkStringBuf::pathDirname()
{
    int lastSlash = hkMath::max2( lastIndexOf('\\'), lastIndexOf('/') );
    if( lastSlash >= 0 )
    {
        slice(0, lastSlash);
    }
    else
    {
        clear();
    }
    return *this;
}

hkStringBuf& hkStringBuf::pathChildname()
{
    int firstNonSlashIndex = 0;
    int firstSlashIndex;

    do
    {
        // Doing min comparison as unsigned so that -1 gets treated as max value.
        firstSlashIndex = static_cast<int>(hkMath::min2<unsigned>(indexOf('\\', firstNonSlashIndex), indexOf('/', firstNonSlashIndex)));
        // Since we want to skip leading slashes, we stop when either the first slash isn't the assumed first non-slash character
        // or when we reach the end of the string (which happens if all characters are slashes)
    } while (firstSlashIndex == firstNonSlashIndex && ++firstNonSlashIndex < getLength());

    if (firstSlashIndex >= 0)
    {
        chompStart(firstSlashIndex + 1);
    }
    else
    {
        clear();
    }

    return *this;
}

hkStringBuf& hkStringBuf::pathNormalize()
{
    // Path with only '/'
    hkStringBuf path(*this);
    path.replace( '\\', '/', REPLACE_ALL );
    const char* startOfResult = path.startsWith("//") ? "//"
        : path.startsWith("/") ? "/"
        : "";
    hkInplaceArray<hkCString, 16>::Temp oldBits;
    path.split('/', oldBits);

    hkInplaceArray<hkCString, 16>::Temp newBits;
    newBits.reserve(oldBits.getSize());

    int numNamedParts = 0;
    for( int i = 0; i < oldBits.getSize(); ++i )
    {
        if( hkString::strCmp("..", oldBits[i]) == 0 )
        {
            if( numNamedParts )
            {
                newBits.popBack();
                numNamedParts -= 1;
            }
            else
            {
                newBits.pushBack(oldBits[i]);
            }
        }
        else if( hkString::strCmp(".", oldBits[i]) == 0 )
        {
            // ignore "." components
        }
        else if( hkString::strCmp(oldBits[i], "") != 0 )
        {
            numNamedParts += 1;
            newBits.pushBack(oldBits[i]);
        }
    }

    *this = startOfResult;
    for( int i = 0; i < newBits.getSize(); ++i )
    {
        this->pathAppend(newBits[i]);
    }
    return *this;
}

hkStringBuf& hkStringBuf::pathExtension()
{
    const int lastSlash = hkMath::max2( lastIndexOf('\\'), lastIndexOf('/') );
    const int lastDot = lastIndexOf('.', lastSlash + 1);
    if( lastDot != -1 )
    {
        chompStart(lastDot);
    }
    else
    {
        clear();
    }
    return *this;
}

hkStringBuf& hkStringBuf::pathWithoutExtension()
{
    const int lastSlash = hkMath::max2( lastIndexOf('\\'), lastIndexOf('/') );
    int lastDot = lastIndexOf('.', lastSlash + 1);
    if( lastDot != -1 )
    {
        setLength(lastDot);
    }
    return *this;
}

hkStringBuf& hkStringBuf::pathAppend(_In_z_ const char* p0, _In_opt_z_ const char* p1, _In_opt_z_ const char* p2)
{
    if (isEmpty())
    {
        // Appending a path to an empty string should take the path as is as the
        // append code below will otherwise destroy e.g. network paths.
        *this = p0;
        if (p1 != HK_NULL)
        {
            pathAppend(p1, p2);
        }
        return *this;
    }

    const char* parts[] = { p0, p1, p2, HK_NULL }; // last one is a sentinel
    m_string.popBack();
    // Don't add a leading / if we're empty
    bool needSlash = m_string.getSize() && m_string.back() != '/';
    for( int i = 0; parts[i] != HK_NULL; ++i )
    {
        const char* p = parts[i];
        if( p && p[0] ) // skip empty strings
        {
            while( p[0] == '/' )
            {
                p += 1;
            }

            if( int len = hkString::strLen(p) )
            {
                while( len && p[len-1] == '/' )
                {
                    len -= 1;
                }
                if( len && needSlash )
                {
                    m_string.pushBack('/');
                }
                m_string.append(p, len);
            }
            needSlash = true;
        }
    }
    m_string.pushBack(0);
    return *this;
}

hkResult hkStringBuf::pathRelativeTo(_In_z_ const char* root)
{
    pathNormalize();

    hkStringBuf rootString(root);
    rootString.pathNormalize();

    hkInplaceArray<hkStringView, 16> thisParts;
    hkInplaceArray<hkStringView, 16> rootParts;
    hkString::tokenize(*this, "/", HK_NULL, thisParts);
    hkString::tokenize(rootString, "/", HK_NULL, rootParts);

    const int total = hkMath::min2(thisParts.getSize(), rootParts.getSize());
    int numInCommon = 0;

    // find how many leading parts we have in common
    for (; numInCommon < total; ++numInCommon)
    {
        
        if (thisParts[numInCommon].compareLower(rootParts[numInCommon]) != 0 )
            break;
    }
    if (numInCommon == 0) // no common root
    {
        return HK_FAILURE;
    }

    ///Edge case: paths are equal.  This is inherently ambiguous:
    ///Problem: If we have two paths that are equal, say, "a:/b/c/" and "a:/b/c/" this is easy to understand.
    ///         For files, say, "a:/b/c.txt", it seems like the correct answer is "c.txt."
    ///         But in the case of "a:/b/c" it is ambiguous if c is a directory or a file
    ///Solution:
    ///         If the paths are equal, "" will be returned.

    hkStringBuf result;

    // go up to the common prefix
    for (auto j = numInCommon; j < rootParts.getSize(); ++j)
    {
        result.append("../");
    }

    // and back down to the target
    for (auto j = numInCommon; j < thisParts.getSize(); ++j)
    {
        result.append(thisParts[j].begin(), thisParts[j].getSize());
        result.append("/");
    }

    //Result is either empty or ends with /
    result.chompEnd(1);
    *this = result;

    return HK_SUCCESS;
}

bool hkStringBuf::replace(char from, char to, ReplaceType rt)
{
    bool replaced = false;
    for( int i = 0; i < getLength(); ++i )
    {
        if( m_string[i] == from )
        {
            m_string[i] = to;
            replaced = true;
            if( rt == REPLACE_ONE )
            {
                break;
            }
        }
    }
    return replaced;
}

bool hkStringBuf::replace(_In_z_ const char* from, _In_z_ const char* to, ReplaceType rtype)
{
    if ( !from ) // trying to replace nothing always fails
    {
        return false;
    }

    const int fromLen = hkString::strLen( from );
    if ( fromLen == 0 ) // trying to replace nothing always fails
    {
        return false;
    }

    if ( !to )
    {
        to = "";
    }
    const int toLen = hkString::strLen( to );

    bool replaced = false;
    if ( toLen > fromLen ) // replacement longer, reallocate a new buffer
    {
        hkStringBuf tmpStr = *this;
        clear();
        int tmpIdx = 0;
        for( ; ; )
        {
            int n = tmpStr.indexOf(from, tmpIdx);
            if( n >= 0 )
            {
                replaced = true;
                append( tmpStr.cString()+tmpIdx, n-tmpIdx );
                append( to, toLen );
                tmpIdx = n+fromLen;
                if( rtype == REPLACE_ONE )
                {
                    break;
                }
            }
            else
            {
                break;
            }
        }

        append( tmpStr.cString()+tmpIdx, tmpStr.getLength()-tmpIdx );
    }
    else // replacement same or shorter, don't allocate any buffer
    {
        char* self = m_string.begin();
        int readFrom=0, searchFrom=0, writeTo=0, tgtIndex;
        while(((tgtIndex = indexOf(from, searchFrom)) != -1))
        {
            while(readFrom < tgtIndex) 
            {
                self[writeTo++] = self [readFrom++];
            }
            for(int a=0; a<toLen; a++) 
            {
                self [writeTo++] = to [a];
            }
            readFrom += fromLen;
            searchFrom = tgtIndex + fromLen;
            replaced = true;
            if( rtype == REPLACE_ONE)
                break;
        }
        while(readFrom < getLength())
        {
            self [writeTo++] = self [readFrom++];
        }
        self [writeTo] = '\0';
        setLength(writeTo);
    }
    return replaced;
}

hkStringBuf& hkStringBuf::lowerCase()
{
    for( int i = 0; i < getLength(); ++i )
    {
        m_string[i] = tolower(m_string[i]);
    }
    return *this;
}

hkStringBuf& hkStringBuf::upperCase()
{
    for( int i = 0; i < getLength(); ++i )
    {
        m_string[i] = toupper(m_string[i]);
    }
    return *this;
}

hkArray<char>::Temp& hkStringBuf::getArray()
{
    return m_string;
}

int hkStringBuf::formatL(_In_opt_z_ const char* format, hkVarArgs::VaTypes argTypes, ... )
{
    hkVarArgs::FixedArray<16> args;
    HK_VARARGS_UNPACK(args, argTypes);
    return formatV(format, args);
}

int hkStringBuf::appendFormatL(_In_opt_z_ const char* format, hkVarArgs::VaTypes argTypes, ... )
{
    hkVarArgs::FixedArray<16> args;
    HK_VARARGS_UNPACK(args, argTypes);
    return appendFormatV(format, args);
}

int hkStringBuf::formatV(_In_opt_z_ const char* format, const hkVarArgs::Vector& args )
{
    this->clear();
    return appendFormatV(format, args);
}

namespace
{
    // Parse format specifier {[num][,width][:extra]} and append formatted item
    // - spec.begin() points to the first character after '{'
    // - spec.end() points to the '}' ending the specifier string
    // References:
    // > String.Format        : http://msdn.microsoft.com/en-us/library/system.string.format.aspx?ppud=4
    // > Composite formatting : http://msdn.microsoft.com/en-us/library/txafckwd.aspx
    HK_INLINE void HK_CALL parseAndAppendSingle( const hkStringView& spec, const hkVarArgs::Vector& args, int implicitArgumentNumber, hkStringBuf& strBufOut, _Out_ int* argumentNumberOut = HK_NULL )
    {
        const char* c = spec.begin();
        const char* const end = spec.end();

        // 1. Parse optional 'num' block
        int num;
        {
            const char* const s = c;
            while ( ( c < end ) && ( *c != ',' ) && ( *c != ':' ) )
            {
                HK_ASSERT_NO_MSG( 0x608ac196, *c != '{' ); // This should have been handled before
                HK_ASSERT_NO_MSG( 0x608ac197, *c != '}' ); // This should have been handled before
                ++c;
            }
            const hkStringView snum = hkStringView( s, c );
            num = ( snum ? getInt( snum ) : implicitArgumentNumber ); // Missing num means "next arg", i.e. "{} {}".format(foo,bar) == "{0} {1}".format(foo,bar)
            if ( argumentNumberOut )
            {
                *argumentNumberOut = num;
            }
        }

        // 2. Parse optional 'width' block
        int width = 0; // getInt("")
        char fillCharacter = ' '; // Character to fill width with
        if ( c < end )
        {
            if ( *c == ',' )
            {
                ++c;
                const char* const s = c;
                if(c < end && *c == '0')
                {
                    // Width specifier begins with a 0 (e.g. ,08x) - probably want zero padding
                    fillCharacter = '0';
                }

                while ( ( c < end ) && ( *c != ':' ) )
                {
                    HK_ASSERT_NO_MSG( 0x608ac196, *c != '{' ); // This should have been handled before
                    HK_ASSERT_NO_MSG( 0x608ac197, *c != '}' ); // This should have been handled before
                    HK_ASSERT( 0x608ac198, *c != ',', "Syntax error: found a second ',' character in 'width' block of format specifier. Syntax: {{[num][,width][:extra]}}." );
                    
                    ++c;
                }
                width = getInt( hkStringView( s, c ) );
            }
        }

        // 3. Parse optional 'extra' block
        
        hkStringView sextra;
        if ( c < end )
        {
            if ( *c == ':' )
            {
                ++c;
                const char* const s = c;
                while ( c < end )
                {
                    HK_ASSERT_NO_MSG( 0x608ac196, *c != '{' ); // This should have been handled before
                    HK_ASSERT_NO_MSG( 0x608ac197, *c != '}' ); // This should have been handled before
                    ++c;
                }
                sextra = hkStringView( s, c );
            }
        }

        // 4. Append formatted result to hkStringBuf instance
        {
            const hkReflect::Var& var = args[num];
            if ( var.getAddress() == HK_NULL )
            {
                strBufOut.append( "(NullVar)" );
                return;
            }

            // Format argument itself with 'extra'
            hkStringBuf tmp;
            var.toString( tmp, sextra );

            // Append formatted argument to "strBufOut" with optional padding
            // - width > 0 means right aligned
            // - width < 0 means left aligned
            // - if abs(width) < actual_width, the actual width is used (no truncating)
            const int actualWidth = hkMath::max2<int>( 0, tmp.getLength() );
            const int numRepeat = hkMath::abs( width ) - actualWidth;
            if ( width > 0 )
            {
                strBufOut.appendRepeat( numRepeat, fillCharacter ); // fill left
                strBufOut.append( tmp );
            }
            else if ( width < 0 )
            {
                strBufOut.append( tmp );
                strBufOut.appendRepeat( numRepeat, fillCharacter ); // fill right
            }
            else
            {
                strBufOut.append( tmp );
            }
        }
    }
}

int hkStringBuf::appendFormatV(_In_opt_z_ const char* format, const hkVarArgs::Vector& args)
{
    if ( !format )
    {
        return 0;
    }

#if defined(HK_DEBUG_SLOW)
    hkInplaceArray<bool, 10> argUsed;
    argUsed.setSize( (int)args.getSize(), false );
#endif

    // How this works - The loop here extracts the {} placeholders and forwards to
    // parseAndAppendSingle which further breaks the placeholder into {[INDEX][,WIDTH][:EXTRA]}
    // and eventually ends up calling "toString" on each argument.

    const char* pstart = format; // Start of current block pending append()
    const char* pc = format; // Current character in format string
    const char* pspec = HK_NULL; // Beginning of format specification (just after '{'), if any
    bool hasFirstCloseCurly = false; // Already saw '}' character just before
    int implicitArgumentNumber = 0;
    while ( true )
    {
        HK_ASSERT_NO_MSG( 0x608ac192, pc >= pstart );

        switch ( *pc )
        {
        case '{':
            if (hasFirstCloseCurly)
            {
                HK_ASSERT(0x608ac194, 0, "Syntax error: found '}}' character outside format string. Use escaped '}}}}' to get the '}}' character itself.");
                // Syntax error: append the rest of the raw string and return
                this->append(pstart);
                return 0;
            }
            if ( pspec )
            {
                // Escaped: '{{' -> '{'
                this->append( pstart, int( pc - pstart ) ); // here (pc > pstart) always
                ++pc; // skip second one
                pstart = pc;
                pspec = HK_NULL;
            }
            else // !pspec
            {
                // Possibly start of format '{...}', or escaped curly '{{'
                ++pc;
                pspec = pc;
            }
            break;

        case '}':
            if ( hasFirstCloseCurly )
            {
                // Found '}}'
                if ( pspec )
                {
                    // Cannot have '}' inside format string
                    HK_ASSERT( 0x608ac193, 0, "Syntax error: found '}}}}' escaped character inside format string." );
                    // Syntax error: append the rest of the raw string and return
                    this->append(pstart);
                    return 0;
                }
                else
                {
                    // Not inside format string; escaped '}}' -> '}'
                    this->append( pstart, int( pc - pstart ) ); // here (pc > pstart) always
                    ++pc; // skip second one
                    pstart = pc;
                    hasFirstCloseCurly = false;
                }
            }
            else // !hasFirstCloseCurly
            {
                if ( pspec )
                {
                    // First '}' is always the closing one; cannot have '}' inside the format string.

                    // Append pending raw string up to (but excluding) the opening '{'
                    if ( pspec > pstart + 1 )
                    {
                        this->append( pstart, int( pspec - pstart - 1 ) );
                    }

                    // Parse this spec string and format the arg
                    int argumentNumber = -1;
                    parseAndAppendSingle( hkStringView( pspec, pc ), args, implicitArgumentNumber, *this, &argumentNumber );
#if defined(HK_DEBUG_SLOW)
                    HK_ASSERT( 0x608ac199, argumentNumber < argUsed.getSize(), "Missing argument #{} for format string '{}'.", argumentNumber, format );
                    argUsed[argumentNumber] = true;
#endif
                    implicitArgumentNumber = argumentNumber + 1;

                    ++pc;
                    pstart = pc; // after closing '}'
                    pspec = HK_NULL;
                    hasFirstCloseCurly = false;
                }
                else
                {
                    // Possibly escaped curly '}}', or syntax error.
                    // Syntax error will be caught in the default case below or the '{' case above.
                    ++pc;
                    hasFirstCloseCurly = true;
                }
            }
            break;
        default:
            if (hasFirstCloseCurly)
            {
                HK_ASSERT(0x608ac194, 0, "Syntax error: found '}}' character outside format string. Use escaped '}}}}' to get the '}}' character itself.");
                // Syntax error: append the rest of the raw string and return
                this->append(pstart);
                return 0;
            }
            if ( *pc == '\0' )
            {
                HK_ASSERT( 0x608ac195, !pspec, "Syntax error: missing '}}' to close format specifier. Reached end of format string." );
                if ( pc > pstart )
                {
                    this->append( pstart, int( pc - pstart ) );
                }
#if defined(HK_DEBUG_SLOW)
                int index = 0;
                for ( auto const& b : argUsed )
                {
                    HK_ASSERT( 0x608ac19a, b, "Argument #{} is unused in format string '{}'.", index, format );
                    ++index;
                }
#endif
                return 0;
            }
            else
            {
                ++pc;
            }
            break;
        }
    }

    HK_UNREACHABLE(0x71e5adfb, "Internal error in hkStringBuf");
}

hkStringBuf& hkStringBuf::trimEnd()
{
    int numCharactersToRemove = 0;
    while ((numCharactersToRemove < (m_string.getSize() - 1)) && isWhiteSpace(m_string[m_string.getSize() - 2 - numCharactersToRemove]))
    {
        numCharactersToRemove++;
    }

    return chompEnd(numCharactersToRemove);
}

hkStringBuf& hkStringBuf::trimStart()
{
    int numCharactersToRemove = 0;
    while (isWhiteSpace(m_string[numCharactersToRemove]) && (numCharactersToRemove < (m_string.getSize() - 1)))
    {
        numCharactersToRemove++;
    }

    return chompStart(numCharactersToRemove);
}

hkResult hkStringBuf::trimEndFromLast(char c)
{
    int numCharactersToRemove = 0;
    while (isWhiteSpace(m_string[m_string.getSize() - 1 - numCharactersToRemove]) && (numCharactersToRemove < (m_string.getSize() - 1)))
    {
        numCharactersToRemove++;
    }

    if ((numCharactersToRemove < (m_string.getSize() - 2)) && (m_string[m_string.getSize() - 2 - numCharactersToRemove] == c))
    {
        chompEnd(numCharactersToRemove + 1);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

hkResult hkStringBuf::trimStartToFirst(char c)
{
    int numCharactersToRemove = 0;
    while (isWhiteSpace(m_string[numCharactersToRemove]) && (numCharactersToRemove < (m_string.getSize() - 1)))
    {
        numCharactersToRemove++;
    }

    if((numCharactersToRemove < (m_string.getSize() - 2)) && (m_string[numCharactersToRemove] == c))
    {
        chompStart(numCharactersToRemove + 1);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

bool operator==( const hkStringBuf& lhs, const hkStringBuf& rhs )
{
    if ( lhs.getLength() == rhs.getLength() )
    {
        return hkString::memCmp( lhs.cString(), rhs.cString(), lhs.getLength() ) == 0;
    }
    return false;
}

#if 0
struct HK_EXPORT_COMMON BoyerMooreHorspool
{
    BoyerMooreHorspool( const char* needle, int needleLen )
        : m_needle(needle), m_needleLen(needleLen >= 0 ? needleLen : hkString::strLen(needle) )
    {
        for( int i = 0; i < HK_COUNT_OF(m_badCharShift); ++i )
        {
            m_badCharShift[i] = needleLen;
        }
        hkString::memSet(m_badCharShift, , sizeof(m_badCharShift));
        for( int i = 0; i < m_needleLen; ++i )
        {
            m_badCharShift[ m_needle[i] ]
        }
    }
    int m_badCharShift[256];
    const char* m_needle;
    int m_needleLen;
};
#endif


struct hkStringBuf::StringImpl : public hkReflect::Detail::StringImpl
{
    StringImpl() { /* keep clang happy */ }

    static hkStringBuf& cast(_In_ void* ptr)
    {
        return *(static_cast<hkStringBuf*>(ptr));
    }

    static const hkStringBuf& cast(_In_ const void* ptr)
    {
        return *(static_cast<const hkStringBuf*>(ptr));
    }

    virtual hkResult setValue(_In_ void* addr, _In_ const hkReflect::StringType*, hkReflect::StringValue newValue) const HK_OVERRIDE
    {
        cast(addr) = newValue;
        return HK_SUCCESS;
    }

    virtual hkResult getValue(_In_ const void* addr, _In_ const hkReflect::StringType* type, _Out_ hkReflect::StringValue* val) const HK_OVERRIDE
    {
        *val = cast(addr).cString();
        return HK_SUCCESS;
    }
};
const hkStringBuf::StringImpl hkStringBuf::s_impl;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
