// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#if defined(__HAVOK_PARSER__)
typedef char* va_list;
#elif defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_IOS)
#include <Common/Base/Fwd/hkcstdarg.h>
#endif

/// String and buffer functions normally found in libc.
namespace hkString
{
        /// Parameter to hkString::replace functions
    enum ReplaceType
    {
        REPLACE_ONE,
        REPLACE_ALL
    };

        /// Return the upper case of character c
    HK_EXPORT_COMMON char HK_CALL toUpper( char c );
        /// Return the lower case of character c
    HK_EXPORT_COMMON char HK_CALL toLower( char c );

        /// Returns whether the given string is null or zero length.
    HK_EXPORT_COMMON bool HK_CALL isNullOrEmpty(_In_opt_z_ const char* s);

        
    HK_EXPORT_COMMON int HK_CALL snprintf(char* buf, int n, const char* fmt, ...);

        /// Equivalent to sprintf except at most n characters are written.
    HK_EXPORT_COMMON int HK_CALL snPrintf(_Out_writes_z_(n) _Out_bytecap_(bufBytes) char* buf,
        _In_range_(>, n) int bufBytes, int n, _Printf_format_string_ const char* fmt, ...);
        /// Equivalent to snprintf, but with a va_list of arguments.
    HK_EXPORT_COMMON int HK_CALL vsnPrintf(_Out_writes_z_(len) _Out_bytecap_(bufBytes) char* buf,
        _In_range_(>, len) int bufBytes, int len, _Printf_format_string_ const char* fmt, va_list hkargs);
        /// Printf formatting to a string buffer.
    HK_EXPORT_COMMON int HK_CALL sprintf(_Out_writes_z_(_Inexpressible_()) char* buf, _Printf_format_string_ const char* fmt, ...);

        /// Returns <=-1,0,>=1 if s1 is lexicographically less than, equal to or greater than s2.
    HK_EXPORT_COMMON int HK_CALL strCmp(_In_z_ const char* s1, _In_z_ const char* s2 );
        /// Returns <=-1,0,>=1 if s1 is lexicographically less than, equal to or greater than s2. Comparison is done using at most the first n characters.
    HK_EXPORT_COMMON int HK_CALL strNcmp(_In_z_ const char* s1, _In_z_ const char* s2, _In_range_(>, 0) int n);
        /// Returns <=-1,0,>=1 if s1 is lexicographically less than, equal to or greater than s2, ignoring case.
    HK_EXPORT_COMMON int HK_CALL strCasecmp(_In_z_ const char* s1, _In_z_ const char* s2 );
        /// Returns <=-1,0,>=1 if s1 is lexicographically less than, equal to or greater than s2, ignoring case. Comparison is done using at most the first n characters.
    HK_EXPORT_COMMON int HK_CALL strNcasecmp(_In_z_ const char* s1, _In_z_ const char* s2, _In_range_(>, 0) int n);
        /// Copy null terminated src into dst. dst must be large enough to hold src.
    HK_EXPORT_COMMON void HK_CALL strCpy(_Pre_cap_for_(src) _Out_bytecap_(dstBytes) char* dst, _In_range_(>, 0) int dstBytes, _In_z_ const char* src);
        /// Copy at most n characters of null terminated src into dst. DST must be large enough to hold src.
    HK_EXPORT_COMMON void HK_CALL strNcpy(_Out_writes_z_(n) _Out_bytecap_(dstBytes) char* dst, _In_range_(>, n) int dstBytes, _In_reads_z_(n) const char* src, _In_range_(>=, 0) int n);
        /// Return the length of null terminated string src.
    _Ret_range_(== , _String_length_(src))
    HK_EXPORT_COMMON int HK_CALL strLen(_In_z_ const char* src);

    template<int _size>
    HK_INLINE int HK_CALL vsnPrintf(_Out_writes_z_(n) char(&dst)[_size], _In_range_(>, 0) int n, _Printf_format_string_ const char* fmt, va_list hkargs);
    template<int _size>
    HK_INLINE void HK_CALL strCpy(_Pre_cap_for_(src) char (&dst)[_size], _In_z_ const char* src);
    template<int _size>
    HK_INLINE void HK_CALL strNcpy(_Out_writes_z_(n) char (&dst)[_size], _In_reads_z_(n) const char* src, _In_range_(>, 0) int n);

        /// Return an integer representing the string (signed, undefined for invalid input).
        /// If a base of 0 is specified atoi will attempt to determine the base from string prefix e.g. '0' for octal 'Ox' or 'OX' for hex
    HK_EXPORT_COMMON int HK_CALL atoi(_In_z_ const char* in, int base = 0);
        /// Return a 64-bit integer representing the string (signed, undefined for invalid input).
        /// If a base of 0 is specified atoll will attempt to determine the base from string prefix e.g. '0' for octal 'Ox' or 'OX' for hex
    HK_EXPORT_COMMON hkInt64 HK_CALL atoll(_In_z_ const char* in, int base = 0);
        /// Return a 64 bits integer representing the string (unsigned, undefined for invalid input).
        /// If a base of 0 is specified atoll will attempt to determine the base from string prefix e.g. '0' for octal 'Ox' or 'OX' for hex
    HK_EXPORT_COMMON hkUint64 HK_CALL atoull(_In_z_ const char* in, int base = 0);
        /// Return a hkReal representing the string.
    HK_EXPORT_COMMON hkReal HK_CALL atof(_In_z_ const char* in);
        /// Return a hkDouble64 representing the string.
    HK_EXPORT_COMMON hkDouble64 HK_CALL atod(_In_z_ const char* in);

        /// Return the first occurrence of needle in haystack or null if not found.
    HK_EXPORT_COMMON _Ret_z_ const char* HK_CALL strStr(_In_z_ const char* haystack, _In_z_ const char* needle);
        /// Find all the occurrences of needle in haystack and put their start indices in the array.
        /// Return true if at least one was found.
    HK_EXPORT_COMMON bool HK_CALL findAllOccurrences(_In_z_ const char* haystack, _In_z_ const char* needle, hkArray<int>& indices, hkString::ReplaceType rtype);
        /// Return the first occurrence of needle in haystack or null if not found.
    HK_EXPORT_COMMON _Ret_z_ const char* HK_CALL strChr(_In_z_ const char* haystack, int needle);
        /// Return the last occurrence of needle in haystack or null if not found.
    HK_EXPORT_COMMON _Ret_z_ const char* HK_CALL strRchr(_In_z_ const char* haystack, int needle);

        /// Checks whether a string begins exactly with a given prefix.
    HK_EXPORT_COMMON bool HK_CALL beginsWith(_In_z_ const char* str, _In_z_ const char* prefix);
        /// Checks case insensitive whether a string begins with a given prefix.
    HK_EXPORT_COMMON bool HK_CALL beginsWithCase(_In_z_ const char* str, _In_z_ const char* prefix);
    /// Checks whether a string ends exactly with a given suffix.
    HK_EXPORT_COMMON bool HK_CALL endsWith(_In_z_ const char* str, _In_z_ const char* suffix);
    /// Checks case insensitive whether a string ends with a given suffix.
    HK_EXPORT_COMMON bool HK_CALL endsWithCase(_In_z_ const char* str, _In_z_ const char* suffix);
    /// Checks whether a string ends exactly with one of an array of given suffixes.
    HK_EXPORT_COMMON bool HK_CALL endsWithAny(_In_z_ const char* str, const hkArrayView<hkCString> suffixes);
    /// Checks case insensitive whether a string ends with one of an array of given suffixes.
    HK_EXPORT_COMMON bool HK_CALL endsWithAnyCase(_In_z_ const char* str, const hkArrayView<hkCString> suffixes);
        /// Finds the last occurrence of a given character in a string.
    HK_EXPORT_COMMON int HK_CALL lastIndexOf(_In_z_ const char* str, char c);
        /// Start and end are inclusive and exclusive ranges respectively.
    HK_EXPORT_COMMON int HK_CALL indexOf(_In_z_ const char* str, char c, _In_range_(0, endIndex) int startIndex=0, _In_range_(startIndex, HK_INT32_MAX) int endIndex=HK_INT32_MAX);

        /// Return a copy of string src. This copy is allocated using hkAllocate<char> and therefore it should be deallocated using hkDeallocate<char>
    HK_EXPORT_COMMON _Ret_z_ char* HK_CALL strDup(_In_z_ const char* src, hkMemoryAllocator& a);
        /// Return a copy of at most maxlen characters of src.
    HK_EXPORT_COMMON _Ret_z_ char* HK_CALL strNdup(_In_z_ const char* src, _In_range_(0, _String_length_(src)) int maxlen, hkMemoryAllocator& a);
        ///
    HK_EXPORT_COMMON void HK_CALL strFree(_In_z_ char* s, hkMemoryAllocator& a);
    HK_EXPORT_COMMON void HK_CALL strFree(_In_z_ char* s);

        /// Tokenize the string. Returns the start positions end lengths of the tokens.
    HK_EXPORT_COMMON void HK_CALL tokenize( _In_z_ const char* string, _In_z_ const char* seperators, _In_z_ const char* whitespace, hkArray<hkStringView>& tokensOut);

        /// Return a copy of string src. This copy is allocated using hkAllocate<char> and therefore it should be deallocated using hkDeallocate<char>
    HK_EXPORT_COMMON _Ret_z_ char* HK_CALL strDup(_In_z_ const char* src);
        /// Return a copy of at most maxlen characters of src.
    HK_EXPORT_COMMON _Ret_z_ char* HK_CALL strNdup(_In_reads_z_(maxlen) const char* src, _In_range_(>, 0) int maxlen);
        /// Change src to lower case in place.
    HK_EXPORT_COMMON _Ret_z_ char* HK_CALL strLwr(_Inout_z_ char* src);
        /// Change src to upper case in place.
    HK_EXPORT_COMMON _Ret_z_ char* HK_CALL strUpr(_Inout_z_ char* src);

        /// Copy n bytes of src into dst.
    HK_EXPORT_COMMON void HK_CALL memCpy(_Out_writes_bytes_all_(n) void* dst, _In_reads_bytes_(n) const void* src, _In_range_(>, 0) int n);
        /// Copy n words of src into dst.
    HK_EXPORT_COMMON HK_INLINE void HK_CALL memCpy4(_Out_writes_bytes_(numWords*4) void* dst, _In_reads_bytes_(numWords*4) const void* src, _In_range_(0, HK_INT32_MAX/4) int numWords);
        /// Copy n quad words of src into dst.
    HK_EXPORT_COMMON HK_INLINE void HK_CALL memCpy16(_Out_writes_bytes_(numQuads*16) void* dst, _In_reads_bytes_(numQuads*16) const void* src, _In_range_(0, HK_INT32_MAX/16) int numQuads);

        /// Super fast copy of a constant size less than 128 bytes
    template<int size>
    HK_INLINE void HK_CALL memCpy16(_Out_writes_bytes_(size) void* dst, const void* src);

        /// Copy n quad words of src into dst. n must be greater than 0
    HK_EXPORT_COMMON HK_INLINE void HK_CALL memCpy16NonEmpty(_Out_writes_bytes_(numQuads*16) void* dst, _In_reads_bytes_(numQuads*16) const void* src, _In_range_(0, HK_INT32_MAX/16) int numQuads);
        /// Copy 128 bytes of src into dst.
    HK_EXPORT_COMMON HK_INLINE void HK_CALL memCpy128(_Out_writes_bytes_(128) void* dst, _In_reads_bytes_(128) const void* src);
        /// Copy 256 bytes of src into dst.
    HK_EXPORT_COMMON HK_INLINE void HK_CALL memCpy256(_Out_writes_bytes_(256) void* dst, _In_reads_bytes_(256) const void* src);
        /// Copy n bytes of src into dst, possibly overlapping.
    HK_EXPORT_COMMON void HK_CALL memMove(_Out_writes_bytes_(n) void* dst, _In_reads_bytes_(n) const void* src, _In_range_(>, 0) int n);
        /// Set n bytes of dst to c.
    HK_EXPORT_COMMON void HK_CALL memSet(_Out_writes_bytes_(nBytes) void* dst, const int c, _In_range_(>, 0) int nBytes);

        /// Set n*4 bytes to c
    HK_EXPORT_COMMON HK_INLINE void HK_CALL memSet4(_Out_writes_bytes_(numWords*4) void* dst, const int value, int numWords);

        /// Set n*16 bytes to 0
    HK_EXPORT_COMMON HK_INLINE void HK_CALL memClear16(_Out_writes_bytes_(numQuads*16) void* dst, int numQuads);

        /// Clear a multiple of 128 bytes which are 128 byte aligned
    HK_EXPORT_COMMON void HK_CALL memClear128(_Out_writes_bytes_(numBytes) void* dst, int numBytes);

        /// Set numQuads*16 bytes to the quad at value
    HK_EXPORT_COMMON HK_INLINE void HK_CALL memSet16(_Out_writes_bytes_(numQuads*16) void* dst, _In_reads_bytes_(16) const void* value, _In_range_(>, 0) int numQuads);

        /// Set <size>/16 quads to the quad at src.
        /// Size must be positive and a multiple of 16
    template<int size>
    HK_INLINE void HK_CALL memSet16(_Out_writes_bytes_(size) void* dst, _In_reads_bytes_(size) const void* src);

        /// Returns a negative number,  0  or positive number  if n bytes of buf1 is less than, equal to or greater than those of buf2.
    HK_EXPORT_COMMON int HK_CALL memCmp(_In_reads_bytes_(n) const void* buf1, _In_reads_bytes_(n) const void* buf2, _In_range_(>, 0) int n);

        /// Returns a negative number,  0  or positive number  if n hkUint32 of buf1 is less than, equal to or greater than those of buf2.
    HK_EXPORT_COMMON HK_INLINE int HK_CALL memCmpUint32(_In_reads_(n) const hkUint32* buf1, _In_reads_(n) const hkUint32* buf2, _In_range_(>, 0) int n);
}

#if defined(HK_PLATFORM_WIN32)
#   define HK_PRINTF_FORMAT_INT64_SIZE  "I64"
#else
#   define HK_PRINTF_FORMAT_INT64_SIZE  "ll"
#endif

#   define HK_PRINTF_FORMAT_INT64       "%" HK_PRINTF_FORMAT_INT64_SIZE "i"
#   define HK_PRINTF_FORMAT_UINT64      "%" HK_PRINTF_FORMAT_INT64_SIZE "u"

#if HK_POINTER_SIZE==4
#   define HK_PRINTF_FORMAT_ULONG "%u"
#   define HK_PRINTF_FORMAT_LONG "%i"
#else
#   define HK_PRINTF_FORMAT_ULONG HK_PRINTF_FORMAT_UINT64
#   define HK_PRINTF_FORMAT_LONG HK_PRINTF_FORMAT_INT64
#endif

#include <Common/Base/Container/String/hkString.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
