// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0


#include <Common/Base/hkBase.h>
#include <Common/Base/Fwd/hkcctype.h>
#include <Common/Base/Fwd/hkcstdarg.h>
#include <Common/Base/Fwd/hkcstdio.h>
#include <Common/Base/Fwd/hkcstdlib.h>
#include <Common/Base/Fwd/hkcstring.h>

using namespace std;

// very small strings will likely cause reallocations
//static const int MINIMUM_STRING_CAPACITY = 64 - 1;

// Some platforms don't have vsnprintf, so we use vsprintf instead.
// vsprintf is not safe from overflows, so we include asserts.

// Win32 renames some standard string manipulation functions.
#if defined(HK_PLATFORM_WIN32)
#include <locale.h>
const ::_locale_t cLocale  = ::_create_locale(LC_ALL, "C");
#   define VSPRINTF(BUF,FMT,ARGS) ::_vsprintf_l(BUF,FMT, cLocale, ARGS)
#   define STRTOLL(STR,ENDPTR,BASE) ::_strtoi64_l(STR,ENDPTR,BASE, cLocale)
#   define STRTOULL(STR,ENDPTR,BASE) ::_strtoui64_l(STR,ENDPTR,BASE, cLocale)
#   define STRTOD(STR,ENDPTR) ::_strtod_l( STR,ENDPTR, cLocale)
#else
#   define VSPRINTF(BUF,FMT,ARGS) vsprintf(BUF, FMT, ARGS)
#   define STRTOLL(STR,ENDPTR,BASE) ::strtoll(STR,ENDPTR,BASE)
#   define STRTOULL(STR,ENDPTR,BASE) ::strtoull(STR,ENDPTR,BASE)
#   define STRTOD(STR,ENDPTR) strtod(STR, ENDPTR)
#endif

// all the static methods go first so that they may be inlined
// in the nonstatic ones further on.

char HK_CALL hkString::toUpper( char c )
{
    return (c>='a'&&c<='z')
            ? (char)(c - ('a' - 'A'))
            : c;
}
char HK_CALL hkString::toLower( char c )
{
    return (c>='A'&&c<='Z')
            ? (char)(c + ('a' - 'A'))
            : c;
}

bool HK_CALL hkString::isNullOrEmpty(_In_opt_z_ const char* s)
{
    return (s == HK_NULL || s[0] == '\0');
}

int HK_CALL hkString::vsnPrintf(_Out_writes_z_(len) _Out_bytecap_(bufBytes) char* buf,
    _In_range_(>, len) int bufBytes, int len, _Printf_format_string_ const char* fmt, va_list args)
{
#ifdef HK_PLATFORM_WIN32
    int ret = vsnprintf_s(buf, bufBytes, len, fmt, args);
#else
    int ret = ::vsnprintf(buf, bufBytes, fmt, args);
#endif

    return ret;
}


int HK_CALL hkString::snprintf(char* buf, int n, const char* fmt, ...)
{
    va_list vlist;
    va_start(vlist, fmt);
    int ret = hkString::vsnPrintf(buf, n, n, fmt, vlist);
    va_end(vlist);
    return ret;
}

int HK_CALL hkString::snPrintf(_Out_writes_z_(n) _Out_bytecap_(bufBytes) char* buf,
    _In_range_(>, n) int bufBytes, int n, _Printf_format_string_ const char* fmt, ...)
{
    va_list vlist;
    va_start(vlist, fmt);
    int ret = hkString::vsnPrintf(buf, bufBytes, n, fmt, vlist);
    va_end(vlist);
    return ret;
}

int HK_CALL hkString::sprintf(_Out_writes_z_(_Inexpressible_()) char* buf, _Printf_format_string_ const char* fmt, ...)
{
    va_list vlist;
    va_start(vlist, fmt);
    int ret = VSPRINTF(buf, fmt, vlist);
    va_end(vlist);
    return ret;
}

int HK_CALL hkString::strCmp( _In_z_ const char* a, _In_z_ const char* b )
{
    HK_ASSERT_NO_MSG(0x2540a587, a != HK_NULL);
    HK_ASSERT_NO_MSG(0x571b5b6a, b != HK_NULL);
    return strcmp(a,b);
}

int HK_CALL hkString::strNcmp( _In_z_ const char* a, _In_z_ const char* b, _In_range_(>, 0) int n )
{
    HK_ASSERT_NO_MSG(0x37ef1ca6, a != HK_NULL);
    HK_ASSERT_NO_MSG(0x68ef40e3, b != HK_NULL);
    return strncmp(a,b,(unsigned)n);
}

int HK_CALL hkString::strCasecmp( _In_z_ const char* a, _In_z_ const char* b )
{
    HK_ASSERT_NO_MSG(0x5cf1cfe9, a != HK_NULL);
    HK_ASSERT_NO_MSG(0x49243d55, b != HK_NULL);

    int i = 0;
    while (a[i] != '\0' || b[i] != '\0')
    {
        if (a[i] == '\0')
        {
            return -1;
        }
        else if (b[i] == '\0')
        {
            return 1;
        }

        const char lowerA = toLower(a[i]);
        const char lowerB = toLower(b[i]);
        if (lowerA < lowerB)
        {
            return -1;
        }
        else if (lowerA > lowerB)
        {
            return 1;
        }

        ++i;
    }

    return 0;
}

int HK_CALL hkString::strNcasecmp(_In_z_ const char* a, _In_z_ const char* b, _In_range_(> , 0) int n)
{
    HK_ASSERT_NO_MSG(0x7a31572d, a != HK_NULL);
    HK_ASSERT_NO_MSG(0x54725cb9, b != HK_NULL);
    HK_ASSERT_NO_MSG(0x7403b0c0, n >= 0);

    int i = 0;
    while ((a[i] != '\0' || b[i] != '\0') && i < n)
    {
        if (a[i] == '\0')
        {
            return -1;
        }
        else if (b[i] == '\0')
        {
            return 1;
        }

        const char lowerA = toLower(a[i]);
        const char lowerB = toLower(b[i]);
        if (lowerA < lowerB)
        {
            return -1;
        }
        else if (lowerA > lowerB)
        {
            return 1;
        }
        ++i;
    }

    return 0;
}

void HK_CALL hkString::strCpy(_Pre_cap_for_(src) _Out_bytecap_(dstBytes) char* dst, _In_range_(>, 0) int dstBytes, _In_z_ const char* src)
{
    HK_ASSERT_NO_MSG(0x49fa77f1, src != HK_NULL);
    HK_ASSERT_NO_MSG(0x7d3f002c, dst != HK_NULL);
#ifdef HK_PLATFORM_WIN32
    strcpy_s(dst, dstBytes, src);
#else
    strcpy(dst, src);
#endif
}

void HK_CALL hkString::strNcpy(_Out_writes_z_(n) _Out_bytecap_(dstBytes) char *dst, _In_range_(>, n) int dstBytes, _In_reads_z_(n) const char *src, _In_range_(>=, 0) int n)
{
    HK_ASSERT_NO_MSG(0x6e527462, src != HK_NULL || n == 0);
    HK_ASSERT_NO_MSG(0x67d37b1f, dst != HK_NULL || n == 0);
    if( n > 0 )
    {
#ifdef HK_PLATFORM_WIN32
        strncpy_s(dst, dstBytes, src, (unsigned)n);
#else
        strncpy(dst, src, (unsigned)n);
#endif
    }
}

_Ret_range_(==, _String_length_(src))
int HK_CALL hkString::strLen( _In_z_ const char* src )
{
    HK_ASSERT_NO_MSG(0x18be8938, src != HK_NULL);
    return (int) strlen(src);
}

int HK_CALL hkString::atoi( _In_z_ const char* in, int base)
{
    return strtoul(in, HK_NULL, base);
}

hkInt64 HK_CALL hkString::atoll( _In_z_ const char* in, int base)
{
    return STRTOLL(in, HK_NULL, base);
}

hkUint64 HK_CALL hkString::atoull( _In_z_ const char* in, int base)
{
    return STRTOULL(in, HK_NULL, base);
}

hkReal HK_CALL hkString::atof( _In_z_ const char* in )
{
    return (hkReal)STRTOD(in, HK_NULL);
}

hkDouble64 HK_CALL hkString::atod( _In_z_ const char* in )
{
    return STRTOD(in, HK_NULL);
}

_Ret_z_ const char* HK_CALL hkString::strStr(_In_z_ const char* haystack, _In_z_ const char* needle)
{
    return strstr(haystack, needle);
}

_Ret_z_ const char* HK_CALL hkString::strChr(_In_z_ const char* haystack, int needle)
{
    return strchr(haystack, needle);
}

_Ret_z_ const char* HK_CALL hkString::strRchr(_In_z_ const char* haystack, int needle)
{
    return strrchr(haystack, needle);
}

void HK_CALL hkString::tokenize( _In_z_ const char* string, _In_z_ const char* seperators, _In_z_ const char* whitespace, hkArray<hkStringView>& tokensOut)
{
    tokensOut.clear();
    const char* s = string;
    while( s[0] != 0 )
    {
        // remove leading white space
        if ( whitespace )
        {
            while( s[0] && strchr(whitespace, s[0]) )
            {
                s++;
            }
        }
        // find separator
        const char* p = s;
        while ( *p && 0 == strchr(seperators, *p ))
        {
            p++;
        }

        // remove trailing whitespace
        const char* end = p-1;
        if (whitespace)
        {
            while ( end>s && strchr(whitespace,*end))
            {
                end--;
            }
        }
        tokensOut.pushBack(hkStringView(s, (int)((hkUlong)(end - s)) + 1));

        s = (p[0]== 0) ? p : p+1;   // make sure we are not falling off the string.
    }
}

_Ret_z_ char* HK_CALL hkString::strDup(_In_z_ const char* src, hkMemoryAllocator& alloc)
{
    HK_ASSERT_NO_MSG(0x13e1d159, src != HK_NULL);
    int len = strLen(src) + 1;
    char* r = (char*)hkMemoryRouter::easyAlloc(alloc, len);
    memCpy(r, src, len);
    return r;
}

void HK_CALL hkString::strFree(_In_z_ char* s, hkMemoryAllocator& alloc)
{
    hkMemoryRouter::easyFree(alloc, s);
}

void HK_CALL hkString::strFree(_In_z_ char* s)
{
    hkString::strFree(s, *hkMemHeapAllocator());
}

_Ret_z_ char* HK_CALL hkString::strNdup(_In_z_ const char* src, _In_range_(0, _String_length_(src)) int maxlen, hkMemoryAllocator& alloc)
{
    HK_ASSERT_NO_MSG(0x2e27506f, src != HK_NULL);

    int len = 0; // can't use strLen, src may not be terminated
    while (len < maxlen && src[len++])
    {
    }
    char* r = (char*)hkMemoryRouter::easyAlloc(alloc, len+1);
    hkString::strNcpy(r, len+1, src, len);
    r[len] = 0;
    return r;
}

_Ret_z_ char* HK_CALL hkString::strDup(_In_z_ const char* src)
{
    return strDup(src, *hkMemHeapAllocator());
}

_Ret_z_ char* HK_CALL hkString::strNdup(_In_reads_z_(maxlen) const char* src, _In_range_(>, 0) int maxlen)
{
    return strNdup(src, maxlen, *hkMemHeapAllocator());
}

_Ret_z_ char* HK_CALL hkString::strLwr(_Inout_z_ char* s)
{
    HK_ASSERT_NO_MSG(0x3779fe34, s != HK_NULL);
    int i=0;
    while(s[i])
    {
        s[i] = (char)(toLower(s[i]));
        i++;
    }
    return s;
}

_Ret_z_ char* HK_CALL hkString::strUpr(_Inout_z_ char* s)
{
    HK_ASSERT_NO_MSG(0x128f807f, s != HK_NULL);
    int i=0;
    while(s[i])
    {
        s[i] = (char)(toUpper(s[i]));
        i++;
    }
    return s;
}

#if defined(HK_PLATFORM_WIIU)


#if 0 && (CAFE_OS_SDK_VERSION >= 20900)
#define CHECK_OSBLOCKMOVE(dst, src, nbytes, flush) OSBlockMove(dst, src, nbytes, flush)
#else
#define CHECK_OSBLOCKMOVE(dst, src, nbytes, flush) if(dst != src) { OSBlockMove(dst, src, nbytes, flush); } else
#endif
void HK_CALL hkString::memCpy(_Out_writes_bytes_all_(n) void* dst, _In_reads_bytes_(n) const void* src, _In_range_(>, 0) int n);
{
    // Ranges must not overlap
    HK_ASSERT_NO_MSG(0xd205cf21, hkMath::max2(hkUlong(dst), hkUlong(src)) >= (n + hkMath::min2(hkUlong(dst), hkUlong(src))));
    CHECK_OSBLOCKMOVE(dst, src, (size_t)n, false);
}

void HK_CALL hkString::memMove(void* dst, const void* src, int n)
{
    CHECK_OSBLOCKMOVE(dst,src,(size_t)n, false);
}

void HK_CALL hkString::memSet(void* dst, const int c, int n)
{
    OSBlockSet(dst,c,(size_t)n);
}
#else

void HK_CALL hkString::memCpy( _Out_writes_bytes_all_(n) void* dst, _In_reads_bytes_(n) const void* src, _In_range_(>, 0) int n)
{
    // Ranges must not overlap
    HK_ASSERT_NO_MSG(0xd205cf22, hkMath::max2(hkUlong(dst), hkUlong(src)) >= (n + hkMath::min2(hkUlong(dst), hkUlong(src))));
    memcpy(dst,src,(unsigned)n);
}

void HK_CALL hkString::memMove(_Out_writes_bytes_(n) void* dst, _In_reads_bytes_(n) const void* src, _In_range_(>, 0) int n)
{
    memmove(dst,src,(unsigned)n);
}

void HK_CALL hkString::memSet(_Out_writes_bytes_(n) void* dst, const int c, _In_range_(>, 0) int n)
{
    memset(dst,c,(unsigned)n);
}
#endif

int HK_CALL hkString::memCmp( _In_reads_bytes_(n) const void *buf1, _In_reads_bytes_(n) const void *buf2, _In_range_(>, 0) int n)
{
    return memcmp(buf1,buf2,(unsigned)n);
}

bool HK_CALL hkString::findAllOccurrences(_In_z_ const char* haystack, _In_z_ const char* needle, hkArray<int>& indices, hkString::ReplaceType rtype)
{
    size_t needleLen = strlen( needle );
    // find the first occurrence
    const char* p = strStr(haystack, needle);
    bool found = false;
    while( p )
    {
        found = true;
        indices.pushBack( static_cast<int>( p - haystack ) );
        // go to the next one if requested
        p = (rtype == REPLACE_ONE) ? HK_NULL : strStr( p + needleLen, needle );
    }
    return found;
}

bool HK_CALL hkString::beginsWith(_In_z_ const char* a, _In_z_ const char* b)
{
    HK_ASSERT_NO_MSG(0x106348be, a != nullptr);
    HK_ASSERT_NO_MSG(0x484a2814, b != nullptr);

    for(int i = 0; b[i] != 0; ++i)
    {
        if( a[i] != b[i] )
        {
            return false;
        }
    }
    return true;
}

bool HK_CALL hkString::beginsWithCase(_In_z_ const char* a, _In_z_ const char* b)
{
    HK_ASSERT_NO_MSG(0x6d2e0c11, a != nullptr);
    HK_ASSERT_NO_MSG(0x6eab070c, b != nullptr);

    for(int i = 0; b[i] != 0; ++i)
    {
        if( tolower(a[i]) != tolower(b[i]) )
        {
            return false;
        }
    }
    return true;
}

bool HK_CALL hkString::endsWith(_In_z_ const char* a, _In_z_ const char* b)
{
    HK_ASSERT_NO_MSG(0x49aeec26, a != nullptr);
    HK_ASSERT_NO_MSG(0x11138304, b != nullptr);

    int alen = hkString::strLen(a);
    int blen = hkString::strLen(b);
    if (alen < blen)
    {
        return false;
    }
    int offset = alen - blen;

    for (int i = 0; i < blen; ++i)
    {
        if (a[i + offset] != b[i])
        {
            return false;
        }
    }
    return true;
}

bool HK_CALL hkString::endsWithCase(_In_z_ const char* a, _In_z_ const char* b)
{
    HK_ASSERT_NO_MSG(0x2d4bd4cb, a != nullptr);
    HK_ASSERT_NO_MSG(0x442b3557, b != nullptr);

    int alen = hkString::strLen(a);
    int blen = hkString::strLen(b);
    if (alen < blen)
    {
        return false;
    }
    int offset = alen - blen;

    for (int i = 0; i < blen; ++i)
    {
        if (tolower(a[i + offset]) != tolower(b[i]))
        {
            return false;
        }
    }
    return true;
}

bool HK_CALL hkString::endsWithAny(_In_z_ const char* a, const hkArrayView<hkCString> suffexes)
{
    HK_ASSERT_NO_MSG(0x5c120819, a != nullptr);

    bool found = false;
    for (int i = 0; i < suffexes.getSize() && !found; i++)
    {
        found = hkString::endsWith(a, suffexes[i]);
    }
    return found;
}

bool HK_CALL hkString::endsWithAnyCase(_In_z_ const char* a, const hkArrayView<hkCString> suffexes)
{
    HK_ASSERT_NO_MSG(0x17fca4f8, a != nullptr);

    bool found = false;
    for (int i = 0; i < suffexes.getSize() && !found; i++)
    {
        found = hkString::endsWithCase(a, suffexes[i]);
    }
    return found;
}

int HK_CALL hkString::lastIndexOf(_In_z_ const char* str, char c)
{
    if( const char* p = strRchr(str, c) )
    {
        return int(hkUlong(p - str));
    }
    return -1;
}

int HK_CALL hkString::indexOf(_In_z_ const char* str, char c, _In_range_(0, endIndex) int startIndex, _In_range_(startIndex, HK_INT32_MAX) int endIndex)
{
    HK_ASSERT_NO_MSG(0x44e02638, startIndex <= endIndex);
    for( int i = 0; i < startIndex; ++i )
    {
        if( str[i] == 0 )
        {
            return -1;
        }
    }
    for( int i = startIndex; i < endIndex && str[i]; ++i )
    {
        if( str[i] == c )
        {
            return i;
        }
    }
    return -1;
}

void HK_CALL hkString::memClear128(_Out_writes_bytes_(numBytes) void* dst, int numBytes)
{
    HK_ASSERT( 0xf0dcf45e, (hkUlong(dst) & 0x7f) == 0, "Your input array must be 128 byte aligned");
    HK_ASSERT( 0xf0dcf45e, (numBytes & 0x7f) == 0, "Your size must be 128 byte aligned");
#if defined(HK_PLATFORM_WIIU)
    DCZeroRange(dst, numBytes);
#else
    hkString::memClear16( dst, numBytes>>4 );
#endif
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
