// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

template <typename T>
hkQueue<T>::hkQueue()
    :   m_data(HK_NULL),
        m_capacity(0),
        m_head(0),
        m_tail(0),
        m_elementsInUse(0)
{
}


template <typename T>
hkQueue<T>::hkQueue( int n  )
    :   m_data(hkAllocateChunk<T>( n , HK_MEMORY_CLASS_ARRAY )),
        m_capacity(n),
        m_head(0),
        m_tail(0),
        m_elementsInUse(0)
{
    HK_ASSERT( 0x4091ba45, n > 0, "this constructor is not appropriate for zero initial capacity" );
}


template <typename T>
hkQueue<T>::~hkQueue()
{
    clearAndDeallocate();
}



template <typename T>
inline hkBool hkQueue<T>::isEmpty() const
{
    return m_elementsInUse == 0;
}


template <typename T>
inline int hkQueue<T>::getSize() const
{
    return m_elementsInUse;
}


template <typename T>
inline int hkQueue<T>::getCapacity() const
{
    return m_capacity;
}

template <typename T>
inline const T* hkQueue<T>::getData() const
{
    return m_data;
}

template <typename T>
inline const T& hkQueue<T>::getElement(int idx) const
{
    HK_ASSERT_NO_MSG(0x364d5142, idx >= 0 && idx < getSize());
    return m_data[(m_head+idx) % m_capacity];
}

template <typename T>
inline void hkQueue<T>::clear()
{
    if (m_elementsInUse)
    {
        if (m_head < m_tail)
        {
            hkArrayUtil::destruct(m_data+m_head, m_elementsInUse, typename hkTrait::IsPodType<T>::type());
        }
        else // wrapped around or completely full
        {
            hkArrayUtil::destruct(m_data, m_tail, typename hkTrait::IsPodType<T>::type());
            hkArrayUtil::destruct(m_data+m_head, m_capacity-m_head, typename hkTrait::IsPodType<T>::type());
        }
    }

    m_head = 0;
    m_tail = 0;
    m_elementsInUse = 0;
}

template <typename T>
inline void hkQueue<T>::clearAndDeallocate()
{
    clear();
    releaseMemoryWithoutCallingDestructors();
}


template <typename T>
inline void hkQueue<T>::releaseMemoryWithoutCallingDestructors()
{
    if( m_capacity )
    {
        hkDeallocateChunk<T>( m_data, m_capacity, HK_MEMORY_CLASS_ARRAY );
        m_capacity = 0;
        m_data = HK_NULL;
    }
}


template <typename T>
inline void hkQueue<T>::setCapacity( int n )
{
    if( m_capacity < n)
    {
        // allocate a new buffer and copy existing data over
        if ( m_capacity * 2 >= n)
        {
            n = m_capacity * 2;
        }

        T* p = hkAllocateChunk<T>( n , HK_MEMORY_CLASS_ARRAY );

        if( p != HK_NULL && m_data != HK_NULL )
        {
            if ( m_elementsInUse)
            {
                if ( m_tail <= m_head )
                {
                    // split
                    int numHead = m_capacity - m_head;
                    int numTail = m_tail;
                    hkString::memCpy( p, m_data+m_head, ( sizeof(T) * numHead ) );
                    hkString::memCpy( p + numHead, m_data, ( sizeof(T) * numTail ) );
                }
                else
                {
                    // one block
                    hkString::memCpy( p, m_data+m_head, ( sizeof(T) * m_elementsInUse ) );
                }
            }
            m_head = 0;
            m_tail = m_elementsInUse;
        }

        // release the old buffer, no destructors since we have moved the elements
        releaseMemoryWithoutCallingDestructors();

        // assign to new buffer and set capacity
        m_data = p;
        m_capacity = n;
    }
}

template <typename T>
inline void hkQueue<T>::increaseCapacity( )
{
    if( m_capacity == 0 )
    {
        // if there is no current capacity default to 8 elements
        setCapacity( 8 );
    }
    else
    {
        setCapacity( m_capacity * 2 );
    }
}


// Places a new element to the back of the queue and expand storage if necessary.
template <typename T>
inline void hkQueue<T>::enqueue( const T& element )
{
    if( m_elementsInUse >= m_capacity )
    {
        increaseCapacity();
    }

    if( m_tail == m_capacity )
    {
        m_tail = 0;
    }

    hkArrayUtil::constructWithCopy( m_data + m_tail, 1, element, typename hkTrait::IsPodType<T>::type());
    m_tail += 1;
    m_elementsInUse += 1;
}

// Places a new element to the front of the queue and expand storage if necessary.
template <typename T>
inline void hkQueue<T>::enqueueInFront( const T& element )
{
    if( m_elementsInUse >= m_capacity )
    {
        increaseCapacity();
    }

    if( m_head == 0 )
    {
        m_head = m_capacity;
    }

    m_head -= 1;
    hkArrayUtil::constructWithCopy(m_data + m_head, 1, element, typename hkTrait::IsPodType<T>::type());
    m_elementsInUse++;
}


template <typename T>
inline void hkQueue<T>::dequeue( T& data )
{
    HK_ASSERT_NO_MSG(0xf032ed23, m_elementsInUse );
    {
        data = m_data[m_head];
        hkArrayUtil::destruct(m_data + m_head, 1, typename hkTrait::IsPodType<T>::type());
        if( ++m_head == m_capacity )
        {
            m_head = 0;
        }
        m_elementsInUse--;
    }
}

template <typename T>
inline void hkQueue<T>::peek( T& data ) const
{
    data = m_data[m_head];
}

template <typename T>
inline int hkQueue<T>::_getHead() const
{
    return m_head;
}

template <typename T>
inline int hkQueue<T>::_getTail() const
{
    return m_tail;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
