// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include<Common/Base/Algorithm/Hash/hkHash.h>

/// Default operations for hkMaps
/// Keys can be passed by value or by const reference, depending on which is more efficient
template <typename KEY>
struct hkMapOperations
{
    HK_DECLARE_CLASS(hkMapOperations, NewPlacement, Reflect);

    inline static unsigned hash( const KEY& key, unsigned mod )
    {
        using namespace hkHash;
        return hkHashValue( key ) & mod;
    }
    inline static void invalidate( KEY& key )
    {
        key = KEY(-1);
    }
    inline static hkBool32 isValid( const KEY& key )
    {
        return key != KEY(-1);
    }
    inline static hkBool32 equal( const KEY& key0, const KEY& key1 )
    {
        return key0 == key1;
    }
};

    /// A class to store key, value pairs.
    /// Note that the key must have an invalid or reserved state which is used to mark
    /// empty slots. For instance by default -1 is disallowed as a integer key. You can override
    /// this behavior by instantiating with a custom operations structure.
template <typename KEY, typename VAL=KEY, typename OPS=hkMapOperations<KEY> >
class hkMapBase
{
    public:

        HK_DECLARE_CLASS(hkMapBase, NewExplicit, TrackAs(hkReflect::Detail::GenericArray), NonCopyable);

            /// Iterator type.
            /// All iterators are invalidated after a mutating operation, i.e., insertion, removal.
        typedef class Dummy* Iterator;
#       define HK_MAP_INDEX_TO_ITERATOR(i) reinterpret_cast<Iterator>( hkUlong(i) )
#       define HK_MAP_ITERATOR_TO_INDEX(it) static_cast<int>( reinterpret_cast<hkUlong>(it) )

            /// Create an empty pointer map.
        hkMapBase() : m_elem(HK_NULL), m_numElems(0), m_hashMod(-1) {}

            /// Create pointer map initially using preallocated memory block.
            /// Use the getSizeInBytesFor(int numKeys) method to find the buffer size
            /// required for a given number of keys.
        hkMapBase(_In_bytecount_(sizeInBytes) void* ptr, int sizeInBytes);

    protected:

            /// For creating an uninitialized map.
        enum InternalInitializer
        {
            UNINITIALIZED,
            s_minimumCapacity = 8
        };

            /// Create an uninitialized map. For internal use.
        hkMapBase( InternalInitializer ii ) {}

    public:

            /// Destroy a pointer map.
        ~hkMapBase();

            /// Get an iterator over the keys of this map.
        Iterator getIterator() const;

            /// Get the key at iterator i.
        const KEY& getKey( _In_ Iterator i ) const;

            /// Get the value at iterator i.
        VAL& getValue( _In_ Iterator i ) const;

            /// Overwrite the value at iterator i.
        void setValue( _In_ Iterator i, const VAL& val );

            /// Get the next iterator after i.
        Iterator getNext( _In_ Iterator i ) const;

            /// Return if the iterator has not reached the end.
        hkBool isValid( _In_ Iterator i ) const;

            /// Insert key with associated value val. Keys are unique and by default
            /// (if using hkPointerMapOperations) cannot be -1.
            /// If key already exists it is overwritten.
        hkBool32 insert( hkMemoryAllocator& alloc, const KEY& key, const VAL& val );

            /// Try to insert key with associated value val. Keys are unique and by default
            /// (if using hkPointerMapOperations) cannot be -1.
            /// If key already exists it is overwritten.
        hkBool32 tryInsert( hkMemoryAllocator& alloc, const KEY& key, const VAL& val, hkResult& res);

            /// Get an iterator at 'key'. Check if key was found with isValid().
        Iterator findKey( const KEY& key ) const;

            /// If key is present return its iterator, else insert (key,val) and return the new iterator.
            /// Thus the returned iterator is always valid.
        Iterator findOrInsertKey( hkMemoryAllocator& alloc, const KEY& key, const VAL& def );

            /// Shortcut for isValid(findKey(key)).
        hkBool hasKey( const KEY& key ) const { return isValid(findKey(key)); }
        bool contains( const KEY& key ) const { return hasKey(key); }

            /// Return the value associated with key or def if not present.
        const VAL& getWithDefault( const KEY& key, const VAL& def ) const;

            /// If key present, write value into out and return HK_SUCCESS. Otherwise return HK_FAILURE.
        hkResult get( const KEY& key, _Out_ VAL* out ) const;

            /// Remove pair at iterator. The iterator is no longer valid after this call.
        void remove( _In_ Iterator it );

            /// If key present, remove it and return HK_SUCCESS. Otherwise return HK_FAILURE.
        hkResult remove( const KEY& key );

            /// Return the number of keys.
        int getSize() const { return m_numElems & static_cast<int>(NUM_ELEMS_MASK); }

            /// Return the number of keys possible to store without reallocation.
        int getCapacity() const { return (m_hashMod + 1) & static_cast<int>(NUM_ELEMS_MASK); }

            /// Return the amount of allocated memory in bytes. Use for statistics.
        int getMemSize() const;

            /// Return the start address in memory of the hashmap. Use for statistics.
        void* getMemStart() const { return m_elem; }

            /// Perform an internal consistency check.
        hkBool isOk() const;

            /// Remove all keys from the map.
        void clear();

            /// Remove all keys from the map.
        void clearAndDeallocate( hkMemoryAllocator& a );

            // Estimates and sets the appropriate table size for a given number of elements.
        void reserve( hkMemoryAllocator& a, int numElements );

            /// Return true if the map was constructed with a buffer which was too small.
        inline hkBool wasReallocated() const { return ( (m_numElems & static_cast<int>(DONT_DEALLOCATE_FLAG)) == 0 ); }

            /// Advanced use only.
        void setOwnedData(_In_bytecount_(capacity) void* ptr, int size, int capacity);

            /// Calculates buffer size required to store the specified number of keys.
        static int HK_CALL getSizeInBytesFor(int numOfKeys);

            /// Contains the implementation of hkMapBase(void* ptr, int sizeInBytes).
            /// For internal use only.
        void init(_In_bytecount_(sizeInBytes) void* ptr, int sizeInBytes);

    protected:

        hkResult resizeTable(hkMemoryAllocator& alloc, int capacity);

    private:
        int _findEmptyPositionInternal( hkMemoryAllocator& alloc, const KEY& key, hkBool32& wasNewKeyOut);

    protected:

        static inline hkBool32 isPower2(unsigned int v) { return (v & (v - 1)) == 0; }

        // Internal flags, set in constructor.
        enum
        {
            NUM_ELEMS_MASK = int(0x7FFFFFFF),
            DONT_DEALLOCATE_FLAG = int(0x80000000) // Indicates that the storage is not the array's to delete
        };

        struct Pair
        {
            KEY key;
            VAL val;
        };

        Pair* m_elem;
        int m_numElems; // high bits are flags
        int m_hashMod; // capacity - 1
};

    /// A hkMapBase with a builtin allocator.
template <typename KEY, typename VAL=KEY, typename OPS=hkMapOperations<KEY>, typename Allocator=hkContainerHeapAllocator>
class hkMap : public hkMapBase<KEY,VAL,OPS>
{
    public:

        HK_DECLARE_CLASS(hkMap, NewTemplate, NonCopyable);
        HK_RECORD_ATTR(hk::MemoryTracker(opaque=true, handler=&hkReflect::Tracker::hkMapHandler<KEY, VAL, Allocator>::func));

        typedef hkMap<KEY,VAL,OPS,hkContainerTempAllocator> Temp;
        typedef hkMap<KEY,VAL,OPS,hkContainerDebugAllocator> Debug;
        typedef hkMap<KEY,VAL,OPS,hkContainerHeapAllocator> Heap;

            ///
        hkMap(int nelem=0) { if (nelem){ reserve(nelem); } }

            /// Create pointer map initially using preallocated memory block.
            /// Use the getSizeInBytesFor(int numKeys) method to find the buffer size
            /// required for a given number of keys.
        hkMap(void* ptr, int sizeInBytes) : hkMapBase<KEY,VAL,OPS>(ptr, sizeInBytes) {}

            /// For creating an uninitialized map.
        enum InternalInitializer
        {
            UNINITIALIZED,
        };

            /// Create an uninitialized map. For internal use.
        hkMap( InternalInitializer ii ) : hkMapBase<KEY,VAL,OPS>( hkMapBase<KEY,VAL,OPS>::UNINITIALIZED )  {}

            /// Destroy a pointer map.
        ~hkMap() { hkMapBase<KEY, VAL, OPS>::clearAndDeallocate(Allocator().get(this)); }

            /// Remove all keys from the map and deallocate internal memory.
        void clearAndDeallocate() { hkMapBase<KEY, VAL, OPS>::clearAndDeallocate(Allocator().get(this)); };

            /// Allow access to clearAndDeallocate overloads in base class.
        using hkMapBase<KEY, VAL, OPS>::clearAndDeallocate;

            /// Insert key with associated value val. Keys are unique and by default
            /// (if using hkPointerMapOperations) cannot be -1.
            /// If key already exists it is overwritten.
        HK_INLINE hkBool32 insert( const KEY& key, const VAL& val ) { return hkMapBase<KEY,VAL,OPS>::insert(Allocator().get(this), key, val); }

            /// Insert key with associated value val. Keys are unique and by default
            /// (if using hkPointerMapOperations) cannot be -1.
            /// If key already exists it is overwritten.
            /// res is set to HK_SUCCESS if the insertion succeeded, HK_FAILURE otherwise
        HK_INLINE hkBool32 tryInsert( const KEY& key, const VAL& val, hkResult& res ) { return hkMapBase<KEY,VAL,OPS>::tryInsert(Allocator().get(this), key, val, res); }

            /// If key is present return its iterator, else insert (key,val) and return the new iterator.
            /// Thus the returned iterator is always valid.
        HK_INLINE typename hkMapBase<KEY,VAL,OPS>::Iterator findOrInsertKey( const KEY& key, const VAL& def ) { return hkMapBase<KEY,VAL,OPS>::findOrInsertKey(Allocator().get(this), key, def); }

            /// Swap all data with another map.
        void swap(hkMap& other);

            // Estimates and sets the appropriate table size for a given number of elements.
        HK_INLINE void reserve( int numElements ) { return hkMapBase<KEY,VAL,OPS>::reserve(Allocator().get(this), numElements); }
};

#include <Common/Base/Container/PointerMap/hkMap.inl>
#include <Common/Base/_Auto/TemplateTypes/hkMap_Types.inl>

#ifdef HK_DYNAMIC_DLL
HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON hkMapBase<unsigned int, unsigned int>;
HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON hkMap<unsigned int, unsigned int>;

HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON hkMapBase<unsigned long, unsigned long>;
HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON hkMap<unsigned long, unsigned long>;

HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON hkMapBase<unsigned long long, unsigned long long>;
HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON hkMap<unsigned long long, unsigned long long>;
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
