// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

/// A array class which uses hkMemory stack based allocations.
/// Stack allocation patterns can be much faster than heap allocations.
/// See the hkMemory/Frame based allocation user guide section.
/// When an hkLocalArray grows beyond its original specified capacity, it
/// falls back on heap allocations so it is important to specify a good
/// initial capacity.
/// hkLocalArray should almost always be a local variable in a
/// method and almost never a member of an object.
/// The array can handle both pod and non-pod types (see hkArray for more details).
template <typename T>
class hkLocalArray : public hkArray<T>
{
    public:

        HK_DECLARE_CLASS(hkLocalArray<T>, NewOpaque); 

            /// Create an array with zero size and given capacity.
            /// For maximum efficiency, the arrays capacity should never grow
            /// beyond the specified capacity. If it does, the array falls back
            /// to using heap allocations which is safe, but the speed advantage
            /// of using the stack based allocation methods is lost.
        HK_INLINE hkLocalArray( int capacity )
            : m_initialCapacity(capacity)
        {
            this->m_data = capacity ? hkAllocateStack<T>(capacity) : HK_NULL;
            this->m_capacityAndFlags = capacity | hkArray<T>::DONT_DEALLOCATE_FLAG;
            m_localMemory = this->m_data;
        }

        HK_INLINE void assertNotResized()
        {
            HK_ASSERT(0x5f792e08, m_localMemory == this->m_data, "A localarray grew beyond its original specified capacity.");
        }

        HK_INLINE bool wasReallocated() const
        {
            return m_localMemory != this->m_data;
        }

            /// Destroy the array.
        HK_INLINE ~hkLocalArray()
        {
            if( wasReallocated() != true )
            {
                hkArray<T>::clear();
            }
            hkDeallocateStack<T>(m_localMemory, m_initialCapacity);
        }

        T* m_localMemory;
        int m_initialCapacity;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
