// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

template<typename ITEM>
HK_NEVER_INLINE hkHashMapDetail::Entry* hkHashBase<ITEM>::_findEntry( const KeyedType& item ) const
{
    HK_ON_DEBUG( selfCheck() );
    hkUint32 hash = hkHashMapDetail::computeHash( item );

    for ( hkUint32 index = hash & m_index.m_hashMod; true; index = (index + 1) & m_index.m_hashMod )
    {
        hkHashMapDetail::Entry* entry = &m_index.m_entries[index];
        if ( entry->isEmpty() )
        {
            return nullptr;
        }
        else if ( hkHashMapDetail::equal( m_items[entry->idx], item ) )
        {
            return entry;
        }
    }
}


template<typename ITEM>
HK_INLINE typename hkHashBase<ITEM>::Iterator hkHashBase<ITEM>::find(const KeyedType& key) const
{
    if(const hkHashMapDetail::Entry* entry = _findEntry(key))
    {
        return Iterator(entry->idx, m_index.indexFromEntry(entry));
    }
    else
    {
        return Iterator(-1, -1);
    }
}

template<typename ITEM>
HK_NEVER_INLINE void hkHashBase<ITEM>::afterReflectNew()
{
    // If this fails, you should get a leak with the call stack of
    // the code that touched entries before afterReflectNew was
    // called on this.

    if (hkHashMapDetail::TolerantIndexRebuild<KeyedType>::result)
    {
        // This was an ASSERT. Making it a WARN is a _work around_ for COM-3937
        // This was a WARN. Making it a comment due to spam in log.
        /*HK_WARN_ON_DEBUG_IF(!(m_index.m_entries == HK_NULL || m_index.isSentinel()), 0x38a8dc0,
            "An instance of hkHashBase was initialized non-empty during cloning. "
            "This initialization is wasteful because the allocations and the default elements "
            "will be discarded and overwritten by the cloned values." );*/
    }
    m_index.clearAndDeallocate();
    m_index.reserve(m_items.getSize());
    for(int i = 0; i < m_items.getSize(); ++i)
    {
        hkUint32 hash = hkHashMapDetail::computeHash(m_items[i]);
        m_index.add(hash, i);
    }
}

template<typename ITEM>
HK_INLINE hkResult hkHashBase<ITEM>::remove(const KeyedType& item)
{
    if (hkHashMapDetail::Entry* emptySlot = _findEntry(item))
    {
        _remove(emptySlot);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

template<typename ITEM>
HK_INLINE void hkHashBase<ITEM>::remove(Iterator it)
{
    HK_ASSERT_NO_MSG(0x12ce00ce, isValid(it));
    // Check cached it.m_h hash index
    hkHashMapDetail::Entry* slot;
    if ((unsigned(it.m_h) <= unsigned(m_index.m_hashMod) && m_index.m_entries[it.m_h].idx == it.m_i))
    {
        slot = m_index.m_entries + it.m_h;
    }
    else
    {
        slot = _findEntry(m_items[it.m_i]);
    }
    HK_ASSERT_NO_MSG(0x6a9dbca0, slot);
    _remove(slot);
}

template<typename ITEM>
HK_NEVER_INLINE void hkHashBase<ITEM>::_remove(hkHashMapDetail::Entry* emptySlot)
{
    int delIdx = emptySlot->idx;
    // Remove from items. Also update entry which is swapped in
    // Assumes removeAt will swap the last element into delIdx
    if(delIdx != m_items.getSize()-1 )
    {
        hkHashMapDetail::Entry* e = _findEntry(m_items.back());
        e->idx = delIdx;
    }
    m_items.removeAt(delIdx);

    // Scan forward to find a slot which can be swapped into emptySlot
    // We must ensure that each entry is reachable, scanning forwards from its "wanted" slot
    // without hitting an empty slot.
    for(hkHashMapDetail::Entry* curSlot = m_index.nextEntry(emptySlot); true; curSlot = m_index.nextEntry(curSlot))
    {
        if(curSlot->isEmpty())
        {
            break;
        }
        // The "natural" position the curSlot wants to occupy.
        hkHashMapDetail::Entry* curWants = m_index.getFirstEntry(curSlot->hash);
        // We can move curSlot to empty if empty is between curWants and curSlot.
        // The modular arithmetic handles wrap-around.
        const hkUlong emptyRelIndex = hkUlong( emptySlot - curWants ) & m_index.m_hashMod;
        const hkUlong curRelIndex = hkUlong( curSlot - curWants ) & m_index.m_hashMod;
        if ( emptyRelIndex < curRelIndex )
        {
            // Put curSlot into emptySlot and now curSlot becomes empty
            *emptySlot = *curSlot;
            emptySlot = curSlot;
        }
    }
    emptySlot->setEmpty();
}


template<typename ITEM>
HK_INLINE void hkHashBase<ITEM>::swap(hkHashBase& other)
{
    m_items.swap(other.m_items);
    m_index.swap(other.m_index);
}

template<typename ITEM>
bool hkHashBase<ITEM>::addToItems(int index, _In_ const ITEM* newEntry)
{
     m_items.insertAt(index, *newEntry);

    
    m_index.clearAndDeallocate();
    afterReflectNew();

     return true;
}

template<typename ITEM>
bool hkHashBase<ITEM>::removeFromItems(int index)
{
    m_items.removeAtAndCopy(index);

    
    m_index.clearAndDeallocate();
    afterReflectNew();

     return true;
}

template<typename ITEM>
void hkHashBase<ITEM>::selfCheck() const
{
    HK_ASSERT(0x688c2a2d, isOk(), "Hash index empty, but items nonempty. afterReflectNew out of order?");
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
