// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Container/Hash/Detail/hkHashMapDetail.h>

    /// Base class for hash maps and sets.
    /// Items must be hashable (have a function hkUint32 hkHashValue(ITEM i))
    /// and be comparable (have an operator== or overload hkHashMapDetail::equals)
template<typename ITEM>
class hkHashBase
{
public:
    HK_DECLARE_CLASS(hkHashBase, NewTemplate, Reflect);

    typedef ITEM Item;
        // For a set, this is just item. For a map<K,V> it is K.
    typedef typename hkHashMapDetail::KeyedType<ITEM>::Type KeyedType;

        ///
    struct Iterator
    {
        HK_ALWAYS_INLINE Iterator(int i, int h) : m_i(i), m_h(h) {}
        int m_i; // Index into m_items array, <0 indicates invalid. Remains valid after additions, but not removals.
        int m_h; // Index of associated hash entry.
    };

        /// Get an iterator at 'key'. Check if key was found with isValid().
    Iterator find(const KeyedType& key) const;

        /// Check whether or not we contains the element
    bool contains(const KeyedType& key) const { return isValid(find(key)); }

        /// Get an iterator over the values of this map.
    Iterator getIterator() const { return Iterator(m_items.getSize() - 1, -1); }

        /// Get the next iterator after i.
    Iterator getNext(Iterator i) const { return Iterator(i.m_i - 1, -1); }

        /// Return if the iterator has not reached the end.
    hkBool isValid(Iterator i) const { return i.m_i >= 0; }

        /// Computes and sets the appropriate table size for a given number of items.
    void reserve(int numItem) { m_items.reserve(numItem); m_index.reserve(numItem); }

        /// Return the number of items in this map.
    int getSize() const { return m_items.getSize(); }

        /// Return whether there are zero items in this container.
    bool isEmpty() const { return m_items.isEmpty(); }

        /// If item present, remove it and return HK_SUCCESS. Otherwise return HK_FAILURE.
    hkResult remove(const KeyedType& item);

        /// Remove element at it. The iterator must no longer be dereferenced, but
        /// a following getNext(it) will return the same iterator as if remove() had
        /// not been called, so calling remove() in an iteration loop will visit
        /// each element exactly once.
    void remove(Iterator it);

        /// Remove all keys from the map.
    void clear() { if (getSize() > 0) { m_items.clear(); m_index.clear(); } }

        /// Remove all keys from the map.
    void clearAndDeallocate() { m_items.clearAndDeallocate(); m_index.clearAndDeallocate(); }

        /// Perform an internal consistency check.
        // If there are items, the index must be initialized.
    bool isOk() const { return m_items.isEmpty() || m_index.m_hashMod!=0; }

        /// Swap storage with the other hkHashBase.
    void swap(hkHashBase& other);

    hkArrayView<const ITEM> viewItems() const { return m_items; }

        /// Array semantic callback
    bool addToItems(int index, _In_ const ITEM* newEntry);

        /// Array semantic callback
    bool removeFromItems(int index);


    void afterReflectNew();
    inline void selfCheck() const;

protected:

    hkHashMapDetail::Entry* _findEntry(const KeyedType& key) const;
    void _remove(hkHashMapDetail::Entry* e);

protected:

    hkArray< ITEM > m_items
        HK_ATTR(hk::DiffArraySemantics(hk::DiffArraySemantics::treatAsSet));
    hkHashMapDetail::Index m_index
        HK_ATTR(hk::Serialize(false));
};

#include <Common/Base/Container/Hash/hkHashBase.inl>
#include <Common/Base/_Auto/TemplateTypes/hkHashBase_Types.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
