// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Algorithm/Hash/hkHash.h>

namespace hkHashMapDetail
{
    template<typename T>
    HK_ALWAYS_INLINE hkUint32 computeHash(const T& t){ using namespace hkHash; return hkHashValue(t);  }

    // nice to haves:
    // * template on storage type
    // * template on index type
    // * choice of non-linear collision probes
    // * dynamically change indexing strategy
    // * how to handle multiple items with the same value?

    struct Entry
    {
        enum Special { SENTINEL=-2, EMPTY=-1 };
        HK_INLINE Entry() { setEmpty(); }
        HK_INLINE Entry(Special s) { hash = 0; idx = s; }
        HK_ALWAYS_INLINE void setEmpty() { HK_ON_DEBUG(hash = 0); idx = EMPTY; }

        HK_ALWAYS_INLINE bool isSentinel() const { return idx == SENTINEL; }
        HK_ALWAYS_INLINE bool isEmpty() const { return idx < 0; }
        HK_ALWAYS_INLINE bool isUsed() const { return idx >= 0; }

        HK_ALWAYS_INLINE void init(hkUint32 h, int i) { hash = h, idx = i; }

        hkUint32 hash;
        hkInt32 idx; // EMPTY, SENTINEL or item index
    };

    template<typename ITEM>
    struct KeyedType
    {
        typedef ITEM Type;
    };

    template<typename T, typename S>
    HK_ALWAYS_INLINE bool equal(const T& t, const S& s){ return hkTrait::EqualValues<T, S>::func(t, s);    }

        // Maps hash values to indices.
        // In the normal case, each entry has an index of -1 (meaning empty) or a positive value (its index).
        // In one special case, an entry can have an index of -2 (sentinel value). This is when the index
        // is completely empty. In order to avoid having to check for null m_entries all the time, when empty
        // we point m_entries to a single shared entry containing an index -2.
    struct Index
    {
        HK_DECLARE_CLASS( Index, New );
        HK_DETAIL_DECLARE_REFLECT_EXPORT( Index, HK_EXPORT_COMMON );

        HK_RECORD_ATTR( hk::ExcludeFromVersionCheck );
        HK_RECORD_ATTR( hk::MemoryTracker( opaque = true, handler = &hkReflect::Tracker::hkHashMapDetailIndexHandler ) );
        HK_ALWAYS_INLINE Index() { setSentinel(); }
        HK_EXPORT_COMMON Index( const Index& other );
        HK_EXPORT_COMMON const Index& operator=( const Index& other );

        HK_EXPORT_COMMON ~Index();

        HK_ALWAYS_INLINE Entry* getFirstEntry( hkUint32 hash ) const
        {
            return m_entries + (hash&m_hashMod);
        }

        HK_ALWAYS_INLINE Entry* nextEntry( _In_ const Entry* e ) const
        {
            return m_entries + ((e - m_entries + 1) & m_hashMod);
        }

        HK_INLINE void reserve( int numItems )
        {
            if ( 3 * numItems > 2 * m_hashMod ) // max 2/3 fill
            {
                reserve2( numItems );
            }
        }

        HK_INLINE void reserveOneMoreThan( int size )
        {
            HK_ASSERT( 0x3a870881, size <= m_hashMod, "hash index not initialized, afterReflectNew out of order?" );
            reserve( size + 1 );
        }

        HK_INLINE void add( hkUint32 hash, int idx )
        {
            HK_ASSERT_NO_MSG( 0x4f0f84e5, isSentinel() == false && idx <= m_hashMod );
            HK_ON_DEBUG( int maxiter = m_hashMod + 1 );
            for ( Entry* e = getFirstEntry( hash ); true; e = nextEntry( e ) )
            {
                HK_ASSERT_NO_MSG( 0x3e9faf3a, maxiter-- );
                if ( e->isEmpty() )
                {
                    e->init( hash, idx );
                    return;
                }
            }
        }

        HK_EXPORT_COMMON void reserve2( int n );

        HK_EXPORT_COMMON void clear();

        HK_EXPORT_COMMON void clearAndDeallocate();

        HK_EXPORT_COMMON void setSentinel();

        // Are we using the sentinel entry?
        HK_INLINE bool isSentinel() const { return m_entries[0].isSentinel(); }

        HK_EXPORT_COMMON void swap( Index& other );

        HK_INLINE int indexFromEntry( const Entry* e ) const
        {
            const int i = int( e - m_entries );
            HK_ASSERT( 0x599d3dfb, i >= 0 && i <= m_hashMod, "Invalid entry" );
            return i;
        }

        Entry* m_entries HK_ATTR( hk::Type( void* ) );

        // When using the sentinel this is 0. Otherwise it is the number of allocated entries - 1.
        int m_hashMod;
    };

    // From HashMap
    template<typename KEY, typename VALUE>
    struct MapTuple : public hkTuple < KEY, VALUE >
    {
        HK_DECLARE_CLASS( MapTuple, New, Reflect );
        HK_INLINE MapTuple() {}
        HK_INLINE bool operator==( const KEY& k ) const { return hkHashMapDetail::equal( this->m_0, k ); }
        HK_INLINE operator const KEY&() const { return this->m_0; }
    };

    template<typename KEY, typename VALUE>
    struct KeyedType < MapTuple<KEY, VALUE> >
    {
        typedef KEY Type;
    };

    template<typename KEY, typename VALUE>
    HK_ALWAYS_INLINE hkUint32 computeHash( const MapTuple<KEY, VALUE>& t )
    {
        return computeHash( t.m_0 );
    }

    template<typename KeyType>
    struct TolerantIndexRebuild : hkTrait::FalseType {};
}

namespace hkReflect { namespace Tracker {
    HK_EXPORT_COMMON void hkHashMapDetailIndexHandler(const hkReflect::Var& var, hkMemoryTrackerSnapshot& snapshot);
}}

#include <Common/Base/_Auto/TemplateTypes/hkHashMapDetail_Types.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
