// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Container/Hash/Detail/hkHashMapDetail.h>
#include <Common/Base/Algorithm/Hash/hkHash.h>
#include <Common/Base/Memory/Tracker/hkMemoryTrackerSnapshot.h>

namespace hkHashMapDetail
{
    const Entry s_entriesSentinel( Entry::SENTINEL );

    //////////////////////////////////////////////////////////////////////////////////
    // Out-of-line for mixed mode DLL.
    // When the following functions are in-line they can cause mixed mode compiler
    // to get confused as some managed code will-have references (through static/globals)
    // which can place CLR-dependent code in the dynamic shutdown for the DLL which
    // can get run *after* the CLR has been unloaded.
    
    
    //////////////////////////////////////////////////////////////////////////////////

    Index::~Index() { clearAndDeallocate(); }
    void Index::setSentinel()
    {
        m_entries = const_cast< Entry* >( &s_entriesSentinel );
        // 0 means that getFirstEntry returns &m_entries.
        m_hashMod = 0;
    }

    //////////////////////////////////////////////////////////////////////////////////

    void Index::reserve2(int n)
    {
        n = hkMath::max2(n, 4);
        int cap = hkNextPowerOf2(n * 2);

        Entry* newEntries = hkMemHeapBlockAlloc<Entry>(cap);
        int newHashMod = cap - 1;

        for(int i = 0; i < cap; ++i)
        {
            newEntries[i].setEmpty();
        }

        if(m_entries && isSentinel() == false)
        {
            int oldHashMod = m_hashMod;
            Entry* oldEntries = m_entries;

            // Set these before calling add.
            m_entries = newEntries;
            m_hashMod = newHashMod;

            for(int i = 0; i < oldHashMod + 1; ++i)
            {
                if(oldEntries[i].isUsed())
                {
                    add(oldEntries[i].hash, oldEntries[i].idx);
                }
            }
            hkMemHeapBlockFree(oldEntries, oldHashMod+1);
        }
        else
        {
            m_entries = newEntries;
            m_hashMod = newHashMod;
        }
    }

    void Index::clear()
    {
        if ( m_entries && isSentinel() == false )
        {
            for ( int i = 0; i < m_hashMod + 1; ++i )
            {
                m_entries[i].setEmpty();
            }
        }
    }

    void Index::clearAndDeallocate()
    {
        if ( m_entries && isSentinel() == false )
        {
            hkMemHeapBlockFree( m_entries, m_hashMod + 1 );
            setSentinel();
        }
    }

    static void dupIndex(Index& dst, const Index& src)
    {
        if(src.isSentinel())
        {
            dst.setSentinel();
        }
        else
        {
            dst.m_hashMod = src.m_hashMod;
            dst.m_entries = hkMemHeapBlockAlloc<Entry>(src.m_hashMod + 1);
            hkMemUtil::memCpy(dst.m_entries, src.m_entries, (src.m_hashMod + 1) * sizeof(Entry));
        }
    }

    Index::Index( const Index& other )
    {
        dupIndex(*this, other);
    }

    const Index& Index::operator=(const Index& other)
    {
        if(&other != this)
        {
            clearAndDeallocate();
            dupIndex(*this, other);
        }
        return *this;
    }

    void Index::swap( Index& other )
    {
        hkMath::swap( m_entries, other.m_entries );
        hkMath::swap( m_hashMod, other.m_hashMod );
    }
}

void hkReflect::Tracker::hkHashMapDetailIndexHandler(const hkReflect::Var& var, hkMemoryTrackerSnapshot& snapshot)
{
    const hkHashMapDetail::Index* index = var.dynCast<hkHashMapDetail::Index>();
    HK_ASSERT_NO_MSG(0x2ace24ba, index);
    if (index->m_hashMod)
    {
        snapshot.addLinkedBlock(hkMemoryTrackerSnapshot::Block(
            "buffer_hkHashMapDetail::Index", index->m_entries, (index->m_hashMod + 1) * sizeof(hkHashMapDetail::Entry)), true);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
