// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Container/BlockStream/hkTypedBlockStream.h>
#include <Common/Base/DebugUtil/DeterminismUtil/hkCheckDeterminismUtil.h>

class hkOstream;


/// Base class of a command.
/// Commands are used so that different parts of Havok can communicate with each other. Each command has a primary and a
/// secondary type. This is done to allow for a more flexible way to configure and custom implement command dispatchers.
/// Commands should be stream-able between platforms of the same endianness like between PPU and SPU, therefore they
/// have no vtable.
class HK_EXPORT_COMMON hkCommand
{
    public:

        /// Primary types are used
        enum PrimaryType
        {
            TYPE_NOP,               ///< no operation
            TYPE_DEBUG_DISPLAY,

            TYPE_PHYSICS_API,       ///< like body::setVelocity
            TYPE_PHYSICS_INTERNAL,
            TYPE_PHYSICS_EVENTS,

            TYPE_DESTRUCTION_API,
            TYPE_DESTRUCTION_INTERNAL,
            TYPE_DESTRUCTION_EVENTS,

            TYPE_AI_WORLD_API,

            TYPE_MAX
        };

        HK_DECLARE_CLASS(hkCommand, Reflect, BypassCtor, New);

        HK_INLINE hkCommand( PrimaryType type, hkUint16 subType, int sizeInBytes );

        /// Called by block stream functions.
        HK_INLINE int getSizeInBytes() const { return m_sizePaddedTo16; }

#if defined (HK_ENABLE_DETERMINISM_CHECKS)
        HK_INLINE int determinismGetSizeInBytes() const { return m_determinismSizeInBytes; }
#endif

    public:

        hkUint16 m_sizePaddedTo16;  ///< Size in bytes rounded to the next multiple of 16 for performance.
        hkUint8  m_filterBits;      ///< Allows for selectively executing commands.
        hkEnum<PrimaryType, hkUint8> m_primaryType;
        hkUint16 m_secondaryType;

#if defined (HK_WANT_DETERMINISM_CHECKS)
        hkUint16 m_determinismSizeInBytes;      ///< Real size in bytes needed for determinism checks.
#endif
};

HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkCommand::PrimaryType);


/// Command dispatcher which dispatches commands using the secondary type.
class HK_EXPORT_COMMON hkSecondaryCommandDispatcher : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE );

        virtual ~hkSecondaryCommandDispatcher() {}
        virtual void exec( const hkCommand& command ) = 0;

        /// optional print function for debugging
        virtual void print( const hkCommand& command, hkOstream& stream ) const {}
};


/// Implementation of hkCommandProcessor which streams into a block stream.
class HK_EXPORT_COMMON hkBlockStreamCommandWriter : public hkSecondaryCommandDispatcher
{
    public:
        HK_DECLARE_CLASS_ALLOCATOR(0);

        hkBlockStreamCommandWriter();

        /// Write a command.
        void exec( const hkCommand& command );
        // non virtual function of exec

        /// Write a command.
        HK_INLINE void* allocBuffer( int size );

    public:

        hkTypedBlockStream<hkCommand>::Writer m_writer;
};

/// command dispatcher which dispatches using the primary type
class HK_EXPORT_COMMON hkPrimaryCommandDispatcher
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, hkPrimaryCommandDispatcher);

        hkPrimaryCommandDispatcher();
        ~hkPrimaryCommandDispatcher();

        void registerDispatcher( hkCommand::PrimaryType type, _In_ hkSecondaryCommandDispatcher* dispatcher );

        void exec( const hkCommand& command );

        void print( const hkCommand& command, hkOstream& stream ) const;

    public:

        hkRefPtr<hkSecondaryCommandDispatcher> m_commandDispatcher[ hkCommand::TYPE_MAX ];
};


#include <Common/Base/Container/CommandStream/hkCommandStream.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
