// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/Container/BlockStream/hkBlockStreamRange.h>


//////////////////////////////////////////////////////////////////////////
// Writer
//////////////////////////////////////////////////////////////////////////

HK_INLINE hkBlockStream::Writer::Writer()
{
    HK_ON_DEBUG( m_finalized = true );
    HK_ON_DEBUG( m_accessSize = 0 );
    HK_ON_DEBUG( m_currentBlock = HK_NULL );

    m_blockStream = HK_NULL;
}

HK_INLINE hkBlockStream::Writer::~Writer()
{
    HK_ASSERT_NO_MSG( 0xf034dedf, m_finalized );
}

HK_INLINE bool hkBlockStream::Writer::isValid() const
{
    return m_blockStream != HK_NULL;
}

HK_INLINE int hkBlockStream::Writer::getBlockBytesLeft() const
{
    return Block::BLOCK_DATA_SIZE - m_currentByteOffset;
}

HK_INLINE int hkBlockStream::Writer::getTotalNumElems() const
{
    return m_blockStream->m_numTotalElements + m_currentBlockNumElems;
}

HK_INLINE hkBlockStream::Block* hkBlockStream::Writer::getCurrentBlock() const
{
    return m_currentBlock;
}

HK_INLINE int hkBlockStream::Writer::getCurrentByteOffset() const
{
    return m_currentByteOffset;
}

HK_INLINE void hkBlockStream::Writer::finalizeLastBlock(
    hkBlockStream::Block* HK_RESTRICT thisBlock,
    hkBlockStream::Block* HK_RESTRICT nextBlock,
    int thisBlockNumElements,
    int bytesUsed )
{
    thisBlock->m_next = nextBlock;
    thisBlock->setNumElements( hkBlockStream::Block::CountType( thisBlockNumElements ) );
    thisBlock->setNumBytesUsed( hkBlockStream::Block::CountType( bytesUsed ) );
    m_blockStream->m_numTotalElements += thisBlockNumElements;
}

HK_INLINE void hkBlockStream::Writer::advance( int numBytes )
{
    // Make sure we are exactly advancing by the size of the previous (write)access.
    HK_ASSERT_NO_MSG( 0xf03454df, numBytes <= m_accessSize );
    HK_ON_DEBUG( m_finalized = false );
    HK_ON_DEBUG( m_accessSize = 0 );

    int nextLocation = m_currentByteOffset + numBytes;
    int numElements = m_currentBlockNumElems + 1;

    m_currentByteOffset = nextLocation;
    m_currentBlockNumElems = numElements;
}

HK_INLINE void hkBlockStream::Writer::undoAdvance( int numBytes, int reservedBytes )
{
    if( numBytes )
    {
        // Make sure we are exactly advancing by the size of the previous (write)access.
        HK_ON_DEBUG( m_finalized = false );
        HK_ON_DEBUG( m_accessSize = reservedBytes );

        int nextLocation = m_currentByteOffset - numBytes;
        int numElements = m_currentBlockNumElems - 1;

        m_currentByteOffset = nextLocation;
        m_currentBlockNumElems = numElements;

        HK_ASSERT_NO_MSG( 0xf0345456, m_currentByteOffset + reservedBytes <= Block::BLOCK_DATA_SIZE );
    }
}

template <typename T>
HK_INLINE T* hkBlockStream::Writer::reserve( int numBytes )
{
    HK_ASSERT_NO_MSG( 0xf03d3401, m_currentBlock != HK_NULL );
    HK_ASSERT_NO_MSG( 0xaf1e132f, numBytes >= sizeof( T ) && numBytes <= Block::BLOCK_DATA_SIZE );
    HK_ON_DEBUG( m_accessSize = numBytes );
    HK_ON_DEBUG( m_finalized = false );

    // Make sure numBytes fit into the remaining storage space in the current block.
    // Otherwise move to next block in stream.
    if( HK_VERY_LIKELY( m_currentByteOffset + numBytes <= Block::BLOCK_DATA_SIZE ) )
    {
        return (T*)hkAddByteOffset( m_currentBlock->begin(), m_currentByteOffset );
    }
    else
    {
        return (T*)allocateAndAccessNewBlock();
    }
}

HK_INLINE char* hkBlockStream::Writer::reserve( int numBytes )
{
    return reserve<char>( numBytes );
}

template <typename T>
HK_INLINE T* hkBlockStream::Writer::write( const T* HK_RESTRICT src, int numBytes )
{
    T *dst = (T*)reserve( numBytes );
    hkString::memCpy( dst, src, numBytes );
    advance( numBytes );
    return dst;
}

template <typename T>
HK_INLINE T* hkBlockStream::Writer::write4( const T* HK_RESTRICT src, int numBytes )
{
    T *dst = (T*)reserve( numBytes );
    HK_ASSERT_NO_MSG( 0xf0545676, (0 == (0x3 & hkUlong(dst))) && (0 == (0x3 & numBytes)) );
    hkString::memCpy4( dst, src, numBytes >> 2 );
    advance( numBytes );
    return dst;
}

template <typename T>
HK_INLINE T* hkBlockStream::Writer::write16( const T* HK_RESTRICT src, int numBytes )
{
    HK_ASSERT_NO_MSG( 0xaf142e11, (numBytes & 0x0f) == 0 );
    T *dst = (T*)reserve( numBytes );
    hkString::memCpy16NonEmpty( dst, src, numBytes >> 4 );
    advance( numBytes );
    return dst;
}

HK_INLINE void hkBlockStream::Writer::checkConsistency()
{
#ifdef HK_DEBUG
    if( m_blockStream )
    {
        m_blockStream->checkConsistency();
    }
#endif
}


//////////////////////////////////////////////////////////////////////////
// Reader
//////////////////////////////////////////////////////////////////////////

HK_INLINE hkBlockStream::Reader::Reader() {}

HK_INLINE hkBlockStream::Reader::~Reader() {}

HK_INLINE void hkBlockStream::Reader::setToStartOfRange( const hkBlockStream::Range* HK_RESTRICT range )
{
    if( range->m_numElements == 0 )
    {
        setEmpty();
        return;
    }

    m_currentBlock = range->m_startBlock;
    m_currentByteLocation = ((char*)m_currentBlock->begin()) + range->m_startByteOffset;

    m_numElementsToReadInThisBlock = range->m_startBlockNumElements;
    HK_ASSERT_NO_MSG( 0xf0343443, m_numElementsToReadInThisBlock <= range->m_numElements && m_numElementsToReadInThisBlock <= int( m_currentBlock->getNumElements() ) );
    m_numElementsToReadInOtherBlocks = range->m_numElements - m_numElementsToReadInThisBlock;
    if( m_numElementsToReadInThisBlock == 0 )
    {
        setEmpty();
    }
}

HK_INLINE void hkBlockStream::Reader::setEmpty()
{
    m_currentByteLocation = HK_NULL;
}

HK_INLINE const hkBlockStream::Block* hkBlockStream::Reader::getCurrentBlock() const
{
    return m_currentBlock;
}

HK_INLINE int hkBlockStream::Reader::getNumUnreadElementsInThisBlock() const
{
    return m_numElementsToReadInThisBlock;
}

HK_INLINE void hkBlockStream::Reader::createBookmark( Bookmark& bookmarkOut ) const
{
    bookmarkOut.m_currentBlock = m_currentBlock;
    bookmarkOut.m_currentByteLocation = m_currentByteLocation;
    bookmarkOut.m_numElementsToReadInOtherBlocks = m_numElementsToReadInOtherBlocks;
    bookmarkOut.m_numElementsToReadInThisBlock = m_numElementsToReadInThisBlock;
}

template <typename T>
HK_INLINE const T* hkBlockStream::Reader::access() const
{
    return (const T*)m_currentByteLocation;
}

HK_INLINE const char* hkBlockStream::Reader::access() const
{
    return (const char*)m_currentByteLocation;
}

template <typename T>
HK_INLINE const T* hkBlockStream::Reader::advanceAndAccessNext( int thisElemSize )
{
    HK_ASSERT_NO_MSG( 0xf034df43, m_currentBlock );
    HK_ASSERT_NO_MSG( 0xf0347687, m_numElementsToReadInThisBlock > 0 );

    m_numElementsToReadInThisBlock = m_numElementsToReadInThisBlock - 1;

    if( m_numElementsToReadInThisBlock > 0 )
    {
        const char* nextLocation = hkAddByteOffsetConst( (const char*)m_currentByteLocation, thisElemSize );
        HK_ASSERT_NO_MSG( 0xf06576df, nextLocation <= m_currentBlock->end() );
        m_currentByteLocation = nextLocation;
        return (const T*)nextLocation;
    }
    else
    {
        const void* data = advanceToNewBlock();
        HK_ASSERT_NO_MSG( 0xf06576df, (const char*)m_currentByteLocation + thisElemSize <= m_currentBlock->end() );
        return (const T*)data;
    }
}

template <typename T>
HK_INLINE const T* hkBlockStream::Reader::advanceAndAccessNthNext( int n )
{
    while( n >= 0 )
    {
        HK_ASSERT_NO_MSG( 0xf034df43, m_currentBlock );
        HK_ASSERT_NO_MSG( 0xf0347687, m_numElementsToReadInThisBlock > 0 );

        const int nBlock = hkMath::min2( n, m_numElementsToReadInThisBlock );
        m_numElementsToReadInThisBlock -= nBlock;

        if( m_numElementsToReadInThisBlock > 0 )
        {
            const char* nextLocation = hkAddByteOffsetConst( (const char*)m_currentByteLocation, sizeof(T) * nBlock );
            HK_ASSERT_NO_MSG( 0xf06576df, nextLocation <= m_currentBlock->end() );
            m_currentByteLocation = nextLocation;
            return (const T*)nextLocation;
        }

        // Keep going
        n -= nBlock;
        advanceToNewBlock();
    }

    return HK_NULL;
}


//////////////////////////////////////////////////////////////////////////
// Modifier
//////////////////////////////////////////////////////////////////////////

HK_INLINE void hkBlockStream::Modifier::setToStartOfRange( const Range* HK_RESTRICT range )
{
    Reader::setToStartOfRange( range );
}

HK_INLINE void hkBlockStream::Modifier::setToStartOfStream( hkThreadLocalBlockStreamAllocator* HK_RESTRICT allocator, hkBlockStream* HK_RESTRICT stream )
{
    //  HK_ASSERT_NO_MSG( 0xf0345dfe, !blockStream->m_isLocked );   // enable this, but writer requires a finalize()
    //  HK_ON_DEBUG( blockStream->m_isLocked = true );
    Reader::setToStartOfStream( stream );
}

template <typename T>
HK_INLINE T* hkBlockStream::Modifier::access()
{
    return const_cast<T*>(Reader::access<T>());
}

HK_INLINE char* hkBlockStream::Modifier::access()
{
    return const_cast<char*>(Reader::access());
}

template <typename T>
HK_INLINE T* hkBlockStream::Modifier::advanceAndAccessNext( int thisElemSize )
{
    HK_ASSERT_NO_MSG( 0xf034df43, m_currentBlock );
    HK_ASSERT_NO_MSG( 0xf0347687, m_numElementsToReadInThisBlock > 0 );

    const char* nextLocation = hkAddByteOffsetConst( (const char*)m_currentByteLocation, thisElemSize );    // get this here as this is the most common code path and allows the compiler to optimize better
    HK_ASSERT_NO_MSG( 0xf06576df, nextLocation <= m_currentBlock->end() );
    m_currentByteLocation = nextLocation;
    m_numElementsToReadInThisBlock = m_numElementsToReadInThisBlock - 1;

    if( m_numElementsToReadInThisBlock > 0 )
    {
        return (T*)((void*)nextLocation);
    }
    else
    {
        const void* data = advanceToNewBlock();
        HK_ASSERT_NO_MSG( 0xf06576df, (const char*)m_currentByteLocation + thisElemSize <= m_currentBlock->end() );
        return (T*)(data);
    }
}


//////////////////////////////////////////////////////////////////////////
// Consumer
//////////////////////////////////////////////////////////////////////////

HK_INLINE const void* hkBlockStream::Consumer::advanceNoConsume( int thisElemSize )
{
    HK_ASSERT_NO_MSG( 0xf034df43, m_currentBlock );
    HK_ASSERT_NO_MSG( 0xf0347687, m_numElementsToReadInThisBlock > 0 );

    m_numElementsToFreeInThisBlock--;
    m_numElementsToReadInThisBlock--;

    if( m_numElementsToReadInThisBlock > 0 )
    {
        const char* nextLocation = hkAddByteOffsetConst( (const char*)m_currentByteLocation, thisElemSize );    // get this here as this is the most common code path and allows the compiler to optimize better
        HK_ASSERT_NO_MSG( 0xf06576df, nextLocation <= m_currentBlock->end() );
        m_currentByteLocation = nextLocation;
        return (const void*)nextLocation;
    }
    else
    {
        const void* data = Consumer::freeAndAdvanceToNewBlock();
        HK_ASSERT_NO_MSG( 0xf06576df, (const char*)m_currentByteLocation + thisElemSize <= m_currentBlock->end() );
        return data;
    }
}

HK_INLINE const void* hkBlockStream::Consumer::consumeAndAccessNext( int thisElemSize )
{
    HK_ASSERT_NO_MSG( 0xf034df43, m_currentBlock );
    HK_ASSERT_NO_MSG( 0xf0347687, m_numElementsToReadInThisBlock > 0 );

    int numElementsToReadInThisBlock = m_numElementsToReadInThisBlock - 1;
    const char* nextLocation = hkAddByteOffsetConst( (const char*)m_currentByteLocation, thisElemSize );    // get this here as this is the most common code path and allows the compiler to optimize better

    m_numElementsToReadInThisBlock = numElementsToReadInThisBlock;
    if( numElementsToReadInThisBlock > 0 )
    {
        HK_ASSERT_NO_MSG( 0xf06576df, nextLocation <= m_currentBlock->end() );
        m_currentByteLocation = nextLocation;
        return (const void*)nextLocation;
    }
    else
    {
        const void* data = freeAndAdvanceToNewBlock();
        HK_ASSERT_NO_MSG( 0xf06576df, (const char*)m_currentByteLocation + thisElemSize <= m_currentBlock->end() );
        return data;
    }
}

HK_INLINE hkBlockStream::Block* hkBlockStream::Consumer::getCurrentBlock()
{
    return const_cast<Block*>((const Block*)m_currentBlock);
}

HK_INLINE char* hkBlockStream::Consumer::getCurrentByteLocation()
{
    return const_cast<char*>((const char*)m_currentByteLocation);
}


//////////////////////////////////////////////////////////////////////////
// BatchConsumer
//////////////////////////////////////////////////////////////////////////

HK_INLINE void hkBlockStream::BatchConsumerBase::initConsumer( hkThreadLocalBlockStreamAllocator* allocator )
{
    HK_ASSERT_NO_MSG( 0xf0df34ff, m_blockStream == 0 && m_allocator == HK_NULL );
    m_allocator = allocator;
    m_numElementsToFreeInThisBlock = 0;
    m_blockToConsume = 0;
}

HK_INLINE const void* hkBlockStream::BatchConsumerBase::access()
{
    HK_ASSERT_NO_MSG( 0xf04f12de, !m_currentByteLocation || m_currentBlock->m_blockStream == m_blockStream );
    return Reader::access();
}

// batch consumer
HK_INLINE void hkBlockStream::BatchConsumerBase::setToStartOfRange( const hkBlockStream::Range* HK_RESTRICT range, hkBlockStream* stream )
{
    HK_ASSERT_NO_MSG( 0xf03dfd45, stream );
    if( 0 == range->m_numElements )
    {
        setEmpty();
        return;
    }

    //int numElementsToFreeInThisBlock = m_numElementsToFreeInThisBlock;
    // if we jump to a new block, remove the old
    if( range->m_startBlock != m_blockToConsume )
    {
        consumeData();
        m_blockToConsume = range->m_startBlock;
        m_numElementsToFreeInThisBlock = range->m_startBlockNumElements;
        m_blockStream = stream;
    }
    else
    {
        m_numElementsToFreeInThisBlock += range->m_startBlockNumElements;
    }
    hkBlockStream::Reader::setToStartOfRange( range );
    HK_ASSERT_NO_MSG( 0xf04587f0, range->m_startBlock );
    HK_ASSERT_NO_MSG( 0xf04587f1, range->m_startBlock->m_blockStream == stream );
}

HK_INLINE const void* hkBlockStream::BatchConsumerBase::consumeAndAccessNext( int thisElemSize )
{
    HK_ASSERT_NO_MSG( 0xf034df43, m_currentBlock );
    HK_ASSERT_NO_MSG( 0xf0347687, m_numElementsToReadInThisBlock > 0 );

    int numElementsToReadInThisBlock = m_numElementsToReadInThisBlock - 1;
    const char* nextLocation = hkAddByteOffsetConst( (const char*)m_currentByteLocation, thisElemSize );    // get this here as this is the most common code path and allows the compiler to optimize better

    m_numElementsToReadInThisBlock = numElementsToReadInThisBlock;
    if( numElementsToReadInThisBlock > 0 )
    {
        HK_ASSERT_NO_MSG( 0xf06576df, nextLocation <= m_currentBlock->end() );
        m_currentByteLocation = nextLocation;
        return (const void*)nextLocation;
    }
    else
    {
        const void* data = freeAndAdvanceToNewBlock();
        HK_ASSERT_NO_MSG( 0xf06576df, (const char*)m_currentByteLocation + thisElemSize <= m_currentBlock->end() );
        return data;
    }
}

HK_INLINE const void* hkBlockStream::BatchConsumerBase::advanceNoConsume( int thisElemSize )
{
    HK_ASSERT_NO_MSG( 0xf034df43, m_currentBlock );
    HK_ASSERT_NO_MSG( 0xf0347687, m_numElementsToReadInThisBlock > 0 );

    int numElementsToReadInThisBlock = m_numElementsToReadInThisBlock - 1;
    const char* nextLocation = hkAddByteOffsetConst( (const char*)m_currentByteLocation, thisElemSize );    // get this here as this is the most common code path and allows the compiler to optimize better
    m_numElementsToReadInThisBlock = numElementsToReadInThisBlock;
    m_numElementsToFreeInThisBlock--;
    if( numElementsToReadInThisBlock > 0 )
    {
        HK_ASSERT_NO_MSG( 0xf06576df, nextLocation <= m_currentBlock->end() );
        m_currentByteLocation = nextLocation;
        return (const void*)nextLocation;
    }
    else
    {
        const void* data = freeAndAdvanceToNewBlock();
        HK_ASSERT_NO_MSG( 0xf06576df, (const char*)m_currentByteLocation + thisElemSize <= m_currentBlock->end() );
        return data;
    }
}

HK_INLINE hkBlockStream::Block* hkBlockStream::BatchConsumerBase::getCurrentBlock()
{
    return const_cast<Block*>((const Block*)m_currentBlock);
}

HK_INLINE char* hkBlockStream::BatchConsumerBase::getCurrentByteLocation()
{
    return const_cast<char*> ((const char*)m_currentByteLocation);
}

HK_ALWAYS_INLINE void hkBlockStream::BatchConsumer::setToStartOfRange( const Range* range, hkBlockStream* stream )
{
    BatchConsumerBase::setToStartOfRange( range, stream );
}


//////////////////////////////////////////////////////////////////////////
// RandomAccessConsumer
//////////////////////////////////////////////////////////////////////////

HK_INLINE hkBlockStream::RandomAccessConsumer::~RandomAccessConsumer()
{
    if( m_currentBlock )
    {
        consumeCurrentBlock();
        m_currentBlock = HK_NULL;
    }
    // if the stream is not partially freed, we never run a cleanup function, so we need to fix the block stream consistency
    // by hand
    if( m_numElementsToFreeTotal && !m_blockStream->m_partiallyFreed )
    {
        fixupNumTotalElements();
    }
}

HK_INLINE void hkBlockStream::RandomAccessConsumer::consume( Block* block, int offset )
{
    if( block == m_currentBlock )
    {
        HK_ASSERT_NO_MSG( 0x27e7e817, offset != m_debugOffset ); // avoid freeing the same block twice
        m_numElementsToFreeInCurrentBlock++;
    }
    else
    {
        if( m_currentBlock )
        {
            consumeCurrentBlock();
        }
        m_currentBlock = block;
        HK_ASSERT_NO_MSG( 0x8f40c3f, m_blockStream == HK_NULL || block->m_blockStream == m_blockStream );
        m_numElementsToFreeInCurrentBlock = 1;
    }
    HK_ON_DEBUG( m_debugOffset = offset );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
