// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
//HK_REFLECTION_PARSER_EXCLUDE_FILE

#pragma once

#include <Common/Base/Container/BlockStream/hkBlockStream.h>


/// A writer is an iterator that adds data in a stream, allocating new blocks into the stream on the fly.
class hkBlockStream::Writer
{
public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, Writer );

    /// Constructor
    HK_INLINE Writer();

    /// Destructor
    HK_INLINE ~Writer();

    /// Set to the start of an existing stream to write data in the stream from the beginning.
    HK_EXPORT_COMMON void setToStartOfStream( hkThreadLocalBlockStreamAllocator* allocator, hkBlockStream* blockStream );

    /// Set to the end to an existing stream, to append data.
    HK_EXPORT_COMMON void setToEndOfStream( hkThreadLocalBlockStreamAllocator* allocator, hkBlockStream* blockStream );

    /// Returns true if the writer is attached to a stream.
    HK_INLINE bool isValid() const;

    /// Returns the address of the current block. Handle with care as you really need to
    /// understand how the block streams work if you want to use this feature.
    HK_INLINE Block* getCurrentBlock() const;

    /// Returns the current byte offset from the start of the current block.
    HK_INLINE int getCurrentByteOffset() const;

    /// Returns the number of bytes left in the current block.
    HK_INLINE int getBlockBytesLeft() const;

    /// Returns the total number of elements in the writer's stream.
    HK_INLINE int getTotalNumElems() const;

    /// Returns the total number of elements in the writer's stream. This method can be called before finalize().
    HK_EXPORT_COMMON int getTotalNumElemsNotFinalized() const;

    /// Call this method to put the stream back in a consistent and trimmed state.
    HK_EXPORT_COMMON void finalize();

    //
    // Advance functions
    //

    /// Write an element of \a numBytes of \a src to the stream.
    /// Returns a pointer to the written data.
    template <typename T>
    HK_INLINE T* write( const T* HK_RESTRICT src, int numBytes );

    /// Write an element of \a numBytes of \a src to the stream. \a numBytes must be a multiple of 4.
    /// Returns a pointer to the written data.
    template <typename T>
    HK_INLINE T* write4( const T* HK_RESTRICT src, int numBytes );

    /// Write an element of \a numBytes of \a src to the stream. \a numBytes must be a multiple of 16.
    /// Returns a pointer to the written data.
    template <typename T>
    HK_INLINE T* write16( const T* HK_RESTRICT src, int numBytes );

    /// Reserve numBytes in the stream.
    /// Returns a pointer where data can be written to.
    /// This does not advance the stream automatically, you must call advance() manually afterwards.
    template<typename T>
    HK_INLINE T* reserve( int numBytes = sizeof(T) );
    HK_INLINE char* reserve( int numBytes );    // default overload

    /// Advance by numBytes. This will move the write location in the current block.
    HK_INLINE void advance( int numBytes );

    /// Undo the last advance by numBytes.
    HK_INLINE void undoAdvance( int numBytes, int initiallyReservedBytes );

    /// Debug helper.
    HK_INLINE void checkConsistency();

protected:

    /// Allocates a new block, appends it to the end of the stream and returns the pointer to the new
    /// block's storage space.
    HK_EXPORT_COMMON void *allocateAndAccessNewBlock();

    /// Wrap up the current block and link the next block.
    /// Make sure that \a thisBlock is actually pointing to the last block in the stream!
    HK_INLINE void finalizeLastBlock( Block* HK_RESTRICT thisBlock, Block* HK_RESTRICT nextBlock, int thisBlockNumElements, int bytesUsed );

protected:

    /// The block stream this writer is working on.
    hkBlockStream* m_blockStream;

    /// The underlying thread allocator that will allocate blocks when writing.
    hkThreadLocalBlockStreamAllocator* m_tlAllocator;

    /// The current block where data is written to.
    Block* m_currentBlock;

    /// The byte offset of the current element from the start of the current block.
    int m_currentByteOffset;

    /// The number of elements stored in the current block.
    int m_currentBlockNumElems;

    // Debugging helpers
    HK_DEBUG_ONLY_MEMBER( hkBool, m_finalized );
    HK_DEBUG_ONLY_MEMBER( int, m_accessSize );      // the numBytes used in the last call to reserve()

    friend class Range;
};


/// A reader is a read-only iterator that goes through blocks from a stream and can return the elements it stores.
class hkBlockStream::Reader
{
public:

    struct Bookmark
    {
        const Block* m_currentBlock;
        const char* m_currentByteLocation;
        int m_numElementsToReadInOtherBlocks;
        int m_numElementsToReadInThisBlock;
    };

public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, Reader );

    HK_INLINE Reader();
    HK_INLINE ~Reader();

    /// Sets the reader to the start of a stream
    HK_EXPORT_COMMON void setToStartOfStream( const hkBlockStream* stream );

    /// Sets the reader to the start of a range.
    HK_INLINE void setToStartOfRange( const Range* range );

    /// Invalidate the reader.
    HK_INLINE void setEmpty();

    /// Returns the address of the current block. Handle with care as you really need to
    /// understand how the block streams work if you want to use this feature.
    HK_INLINE const Block* getCurrentBlock() const;

    /// Returns the number of elements left to read in the current block
    HK_INLINE int getNumUnreadElementsInThisBlock() const;

    /// Store the current location in the stream as a bookmark.
    HK_INLINE void createBookmark( Bookmark& bookmarkOut ) const;

    /// Move the reader to the bookmarked location in the stream.
    HK_EXPORT_COMMON void gotoBookmark( const Bookmark& bookmark );

    /// Access the current element
    template <typename T>
    HK_INLINE const T* access() const;
    HK_INLINE const char* access() const;

    /// Advance and access the next element, assumes the current entry is valid
    template <typename T>
    HK_INLINE const T* advanceAndAccessNext( int thisElemSize );

    /// Advance and access the n-th next element, assumes the current entry is valid
    template <typename T>
    HK_INLINE const T* advanceAndAccessNthNext( int n );

protected:

    /// Advances to the next block in the stream.
    HK_EXPORT_COMMON const void* advanceToNewBlock();

protected:

    /// Pointer to the current block, only valid if m_currentByteLocation is set
    const Block* m_currentBlock;

    /// Pointer to the current element in the block.
    const char* m_currentByteLocation;

    /// Number of elements left to read in other blocks, to know whether to prefetch next block
    int m_numElementsToReadInOtherBlocks;

    /// Number of elements left to read in the block.
    int m_numElementsToReadInThisBlock;
};


/// Modifier inherits from reader but allows to get read-write access to modify its current element.
class hkBlockStream::Modifier : public Reader
{
public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, Modifier );

    HK_INLINE Modifier() {}

    /// Sets the modifier to the first block of a stream.
    HK_INLINE void setToStartOfStream( hkThreadLocalBlockStreamAllocator* allocator, hkBlockStream* stream );

    /// Set to a whole range.
    HK_INLINE void setToStartOfRange( const Range* range );

    /// Access the current element
    template <typename T>
    HK_INLINE T* access();
    HK_INLINE char* access();

    /// Advance and access the next element, assumes the current entry is valid
    template <typename T>
    HK_INLINE T* advanceAndAccessNext( int thisElemSize );

protected:

    /// Advances to the next block (when we are at the end of a block).
    HK_EXPORT_COMMON void* advanceToNewBlock();

protected:

    hkThreadLocalBlockStreamAllocator* m_allocator;
};


/// A reader, which frees the memory once it is fully read.
/// You can run different Consumers using different ranges of the same block stream,
/// even in multiple threads.
/// By default the block stream becomes useless as soon as one consumer worked on it.
/// If you want to continue to use the block stream you have to:
///    - finalize with a call to hkBlockStream::fixupConsumedBlocks(). This will free completely unused blocks
class hkBlockStream::Consumer : public Reader
{
public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, Consumer );

    HK_INLINE Consumer() {}

    /// Sets the consumer to the first block of a stream.
    HK_EXPORT_COMMON void setToStartOfStream( hkThreadLocalBlockStreamAllocator* allocator, hkBlockStream* stream );

    /// Set to a whole range.
    HK_EXPORT_COMMON void setToStartOfRange( hkThreadLocalBlockStreamAllocator* allocator, hkBlockStream* stream, const Range* range );

    /// Advance and access the next element, freeing the old data. Assumes the current entry is valid.
    HK_INLINE const void* consumeAndAccessNext( int thisElemSize );

    /// Advance and access the next element, without consuming. After doing this, it is no longer possible
    /// to walk a contiguous range, as there may be gaps in the blocks.
    HK_INLINE const void* advanceNoConsume( int thisElemSize );

protected:

    /// Frees current block and advance to next one.
    HK_EXPORT_COMMON const void* freeAndAdvanceToNewBlock();

    /// Non-const accessors (casts the const away).
    HK_INLINE Block* getCurrentBlock();
    HK_INLINE char* getCurrentByteLocation();

protected:

    hkBlockStream* m_blockStream;
    int m_numElementsToFreeInThisBlock;
    hkThreadLocalBlockStreamAllocator* m_allocator;
};


/// A reader, which frees the memory once it is fully read.
/// This is similar to Consumer except this is optimized for consuming multiple ranges
class hkBlockStream::BatchConsumerBase : public Reader
{
public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, BatchConsumerBase );

    HK_INLINE BatchConsumerBase() { m_blockStream = 0; HK_ON_DEBUG( m_allocator = HK_NULL ); }
    HK_INLINE ~BatchConsumerBase() { HK_ASSERT_NO_MSG( 0xf0df34fe, m_blockStream == 0 ); }

    /// Sets the allocator
    HK_INLINE void initConsumer( hkThreadLocalBlockStreamAllocator* allocator );

    HK_EXPORT_COMMON void finalizeConsumer();

    /// Non-const accessors (casts the const away).
    HK_INLINE Block* getCurrentBlock();
    HK_INLINE char* getCurrentByteLocation();

    HK_INLINE const void* access();

    HK_INLINE const void* advanceNoConsume( int thisElemSize );

protected:

    /// Set to a whole range assuming the original stream
    HK_INLINE void setToStartOfRange( const Range* range, hkBlockStream* stream );

    /// Advance and access the next element, freeing the old data. Assumes the current entry is valid.
    HK_INLINE const void* consumeAndAccessNext( int thisElemSize );

    /// Frees current block and advance to next one.
    HK_EXPORT_COMMON const void* freeAndAdvanceToNewBlock();

    HK_INLINE void _consumeData();
    HK_EXPORT_COMMON void consumeData();    // out of line version

protected:

    hkBlockStream* m_blockStream;
    Block* m_blockToConsume; // this points to the block which has data to be freed
    int m_numElementsToFreeInThisBlock;
    hkThreadLocalBlockStreamAllocator* m_allocator;
};


/// This is similar to BatchConsumerBase plus the ability to set it to range or full stream
class hkBlockStream::BatchConsumer : public BatchConsumerBase
{
public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, BatchConsumer );

    /// Sets the consumer to the first block of a stream.
    HK_EXPORT_COMMON void setToStartOfStream( hkBlockStream* stream );

    /// Set to a whole range assuming the original stream
    HK_ALWAYS_INLINE void setToStartOfRange( const Range* range, hkBlockStream* stream );
};


/// Like BatchConsumer but has the concept of a default stream and works on ranges only
class hkBlockStream::BatchConsumerRangeOnly : public BatchConsumerBase
{
public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, BatchConsumerRangeOnly );

    HK_INLINE void initConsumer( hkThreadLocalBlockStreamAllocator* allocator ) { BatchConsumerBase::initConsumer( allocator ); }
    HK_INLINE void initConsumer( hkThreadLocalBlockStreamAllocator* allocator, hkBlockStream* stream ) { BatchConsumerBase::initConsumer( allocator ); m_blockStream = stream; }

    HK_INLINE void setStream( hkBlockStream* stream ) { if( stream != m_blockStream ) { setStreamImpl( stream ); } }

    /// Set to a whole range assuming the original stream
    HK_INLINE void setToStartOfRange( const Range* range ) { BatchConsumerBase::setToStartOfRange( range, m_blockStream ); }

protected:

    HK_EXPORT_COMMON void setStreamImpl( hkBlockStream* stream );
};


/// Helper class which helps to consume random blocks in the engine
class hkBlockStream::RandomAccessConsumer
{
public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, RandomAccessConsumer );

    HK_INLINE RandomAccessConsumer( hkThreadLocalBlockStreamAllocator* allocator, hkBlockStream* blockStream )
        : m_blockStream( blockStream ), m_allocator( allocator ), m_currentBlock( HK_NULL ) {
        m_numElementsToFreeTotal = 0;
    }
    HK_INLINE ~RandomAccessConsumer();

    HK_INLINE void consume( Block* block, int offset );

protected:

    HK_EXPORT_COMMON void consumeCurrentBlock();
    HK_EXPORT_COMMON void fixupNumTotalElements();

protected:

    hkBlockStream* m_blockStream;               // used for debugging
    HK_DEBUG_ONLY_MEMBER(int, m_debugOffset );  // used for debugging
    hkThreadLocalBlockStreamAllocator* m_allocator;
    Block* m_currentBlock;
    int m_numElementsToFreeInCurrentBlock;
    int m_numElementsToFreeTotal;
};


#include <Common/Base/Container/BlockStream/hkBlockStreamIterators.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
