// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Container/BlockStream/hkBlockStreamIterators.h>
#include <Common/Base/Container/BlockStream/hkBlockStreamRange.h>
#include <Common/Base/Container/BlockStream/Allocator/hkThreadLocalBlockStreamAllocator.h>
#include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>

HK_INLINE void hkBlockStream_atomicDecreaseElementCount( hkBlockStream::Block* block, int numElementsToFree )
{
    HK_ASSERT_NO_MSG( 0xab759838, block->getNumElements() - numElementsToFree >= 0 );
    hkAtomic::add( &block->m_numElementsAndBytesUsed, hkUint32( -numElementsToFree ) );
}


//////////////////////////////////////////////////////////////////////////
// Writer
//////////////////////////////////////////////////////////////////////////

void hkBlockStream::Writer::setToStartOfStream( hkThreadLocalBlockStreamAllocator* allocator, hkBlockStream* blockStream )
{
    HK_ASSERT_NO_MSG( 0xf0345dfe, !blockStream->m_isLocked );
    HK_ASSERT( 0x1990dde0, allocator->getBlockStreamAllocator() == blockStream->m_allocator, "The writer and the stream being written to must use the same block allocator" );
    HK_ASSERT_NO_MSG( 0xf0456567, blockStream->isEmpty() && blockStream->m_numTotalElements == 0 );
    HK_ON_DEBUG( blockStream->m_isLocked = true );
    HK_ON_DEBUG( m_finalized = false );
    hkBlockStream::Block* firstBlock = blockStream->beginRw();
    m_tlAllocator = allocator;
    m_blockStream = blockStream;
    m_currentByteOffset = 0;
    m_currentBlockNumElems = 0;

    hkMath::prefetch128( firstBlock );
    m_currentBlock = firstBlock;
}

void hkBlockStream::Writer::setToEndOfStream( hkThreadLocalBlockStreamAllocator* allocator, hkBlockStream* blockStream )
{
    HK_ASSERT_NO_MSG( 0xf0345dfe, !blockStream->m_isLocked );
    HK_ASSERT( 0x1990dde0, allocator->getBlockStreamAllocator() == blockStream->m_allocator, "The writer and the stream being written to must use the same block allocator" );
    HK_ON_DEBUG( blockStream->m_isLocked = true );
    HK_ON_DEBUG( m_finalized = false );

    hkBlockStream::Block* lastBlock = blockStream->lastRw();
    m_tlAllocator = allocator;
    m_blockStream = blockStream;

    m_currentByteOffset = lastBlock->getNumBytesUsed();
    m_currentBlockNumElems = lastBlock->getNumElements();

    
    blockStream->m_numTotalElements = blockStream->m_numTotalElements - m_currentBlockNumElems;
    HK_ASSERT_NO_MSG( 0xf0dfde34, blockStream->m_numTotalElements >= 0 );
    m_currentBlock = lastBlock;
}

void hkBlockStream::Writer::finalize()
{
    HK_ASSERT_NO_MSG( 0xf0445dfe, m_blockStream->m_isLocked );
    HK_ON_DEBUG( m_blockStream->m_isLocked = false );
    HK_ON_DEBUG( m_finalized = true );

    // Only remove the last block of the stream if it is empty and if it is not the only block in the stream (which
    // has been allocated when the block stream itself was created).
    if( m_currentBlockNumElems == 0 && m_blockStream->m_blocks.getSize() > 1 )
    {
        m_blockStream->popBack( m_tlAllocator );
    }
    else
    {
        finalizeLastBlock( m_currentBlock, HK_NULL, m_currentBlockNumElems, m_currentByteOffset );
    }

    HK_ON_DEBUG( m_blockStream->checkConsistency() );
    m_currentBlock = HK_NULL;
}

int hkBlockStream::Writer::getTotalNumElemsNotFinalized() const
{
    HK_ASSERT_NO_MSG( 0xaf13e21f, m_finalized == false );
    return m_currentBlockNumElems + m_blockStream->m_numTotalElements;
}

// Internal functions
void* hkBlockStream::Writer::allocateAndAccessNewBlock()
{
    hkBlockStream::Block* nextBlock = m_blockStream->blockAlloc( m_tlAllocator );

    finalizeLastBlock( m_currentBlock, nextBlock, m_currentBlockNumElems, m_currentByteOffset );

    m_currentBlock = nextBlock;
    m_currentByteOffset = 0;
    m_currentBlockNumElems = 0;

    return nextBlock->begin();
}


//////////////////////////////////////////////////////////////////////////
// Reader
//////////////////////////////////////////////////////////////////////////

void hkBlockStream::Reader::setToStartOfStream( const hkBlockStream* HK_RESTRICT stream )
{
    const hkBlockStream::Block* startBlock = stream->begin();

    HK_ASSERT_NO_MSG( 0xf0dfde39, startBlock != HK_NULL );

    m_currentBlock = startBlock;
    m_currentByteLocation = ((char*)startBlock->begin());

    int numDatasThisBlock = startBlock->getNumElements();
    m_numElementsToReadInThisBlock = numDatasThisBlock;
    m_numElementsToReadInOtherBlocks = HK_INT32_MAX;
    if( numDatasThisBlock == 0 )
    {
        m_currentByteLocation = HK_NULL;
    }
}

const void* hkBlockStream::Reader::advanceToNewBlock()
{
    if( m_numElementsToReadInOtherBlocks == 0 )
    {
        m_currentBlock = HK_NULL;
        m_currentByteLocation = HK_NULL;
        return HK_NULL;
    }

    const Block* currentBlock = m_currentBlock->m_next;
    m_currentBlock = currentBlock;
    if( !currentBlock )
    {
        m_currentByteLocation = HK_NULL;
        return HK_NULL;
    }

    int numElementsToReadInThisBlock = hkMath::min2( (int)currentBlock->getNumElements(), (int)m_numElementsToReadInOtherBlocks );
    if( numElementsToReadInThisBlock <= 0 )
    {
        m_currentByteLocation = HK_NULL;
        return HK_NULL;
    }
    m_numElementsToReadInThisBlock = numElementsToReadInThisBlock;
    m_numElementsToReadInOtherBlocks = m_numElementsToReadInOtherBlocks - m_numElementsToReadInThisBlock;
    m_currentByteLocation = (const char*)currentBlock->begin();

    hkMath::prefetch128( currentBlock->m_next );    // prefetch the next block header
    hkMath::prefetch128( hkAddByteOffsetConst( currentBlock, 128 ) );

    HK_ASSERT_NO_MSG( 0xf06576df, m_numElementsToReadInThisBlock > 0 );
    return currentBlock->begin();
}

void hkBlockStream::Reader::gotoBookmark( const Bookmark& bookmark )
{
    const Block* currentBlock = bookmark.m_currentBlock;
    if( currentBlock != m_currentBlock )
    {
        hkMath::prefetch128( currentBlock->m_next );    // prefetch the next block header
        hkMath::prefetch128( hkAddByteOffsetConst( currentBlock, 128 ) );
        currentBlock->begin();
    }
    m_currentBlock = bookmark.m_currentBlock;
    m_currentByteLocation = bookmark.m_currentByteLocation;
    m_numElementsToReadInOtherBlocks = bookmark.m_numElementsToReadInOtherBlocks;
    m_numElementsToReadInThisBlock = bookmark.m_numElementsToReadInThisBlock;
}


//////////////////////////////////////////////////////////////////////////
// Modifier
//////////////////////////////////////////////////////////////////////////

void* hkBlockStream::Modifier::advanceToNewBlock()
{
    if( m_numElementsToReadInOtherBlocks == 0 )
    {
        m_currentBlock = HK_NULL;
        m_currentByteLocation = HK_NULL;
        return HK_NULL;
    }

    m_currentBlock = m_currentBlock->m_next;
    if( !m_currentBlock )
    {
        m_currentByteLocation = HK_NULL;
        return HK_NULL;
    }

    int numElementsToReadInThisBlock = hkMath::min2( (int)m_currentBlock->getNumElements(), (int)m_numElementsToReadInOtherBlocks );
    if( numElementsToReadInThisBlock <= 0 )
    {
        m_currentByteLocation = HK_NULL;
        return HK_NULL;
    }
    m_numElementsToReadInThisBlock = numElementsToReadInThisBlock;
    m_numElementsToReadInOtherBlocks = m_numElementsToReadInOtherBlocks - m_numElementsToReadInThisBlock;
    m_currentByteLocation = (const char*)m_currentBlock->begin();

    HK_ASSERT_NO_MSG( 0xf06576df, m_numElementsToReadInThisBlock > 0 );
    return const_cast<void*>(m_currentBlock->begin());
}


//////////////////////////////////////////////////////////////////////////
// Consumer
//////////////////////////////////////////////////////////////////////////

void hkBlockStream::Consumer::setToStartOfStream( hkThreadLocalBlockStreamAllocator* HK_RESTRICT allocator, hkBlockStream* HK_RESTRICT stream )
{
    stream->m_partiallyFreed = 1;
    //  HK_ASSERT_NO_MSG( 0xf0345dfe, !blockStream->m_isLocked );   // enable this, but it requires a finalize()
    //  HK_ON_DEBUG( blockStream->m_isLocked = true );
    hkBlockStream::Reader::setToStartOfStream( stream );
    m_blockStream = stream;
    m_allocator = allocator;
    m_numElementsToFreeInThisBlock = m_numElementsToReadInThisBlock;
}

void hkBlockStream::Consumer::setToStartOfRange( hkThreadLocalBlockStreamAllocator* allocator, hkBlockStream* HK_RESTRICT stream, const hkBlockStream::Range* HK_RESTRICT range )
{
    if( 0 == range->m_numElements )
    {
        setEmpty();
        return;
    }

    stream->m_partiallyFreed = 1;
    hkBlockStream::Reader::setToStartOfRange( range );
    m_blockStream = stream;
    m_allocator = allocator;
    m_numElementsToFreeInThisBlock = m_numElementsToReadInThisBlock;

    HK_ASSERT_NO_MSG( 0xf0345456, range->m_startBlock );
    if( HK_NULL == range->m_startBlock )
    {
        m_numElementsToFreeInThisBlock = 0;
    }
}

const void* hkBlockStream::Consumer::freeAndAdvanceToNewBlock()
{
    hkBlockStream::Block* HK_RESTRICT toFree = getCurrentBlock();
    int numElementsToFreeInThisBlock = m_numElementsToFreeInThisBlock;

    const void* ret = hkBlockStream::Reader::advanceToNewBlock();

    m_numElementsToFreeInThisBlock = m_numElementsToReadInThisBlock;

    // Free consumed block
    if( int( toFree->getNumElements() ) == numElementsToFreeInThisBlock )
    {
        m_blockStream->freeBlock( m_allocator, toFree );
    }
    else
    {
        hkBlockStream_atomicDecreaseElementCount( toFree, numElementsToFreeInThisBlock );
        // Note: there is still a tiny chance that 2 threads will not free a block,
        // but the blocks will be freed later anyway in the destructor of the block stream
    }

    return ret;
}

HK_INLINE void hkBlockStream::BatchConsumerBase::_consumeData()
{
    if( m_numElementsToFreeInThisBlock )
    {
        HK_ASSERT_NO_MSG( 0xf023dffe, m_blockToConsume->m_blockStream == m_blockStream );

        // Free consumed block
        // note that getNumElements() might be stale due to missing memory synchronization, but this not an issue
        // here as it only acts as an optimization to avoid calling the atomic operation.
        if( int( m_blockToConsume->getNumElements() ) == m_numElementsToFreeInThisBlock )
        {
            //Log_Info( "Free {:x}", m_blockToConsume );
            m_blockStream->freeBlock( m_allocator, m_blockToConsume );
        }
        else
        {
            hkBlockStream_atomicDecreaseElementCount( m_blockToConsume, m_numElementsToFreeInThisBlock );
        }
        m_numElementsToFreeInThisBlock = 0;
        m_blockStream->m_partiallyFreed = 1;
    }
}
void hkBlockStream::BatchConsumerBase::consumeData()
{
    _consumeData();
}

void hkBlockStream::BatchConsumerBase::finalizeConsumer()
{
    _consumeData();
    m_blockStream = 0;
    m_allocator = HK_NULL;
}

void hkBlockStream::BatchConsumer::setToStartOfStream( hkBlockStream* stream )
{
    if( stream->isEmpty() )
    {
        setEmpty();
        return;
    }
    _consumeData(); // we assume at this point that the new stream will be different from the old

    hkBlockStream::Reader::setToStartOfStream( stream );

    m_blockStream = stream;
    m_blockToConsume = getCurrentBlock();
    m_numElementsToFreeInThisBlock = m_numElementsToReadInThisBlock;
}

const void* hkBlockStream::BatchConsumerBase::freeAndAdvanceToNewBlock()
{
    const void* ret = hkBlockStream::Reader::advanceToNewBlock();

    if( m_currentBlock && m_currentBlock != m_blockToConsume )
    {
        _consumeData();
        m_blockToConsume = getCurrentBlock();
        m_numElementsToFreeInThisBlock = m_numElementsToReadInThisBlock;
    }
    return ret;
}

void hkBlockStream::BatchConsumerRangeOnly::setStreamImpl( hkBlockStream* stream )
{
    _consumeData();
    m_blockStream = stream;
}

void hkBlockStream::RandomAccessConsumer::consumeCurrentBlock()
{
    if( !m_blockStream )
    {
        return; // feature is disabled
    }
    HK_ASSERT_NO_MSG( 0xe162511, m_currentBlock );
    HK_ASSERT_NO_MSG( 0x74b39014, m_currentBlock->m_blockStream == m_blockStream );

    hkBlockStream::Block* HK_RESTRICT toFree = m_currentBlock;

    // Free consumed block
    if( int( toFree->getNumElements() ) == m_numElementsToFreeInCurrentBlock )
    {
        m_blockStream->freeBlock( m_allocator, toFree );
    }
    else
    {
        hkBlockStream_atomicDecreaseElementCount( toFree, m_numElementsToFreeInCurrentBlock );
        // Note: there is still a tiny chance that 2 threads will not free a block because
        // they both decrement the m_numElementsToFreeInCurrentBlock at the very same moment.
        // but we blocks will be freed later anyway in the destructor of the block stream
    }
}

void hkBlockStream::RandomAccessConsumer::fixupNumTotalElements()
{
    hkAtomic::add( (hkUint32*)&m_blockStream->m_numTotalElements, hkUint32( -m_numElementsToFreeTotal ) );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
