// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
//HK_REFLECTION_PARSER_EXCLUDE_FILE

#pragma once

class hkBlockStreamAllocator;
class hkThreadLocalBlockStreamAllocator;
class hkStreamReader;

/// A block stream is a sequence of arbitrary memory blocks to be accessed by readers and writers.
/// Writers make it grow by allocating more block into the stream as they add more data, while
/// Consumers free the blocks as they read it (so the consumed blocks of the stream are
/// no longer valid).
class hkBlockStream
{
public:

    // Iterators (see hkBlockStreamIterators.h)
    class Reader;
    class Writer;
    class Modifier;
    class Consumer;
    class BatchConsumer;
    class BatchConsumerBase;
    class BatchConsumerRangeOnly;
    class RandomAccessConsumer;

    // Ranges (see hkBlockStreamRange.h)
    class Range;
    class LinkedRange;

    /// The basic storage container for the block stream.
    /// Each block can be filled with elements of arbitrary size (maximum allowed size is BLOCK_DATA_SIZE).
    /// Blocks should be aligned to 'BLOCK_ALIGNMENT' bytes.
    struct HK_EXPORT_COMMON Block
    {
    public:

        /// Defines the size of the block depending on the platform.
        enum
        {
#if defined(HK_PLATFORM_CTR)
            BLOCK_TOTAL_SIZE = 512,
#else
            BLOCK_TOTAL_SIZE = 1024 * 4 - 256,  // allow for hkMemoryRouter overhead (HNP-1362)
#endif
            BLOCK_HEADER_SIZE = 32,
            BLOCK_DATA_SIZE = BLOCK_TOTAL_SIZE - BLOCK_HEADER_SIZE,
            BLOCK_ALIGNMENT = 128,
        };

        /// Integral type big enough to count the number of bytes used.
        
        typedef hkUint16 CountType;

    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, Block );

        /// Constructor.
        HK_INLINE Block();

        /// Zeros the block header. Usually called by the stream just after the block has been handed by the allocator.
        HK_INLINE void setHeaderToZero();

        // Read-only and read-write access to the beginning of the data in the block.
        HK_INLINE void* begin();
        HK_INLINE const void* begin() const;

        // Returns a pointer past the end of the block.
        HK_INLINE void* end();
        HK_INLINE const void* end() const;

        // Accessors for the number of elements in the block.
        HK_INLINE CountType getNumElements() const;
        HK_INLINE void setNumElements( CountType numElements );

        // Accessors for the number of bytes used in the block.
        HK_INLINE CountType getNumBytesUsed() const;
        HK_INLINE void setNumBytesUsed( CountType numBytes );

    public:

        /// This variable stores two 16-bits integers, respectively the number of elements that have yet not been
        /// consumed, and the number of bytes used in the block. They should be accessed by the get and set functions.
        /// numElements needs to be atomically decreased by Consumers, but atomic operations are only guaranteed
        /// on 32 bits int on every platform, so that's why it's stored in the least 16 significant bits.
        HK_ALIGN16( hkUint32 m_numElementsAndBytesUsed );

        int m_blockIndexInStream;                                       ///< The index of this block in a stream
        Block* m_next;                                                  ///< Next block in this linked list
        HK_DEBUG_ONLY_MEMBER( hkBlockStreamAllocator*, m_allocator );   ///< Filled in only in debug, used for debugging checks and reporting
        HK_DEBUG_ONLY_MEMBER( hkBlockStream*, m_blockStream );          ///< Filled in only in debug, used for debugging checks and reporting

        HK_ALIGN_REAL( hkUchar m_data[BLOCK_DATA_SIZE] );               ///< The actual data
    };

public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, hkBlockStream );

    /// Constructor.
    HK_INLINE hkBlockStream( hkThreadLocalBlockStreamAllocator* tlAllocator, const char* debugName = HK_NULL );

    /// Empty constructor. You must call init() to initialize the stream.
    HK_INLINE hkBlockStream() {}

    /// Destructor.
    /// Note: You must call clear() before this (becuase you must provide a thread local allocator).
    HK_INLINE ~hkBlockStream();

    //
    // Block management functions
    //

    /// Initializes the stream by taking a new block from the given allocator.
    HK_EXPORT_COMMON void init( hkThreadLocalBlockStreamAllocator* tlAllocator, const char* debugName = HK_NULL );

    /// Resets the stream : clears it and allocates 1 empty block.
    HK_EXPORT_COMMON void reset( hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator );

    /// Clears the stream without resetting it.
    /// Only use if you call append immediately after this or before destruction.
    HK_EXPORT_COMMON void clear( hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator );

    /// Appends all the blocks of the stream to this one, and delete the other.
    /// If both streams are empty the stream will be initialized with 1 empty block.
    HK_EXPORT_COMMON void append( hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator, hkBlockStream* inStream );

    /// Removes the consumed blocks from the stream.
    /// Call this if you want to reuse this block stream after you consumed some blocks.
    HK_EXPORT_COMMON void fixupConsumedBlocks( hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator );

    /// Explicitly frees a block in the stream.
    HK_EXPORT_COMMON void freeBlock( hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator, Block* HK_RESTRICT block );

    //
    // Utilities
    //

    /// Returns the underlying block allocator.
    HK_INLINE const hkBlockStreamAllocator* getAllocator() const;

    /// Returns true if the stream has no elements in its blocks (equivalent to getTotalNumElems() == 0).
    HK_INLINE bool isEmpty() const;

    /// Returns the total number of elements stored in the stream's blocks.
    HK_INLINE int getTotalNumElems() const;

    /// Returns the total size of all the blocks in the stream.
    HK_INLINE int getTotalBytesAllocated() const;

    /// Calculates the total bytes actually used.
    HK_EXPORT_COMMON int calcTotalBytesUsed() const;

    /// Serialize to binary, just raw data, can only be loaded by the same platform
    HK_EXPORT_COMMON void serialize( hkStreamWriter& writer ) const;

    /// Deserialize from binary, just raw data, can only be loaded by the same platform
    HK_EXPORT_COMMON void deserialize( hkStreamReader& reader, hkThreadLocalBlockStreamAllocator* tlAllocator );

    //
    // Consistency checks
    //

    /// Checks the consistency of the block headers, not the block contents
    HK_EXPORT_COMMON void checkConsistency() const;

    /// Checks whether an input array of ranges is consistent with the block stream
    HK_EXPORT_COMMON void checkConsistencyWithGrid(
        const Range* rangesIn, int numRanges, int rangeStriding, bool allowForUnusedData ) const;

    /// Checks whether an array of ranges sorted according to Range::compareRange() is consistent with the block
    /// stream.
    HK_EXPORT_COMMON void checkConsistencyWithSortedRanges(
        const Range* sortedRanges, int numRanges, int rangeStriding, bool allowForUnusedData ) const;

    /// Checks whether a single range is consistent with the block stream.
    HK_EXPORT_COMMON void checkConsistencyOfRange( const Range& range ) const;

protected:

    //
    // Accessors for beginning and end of stream.
    //

    /// Returns the first block of the stream.
    HK_INLINE const Block* begin() const;

    /// Returns the last block of the stream.
    HK_INLINE const Block* last() const;

    /// Returns the first block of the stream, with read-write access.
    HK_INLINE Block* beginRw();

    /// Returns the last block of the stream, with read-write access.
    HK_INLINE Block* lastRw();

    /// Allocate and append a new block to the stream.
    Block* blockAlloc( hkThreadLocalBlockStreamAllocator* tlAllocator );

    /// Remove and free the last block of the stream.
    Block* popBack( hkThreadLocalBlockStreamAllocator* tlAllocator );

protected:

    /// The base allocator used to get the blocks
    hkBlockStreamAllocator* m_allocator;

    /// The total number of elements in this stream. This value is only valid after a call to Writer::finalizeLastBlock().
    int m_numTotalElements;

    /// True if some blocks of the stream have been freed.
    hkBool m_partiallyFreed;

    /// Lock for readers and writers.
    hkBool m_isLocked;

    /// Blocks owned by this stream.
    hkInplaceArrayAligned16< Block*, 24 > m_blocks;

public:

    /// A name to identify this block stream. Useful when debugging.
    const char* m_debugName;
};

#include <Common/Base/Container/BlockStream/hkBlockStream.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
