// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Container/BlockStream/hkBlockStream.h>
#include <Common/Base/Container/BlockStream/hkBlockStreamRange.h>
#include <Common/Base/Container/BlockStream/Allocator/hkThreadLocalBlockStreamAllocator.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/DebugUtil/DeterminismUtil/hkCheckDeterminismUtil.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>


void hkBlockStream::init( _In_ hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator, const char* debugName )
{
    //HK_ASSERT( 0xf0343234, tlAllocator->m_blockStreamManager == allocator, "Your thread local allocator does not match the allocator" );
    m_allocator = tlAllocator->getBlockStreamAllocator();
    m_numTotalElements = 0;
    HK_ON_DEBUG(m_isLocked = false);
    m_partiallyFreed = false;
    blockAlloc(tlAllocator);
    m_debugName = debugName;
    HK_ON_DEBUG(checkConsistency());
}

void hkBlockStream::reset( _In_ hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator )
{
    HK_ASSERT_NO_MSG( 0xf034defd, tlAllocator->getBlockStreamAllocator() == m_allocator );
    if ( !m_blocks.isEmpty() )
    {
        if ( isEmpty() && m_blocks.getSize()==1 && m_partiallyFreed == false )
        {
            return;
        }
        clear(tlAllocator);
    }
    blockAlloc(tlAllocator);

    HK_ON_DEBUG(checkConsistency());
}

void hkBlockStream::clear( _In_ hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator )
{
    HK_ASSERT_NO_MSG( 0xf0345dff, !m_isLocked );
    int size = m_blocks.getSize();

    if ( m_partiallyFreed )
    {
        // compress used blocks
        hkBlockStream::Block** src = m_blocks.begin();
        hkBlockStream::Block** dst = src;
        for (int i =0; i < size; i++)
        {
            if ( *src )
            {
                *(dst++) = *(src);
            }
            src++;
        }
        size = int(dst - m_blocks.begin());
        m_partiallyFreed = false;
    }

    if ( size )
    {
        tlAllocator->blockFreeBatch( m_blocks.begin(),  size );
    }
    m_blocks.clear();   // needs to be done always since m_partiallyFreed
    m_numTotalElements = 0;

    HK_ON_DEBUG(checkConsistency());
}

void hkBlockStream::append( _In_ hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator, _In_ hkBlockStream* HK_RESTRICT inStream )
{
    HK_ASSERT_NO_MSG( 0xf034defd, tlAllocator->getBlockStreamAllocator() == m_allocator );

    HK_ASSERT_NO_MSG( 0xf0345dfe, !m_isLocked && !inStream->m_isLocked && !m_partiallyFreed && !inStream->m_partiallyFreed);

    HK_ASSERT_NO_MSG( 0x613fe05d, this != inStream );

    if( inStream->isEmpty() )
    {
        if ( m_blocks.getSize() == 0)   // just cleared so we need to create a dummy block
        {
            blockAlloc(tlAllocator);
        }
        inStream->clear( tlAllocator );
        HK_ON_DEBUG(checkConsistency());
        return;
    }

    if ( m_blocks.getSize() == 0)   // just cleared so we can append
    {
    }
    else if( isEmpty() )
    {
        tlAllocator->blockFree( m_blocks[0] );
        m_blocks.clear();
    }
    else
    {
        hkBlockStream::Block* prevLast = lastRw();
        HK_ASSERT_NO_MSG(0xf0345456, !prevLast->m_next && prevLast->getNumElements());
        prevLast->m_next = inStream->beginRw();
    }
    int oldSize = m_blocks.getSize();
    m_blocks.insertAt( oldSize, inStream->m_blocks.begin(), inStream->m_blocks.getSize() );
    inStream->m_blocks.clear();

    if (oldSize)
    {
        for (int i = oldSize; i< m_blocks.getSize(); i++)
        {
            m_blocks[i]->m_blockIndexInStream = i;
            HK_ON_DEBUG( m_blocks[i]->m_blockStream = this);
        }
    }
    else
    {
#if defined(HK_DEBUG)
        for (int i = oldSize; i< m_blocks.getSize(); i++)   {   m_blocks[i]->m_blockStream = this;      }
#endif
    }
    m_numTotalElements += inStream->m_numTotalElements;
    inStream->m_numTotalElements = 0;

    HK_ON_DEBUG(checkConsistency());

    return;
}

void hkBlockStream::fixupConsumedBlocks( _In_ hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator )
{
    if ( !m_partiallyFreed )
    {
        HK_ASSERT_NO_MSG( 0xf0dfe565, m_blocks.indexOf(HK_NULL) == -1 );
        return;
    }
    int size = m_blocks.getSize();
    // compress used blocks
    hkBlockStream::Block** src = m_blocks.begin();
    hkBlockStream::Block** dst = src;
    int d = 0;
    int numTotalElements = 0;
    hkBlockStream::Block* lastBlock = HK_NULL;
    for (int i =0; i < size; i++)
    {
        if ( *src )
        {
            hkBlockStream::Block* block = *(src);
            if ( lastBlock )
            {
                lastBlock->m_next = block;
            }
            lastBlock = block;
            *dst = block;
            int numElements = block->getNumElements();
            numTotalElements += numElements;
            block->m_blockIndexInStream = d;
            dst++;
            d++;
        }
        src++;
    }
    if (lastBlock)
    {
        lastBlock->m_next = HK_NULL;
    }
    size = int(dst - m_blocks.begin());
    m_numTotalElements = numTotalElements;
    m_blocks.setSizeUnchecked(size);
    if (!size)
    {
        blockAlloc(tlAllocator);
    }
    checkConsistency();
    m_partiallyFreed = false;
}

int hkBlockStream::calcTotalBytesUsed() const
{
    int sum = 0;
    for (int i =0; i < m_blocks.getSize(); i++ )
    {
        sum += m_blocks[i]->getNumBytesUsed();
    }
    return sum;
}

struct hkBlockStream_SerializeHeader
{
    enum { MAGIC = 0xf054f465 };
    int     m_magic;
    int     m_numTotalElements;
    int     m_numblocks;
};

struct hkBlockStream_BlockHeader
{
    int     m_magic;
    hkUint32 m_numElementsAndBytesUsed;
};

void hkBlockStream::serialize( hkStreamWriter& writer ) const
{
    hkBlockStream_SerializeHeader h;
    h.m_magic = hkBlockStream_SerializeHeader::MAGIC;
    h.m_numTotalElements = m_numTotalElements;
    h.m_numblocks = m_blocks.getSize();

    writer.write( &h, sizeof(h) );

    for (int i = 0; i < m_blocks.getSize(); i++ )
    {
        const Block* b = m_blocks[i];

        hkBlockStream_BlockHeader bh;
        bh.m_magic = hkBlockStream_SerializeHeader::MAGIC + i;
        bh.m_numElementsAndBytesUsed = b->m_numElementsAndBytesUsed;
        writer.write( &bh, sizeof(bh) );
        writer.write( b->begin(), b->getNumBytesUsed() );
    };
}

void hkBlockStream::deserialize( hkStreamReader& reader, _In_ hkThreadLocalBlockStreamAllocator* tlAllocator )
{
    hkBlockStream_SerializeHeader h;
    reader.read( &h, sizeof( h ) );
    HK_ASSERT_NO_MSG( 0xf0456567, h.m_magic == (int)hkBlockStream_SerializeHeader::MAGIC );
    m_numTotalElements = h.m_numTotalElements;
    int numblocks   = h.m_numblocks;
    m_blocks.reserve( numblocks );

    for (int i = 0; i < numblocks; i++ )
    {
        Block* block;
        if ( i >= m_blocks.getSize() )
        {
            block = blockAlloc( tlAllocator );
        }
        else
        {
            block = m_blocks[i];
        }
        block->m_next = 0;
        if ( i> 0)
        {
            m_blocks[i-1]->m_next = block;
        }

        hkBlockStream_BlockHeader bh;
        reader.read( &bh, sizeof(bh) );
        HK_ASSERT_NO_MSG( 0xf0456567, bh.m_magic == int( hkBlockStream_SerializeHeader::MAGIC + i ) );
        block->m_numElementsAndBytesUsed = bh.m_numElementsAndBytesUsed;
        reader.read( block->begin(), block->getNumBytesUsed() );
    };
}

hkBlockStream::Block* hkBlockStream::blockAlloc( _In_ hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator )
{
    // Get a block from the allocator
    HK_ASSERT_NO_MSG( 0xf034defd, tlAllocator->getBlockStreamAllocator() == m_allocator );
    hkBlockStream::Block* block = tlAllocator->blockAlloc();

    // zero if needed
    block->setHeaderToZero();
    hkCheckDeterminismUtil::randomizeMemory( block->begin(), block->BLOCK_DATA_SIZE );

    block->m_blockIndexInStream = m_blocks.getSize();
    m_blocks.pushBack( block );

    HK_ON_DEBUG( block->m_allocator = m_allocator );
    HK_ON_DEBUG( block->m_blockStream = this );

    return block;
}

hkBlockStream::Block* hkBlockStream::popBack( _In_ hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator )
{
    int numBlocks = m_blocks.getSize();
    hkBlockStream::Block* lastBlock;
    hkBlockStream::Block* secondLastBlock;

    HK_ASSERT_NO_MSG( 0xf034defd, tlAllocator->getBlockStreamAllocator() == m_allocator );
    lastBlock = m_blocks[numBlocks - 1];
    HK_ASSERT_NO_MSG( 0xf034deff, lastBlock->m_blockIndexInStream == m_blocks.getSize() - 1 );
    secondLastBlock = m_blocks[numBlocks - 2];
    HK_ASSERT_NO_MSG( 0xf034fddf, secondLastBlock->m_next == lastBlock );
    tlAllocator->blockFree( lastBlock );
    secondLastBlock->m_next = HK_NULL;

    m_blocks.popBack();
    HK_ON_DEBUG(checkConsistency());
    return secondLastBlock;
}

void hkBlockStream::freeBlock( _In_ hkThreadLocalBlockStreamAllocator* HK_RESTRICT tlAllocator, _In_ hkBlockStream::Block* HK_RESTRICT block )
{
    const int index = block->m_blockIndexInStream;
    hkBlockStream::Block** blocks = m_blocks.begin();
    HK_ASSERT_NO_MSG( 0xf03df1d8, block->m_blockStream == this && block->m_blockIndexInStream == index && m_blocks[index] == block );
    blocks[index] = HK_NULL;
    m_partiallyFreed = true;    // on spu this is set by the consumer::setToRange()
    tlAllocator->blockFree( block );
}

void hkBlockStream::checkConsistency() const
{
#if defined(HK_DEBUG)
    int totalNumElems = 0;
    for (int bi = 0; bi < m_blocks.getSize(); bi++)
    {
        const hkBlockStream::Block* block = m_blocks[bi];
        if ( bi < m_blocks.getSize() - 1 )
        {
            // Not last block
            HK_ASSERT_NO_MSG( 0xf0ccfe34, block->m_next == m_blocks[bi+1] );
        }
        else
        {
            HK_ASSERT_NO_MSG( 0xf0ccfe34, block->m_next == HK_NULL );
        }
        totalNumElems += block->getNumElements();
        HK_ASSERT_NO_MSG( 0xf0ccfe34, block->m_blockStream == this && block->m_blockIndexInStream == bi && block->m_allocator == m_allocator );
    }
    HK_ASSERT_NO_MSG( 0xf0ccfe37, totalNumElems == m_numTotalElements );
#endif
}

void hkBlockStream::checkConsistencyOfRange( const hkBlockStream::Range& range ) const
{
#if defined(HK_DEBUG)
    if (range.isEmpty())
    {
        return;
    }
    // check if the range points to a block owned by this stream
    HK_ASSERT_NO_MSG( 0xf034df34, m_blocks[range.m_startBlock->m_blockIndexInStream] == range.m_startBlock );

    HK_ASSERT_NO_MSG( 0xf034df34, range.m_startBlockNumElements <= range.m_startBlock->getNumElements() );
#endif
}

void hkBlockStream::checkConsistencyWithGrid(
    _In_reads_(numRanges) const hkBlockStream::Range* rangesIn, int numRanges, int rangeStriding, bool allowForUnusedData ) const
{
#if defined(HK_DEBUG)
    hkLocalBuffer<hkBlockStream::Range> ranges( numRanges );
    {
        for (int i =0; i < numRanges; i++)
        {
            ranges[i] = *rangesIn;
            rangesIn = hkAddByteOffsetConst( rangesIn, rangeStriding );
        }
        hkSort( ranges.begin(), numRanges, Range::compareRanges );
    }

    checkConsistencyWithSortedRanges(ranges.begin(), numRanges, rangeStriding, allowForUnusedData);
#endif
}

void hkBlockStream::checkConsistencyWithSortedRanges(
    _In_reads_(numRanges) const hkBlockStream::Range* sortedRanges, int numRanges, int rangeStriding, bool allowForUnusedData ) const
{
#if defined(HK_DEBUG)
    checkConsistency();

    const hkBlockStream::Block* currentBlock = HK_NULL;
    hkUint32 numElementsInCurrentBlock = 0;
    int curentBlockLastByte = -1;
    int totalNumElemsInRanges = 0;

    for (int i =0; i < numRanges; i++)
    {
        const hkBlockStream::Range& range = sortedRanges[i];
        if (range.isEmpty())
        {
            continue;
        }
        totalNumElemsInRanges += range.getNumElements();

        // check if the range points to a block owned by this stream
        HK_ASSERT_NO_MSG( 0xf034df34, m_blocks[range.m_startBlock->m_blockIndexInStream] == range.m_startBlock );

        if ( range.m_startBlock != currentBlock )
        {
            if ( currentBlock )
            {
                HK_ASSERT_NO_MSG(0xf034df34, numElementsInCurrentBlock <= currentBlock->getNumElements());
            }
            currentBlock = range.m_startBlock;
            curentBlockLastByte = range.m_startByteOffset;
            numElementsInCurrentBlock = range.m_startBlockNumElements;
        }
        else
        {
            HK_ASSERT_NO_MSG( 0xf034df34, range.m_startByteOffset > curentBlockLastByte );
        }
    }
    HK_ASSERT_NO_MSG(0xf034df34, totalNumElemsInRanges <= m_numTotalElements );
    if ( !allowForUnusedData)
    {
        HK_ASSERT_NO_MSG(0xf034df34, totalNumElemsInRanges == m_numTotalElements );
    }
#endif
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
