// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Container/BlockStream/Allocator/hkThreadLocalBlockStreamAllocator.h>

typedef hkBlockStreamAllocator::Block Block;

hkThreadLocalBlockStreamAllocator::hkThreadLocalBlockStreamAllocator( hkBlockStreamAllocator* bsAllocator )
    : m_blockStreamAllocator( bsAllocator )
    , m_numFreeBlocks(0)
    , m_numBlocksToCache( hkMath::min2(bsAllocator->getNumThreadLocalBlocks(), hkBlockStreamAllocator::MAX_FREE_LIST_SIZE) )
{
}

hkThreadLocalBlockStreamAllocator::~hkThreadLocalBlockStreamAllocator()
{
    clear();
}

Block* hkThreadLocalBlockStreamAllocator::blockAlloc()
{
#ifdef HK_DEBUG
    checkConsistency();
#endif

    hkBlockStreamAllocator::Block* block;
    if( m_numFreeBlocks > 0 )
    {
        m_numFreeBlocks = m_numFreeBlocks - 1;
        block = m_freeBlocks[ m_numFreeBlocks ];
        HK_ASSERT_NO_MSG( 0xf05678ad, block->getNumElements() == 0 );
    }
    else
    {
        m_blockStreamAllocator->blockAllocBatch( &m_freeBlocks[0], BATCH_ALLOC_SIZE );
#ifdef HK_DEBUG
        for( int i=0; i<BATCH_ALLOC_SIZE; i++ )
        {
            m_freeBlocks[i]->setNumElements(0);
        }
#endif
        m_numFreeBlocks = BATCH_ALLOC_SIZE-1;
        block = m_freeBlocks[BATCH_ALLOC_SIZE-1];
    }
    return block;
}

void hkThreadLocalBlockStreamAllocator::blockFree( _Inout_ hkBlockStreamAllocator::Block* block )
{
#ifdef HK_DEBUG
    HK_ASSERT_NO_MSG( 0xf04df1d4, block->m_allocator == (hkBlockStreamAllocator*)m_blockStreamAllocator);
    block->setNumElements(0);   // for check consistency
    block->m_blockStream = HK_NULL;
    checkConsistency();
    HK_ASSERT_NO_MSG( 0xf0456567, block != hkBlockStreamAllocator::s_safeGuard );
#endif

    HK_ASSERT_NO_MSG(0x3e515b17, m_numFreeBlocks < hkBlockStreamAllocator::MAX_FREE_LIST_SIZE);
    m_freeBlocks[ m_numFreeBlocks ] = block;
    m_numFreeBlocks = m_numFreeBlocks+1;
    if ( m_numFreeBlocks < hkBlockStreamAllocator::MAX_FREE_LIST_SIZE )
    {
        return;
    }

    const int newEndIndex = m_numFreeBlocks - BATCH_ALLOC_SIZE;
    m_blockStreamAllocator->blockFreeBatch( &m_freeBlocks[newEndIndex], BATCH_ALLOC_SIZE );
    m_numFreeBlocks = newEndIndex;
}

void hkThreadLocalBlockStreamAllocator::clear()
{
    if ( m_numFreeBlocks )
    {
        m_blockStreamAllocator->blockFreeBatch( &m_freeBlocks[0], m_numFreeBlocks );
        m_numFreeBlocks = 0;
    }
}

void hkThreadLocalBlockStreamAllocator::checkConsistency() const
{
    const hkThreadLocalBlockStreamAllocator* lc = this;
    for(int i=0; i< lc->m_numFreeBlocks; i++)
    {
        HK_ASSERT_NO_MSG(0xf0768790, lc->m_freeBlocks[i]->getNumElements() == 0);
        for(int j=i+1; j<lc->m_numFreeBlocks; j++)
        {
            HK_ASSERT_NO_MSG(0xf0768791, lc->m_freeBlocks[i] != lc->m_freeBlocks[j]);
        }
    }
}

void hkThreadLocalBlockStreamAllocator::blockFreeBatch( _Inout_updates_(nblocks) Block** blocks, int nblocks )
{
#ifdef HK_DEBUG
    for( int i=0; i<nblocks; i++ )
    {
        blocks[i]->setNumElements(0);
    }
#endif

    if ( nblocks >= BATCH_ALLOC_SIZE )
    {
        m_blockStreamAllocator->blockFreeBatch( blocks, nblocks );
        return;
    }

    if ( nblocks + m_numFreeBlocks >= hkBlockStreamAllocator::MAX_FREE_LIST_SIZE)
    {
        // batch free existing elements
        int numBlocksToFree = hkMath::_min2<int>( BATCH_ALLOC_SIZE, m_numFreeBlocks);
        const int newEndIndex = m_numFreeBlocks - numBlocksToFree;
        m_blockStreamAllocator->blockFreeBatch( &m_freeBlocks[newEndIndex], numBlocksToFree );
        m_numFreeBlocks = newEndIndex;
    }

    // append elements to free list
    for (int i = 0; i < nblocks; i++)
    {
        HK_ASSERT_NO_MSG( 0xf04df1d4, blocks[i]->m_allocator == (hkBlockStreamAllocator*)m_blockStreamAllocator);
        m_freeBlocks[ i + m_numFreeBlocks ] = blocks[i];
    }
    m_numFreeBlocks = m_numFreeBlocks + nblocks;

    //  SHOW_FREELIST;
    if(m_numFreeBlocks > m_numBlocksToCache)
    {
        m_blockStreamAllocator->blockFreeBatch( &m_freeBlocks[m_numBlocksToCache], m_numFreeBlocks - m_numBlocksToCache );
        HK_ON_DEBUG( for(int i = m_numBlocksToCache; i < m_numFreeBlocks; i++){ m_freeBlocks[i] = HK_NULL; } );
        m_numFreeBlocks = m_numBlocksToCache;
    }

    HK_ON_DEBUG( checkConsistency() );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
