// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Container/BlockStream/Allocator/hkBlockStreamAllocator.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

/// A simple implementation of block stream allocator using a fixed size buffer asserting when out of memory.
/// This is the only implementation working on PlayStation(R)3.
class HK_EXPORT_COMMON hkFixedBlockStreamAllocator : public hkBlockStreamAllocator
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_COLLIDE);
        HK_RECORD_ATTR( hk::MemoryTracker(handler=&hkFixedBlockStreamAllocator::trackerHandler) );

        hkFixedBlockStreamAllocator( int allocSize = 0 );
        ~hkFixedBlockStreamAllocator();

        virtual void blockAllocBatch(_Out_writes_(nblocks) Block** blocksOut, int nblocks);
        virtual void blockFreeBatch(_In_reads_(nblocks) Block** blocks, int nblocks);

        /// get the current bytes used. Note that data in the thread local allocator will
        /// count as bytes used (as they are not available to other threads in a worst case
        /// scenario).
        virtual int getBytesUsed() const;

        /// the peak usage
        virtual int getMaxBytesUsed() const;

        /// Returns the total number of bytes available
        int getCapacity() const;

        void init( int byteSize );

        void init( _In_bytecount_(byteSize) void* buffer, int byteSize );

        /// free all memory, only necessary if this class allocated the memory
        void clear();

        /// Asserts if any data is allocated.
        void freeAllRemainingAllocations();

        void getMemoryStatistics( hkMemoryAllocator::MemoryStatistics& statsOut ) const;

    public:

        /// Internal use.
        static void trackerHandler(const hkReflect::Var& var, class hkMemoryTrackerSnapshot& snapshot);

    private:

        void rebuildFreelist();

    public:

        /// The critical section, must go before m_storage to avoid DMAing it back and forth SPU
        mutable hkCriticalSection m_criticalSection;

        // Aligned for SPU access.
        HK_ALIGN16(Block* m_blocks) HK_ATTR( hk::MemoryTracker(opaque=true) );

        hkArray< Block* > m_freeList HK_ATTR( hk::Type(hkArray<hkUlong>) );

        int     m_capacity;                     ///< Total number of blocks available
        int     m_minFreeListSize;              ///< For tracking peak memory usage
        hkBool  m_dontDeallocate;
        hkBool  m_enableAllocationTimerInfo;    ///< Add extra used memory timer infos into the timer stream, debug only
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
