// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Container/BlockStream/Allocator/hkBlockStreamAllocator.h>
#include <Common/Base/Container/BlockStream/Allocator/Fixed/hkFixedBlockStreamAllocator.h>
#include <Common/Base/Memory/Tracker/hkMemoryTrackerSnapshot.h>

typedef hkBlockStreamAllocator::Block Block;

//
// Simple BSA
//

hkFixedBlockStreamAllocator::hkFixedBlockStreamAllocator(int allocSize): m_criticalSection (4000)
{
    m_capacity = 0;
    m_dontDeallocate = false;
    m_enableAllocationTimerInfo = false;
    m_blocks = 0;
    if ( allocSize )
    {
        init( allocSize );
    }
}

void hkFixedBlockStreamAllocator::init( int byteSize )
{
    HK_ASSERT( 0xf0345466, m_blocks==HK_NULL, "You cannot initialize a stream allocator twice" );
    m_capacity = byteSize/sizeof( Block );
    m_dontDeallocate = false;
    m_blocks = hkAlignedAllocate<Block>( Block::BLOCK_ALIGNMENT, m_capacity, HK_MEMORY_CLASS_COLLIDE );
    HK_ASSERT_NO_MSG(0x6acbf233, (hkUlong(m_blocks) & (Block::BLOCK_ALIGNMENT-1) ) == 0);

    rebuildFreelist();
}

void hkFixedBlockStreamAllocator::init(_In_bytecount_(byteSize) void* buffer, int byteSize)
{
    HK_ASSERT( 0xf0345465, m_blocks==HK_NULL, "You cannot initialize a stream allocator twice" );
    char* alignedBuffer = (char*)(HK_NEXT_MULTIPLE_OF(Block::BLOCK_ALIGNMENT, hkUlong(buffer)));
    int bytesUsedForAlignment = int(alignedBuffer - (char*)buffer);
    byteSize -= bytesUsedForAlignment;

    m_capacity = byteSize/sizeof( Block );
    m_blocks = (Block*)alignedBuffer;

    rebuildFreelist();

    m_dontDeallocate = true;
}

void hkFixedBlockStreamAllocator::rebuildFreelist()
{
    m_freeList.setSize( m_capacity );

    int d = 0;
    for(int i = m_capacity-1; i>=0; d++, i--)
    {
        Block* block = &m_blocks[ i ];
        //HK_ON_DEBUG( block->setHeaderToZero() );
        HK_ON_DEBUG(block->m_allocator = this);
        HK_ON_DEBUG(block->m_blockStream = HK_NULL);
        m_freeList[d] = block;
    }
    m_minFreeListSize = m_freeList.getSize();
}

hkFixedBlockStreamAllocator::~hkFixedBlockStreamAllocator()
{
    clear();
    HK_ASSERT( 0xf0345466, m_blocks==HK_NULL, "You must call clear before the destructor" );
}

void hkFixedBlockStreamAllocator::freeAllRemainingAllocations()
{
    if ( m_freeList.getSize() != m_capacity )
    {
        rebuildFreelist();
    }
    HK_ASSERT(0xed4db99, m_freeList.getSize() == m_capacity, "Delete all the thread local cache?" );
}

void hkFixedBlockStreamAllocator::clear()
{
    HK_ASSERT(0xed4db99, m_freeList.getSize() == m_capacity, "Delete all the thread local cache?" );
    if( !m_dontDeallocate )
    {
        hkAlignedDeallocate( m_blocks );
    }
    m_dontDeallocate = true;
    m_freeList.clear();
    m_capacity = 0;
    m_blocks = HK_NULL;
}

int hkFixedBlockStreamAllocator::getCapacity() const
{
    return m_capacity * sizeof(Block);
}

int hkFixedBlockStreamAllocator::getBytesUsed() const
{
    int numAllocated = m_capacity - m_freeList.getSize();
    return numAllocated * sizeof (Block);
}

int hkFixedBlockStreamAllocator::getMaxBytesUsed() const
{
    int numAllocated = m_capacity - m_minFreeListSize;
    return numAllocated * sizeof (Block);
}

void hkFixedBlockStreamAllocator::getMemoryStatistics( hkMemoryAllocator::MemoryStatistics& statsOut ) const
{
    statsOut.m_allocated        = getCapacity();
    statsOut.m_inUse            = getBytesUsed();
    statsOut.m_peakInUse        = getMaxBytesUsed();
    statsOut.m_available        = statsOut.m_allocated - getBytesUsed();
    statsOut.m_totalAvailable   = statsOut.m_available;
    statsOut.m_largestBlock     = sizeof(Block);
}

HK_COMPILE_TIME_ASSERT((HK_OFFSET_OF( hkFixedBlockStreamAllocator, m_blocks ) > HK_OFFSET_OF( hkFixedBlockStreamAllocator, m_criticalSection )));

void hkFixedBlockStreamAllocator::blockAllocBatch( _Out_writes_(nblocks) Block** blocksOut, int nblocks )
{
    m_criticalSection.enter();

#if 0 && defined(HK_DEBUG)
    for (int i =0; i < m_freeList.getSize(); i++ )
    {
        Block* block = m_freeList[i];
        HK_ASSERT_NO_MSG( 0xf045fcdf, block->m_blockStream == 0 && block->m_allocator == this );
    }
#endif

    //HK_ON_DEBUG( HK_MONITOR_ADD_VALUE("BlockStreamManagerCsAlloc", 1.f, HK_MONITOR_TYPE_INT) );

    for(int i=0; i<nblocks; i++)
    {
        if ( m_freeList.isEmpty() )
        {
            HK_ERROR( 0xf03df676, "Out of blockstream memory" );
            // Set the remaining pointers to null.
            hkString::memSet(blocksOut, 0x0, (nblocks - i) * sizeof(Block*) );
            break;
        }
        *blocksOut = m_freeList.back();
        //Log_Info( "Allocate {}  This: {}", (void*)*blocksOut, (void*)this );
        m_freeList.popBack();
        blocksOut++;
    }
    if ( m_freeList.getSize() < m_minFreeListSize )
    {
        m_minFreeListSize = m_freeList.getSize();
    }
    if ( m_enableAllocationTimerInfo)
    {
        HK_MONITOR_ADD_VALUE( "MemSizeUsed", float(getBytesUsed()), HK_MONITOR_TYPE_INT );
    }

    m_criticalSection.leave();
}

void hkFixedBlockStreamAllocator::blockFreeBatch( _In_reads_(nblocks) Block** blocks, int nblocks )
{
    if( !nblocks )
    {
        return;
    }
    m_criticalSection.enter();
    //HK_ON_DEBUG( HK_MONITOR_ADD_VALUE("BlockStreamManagerCsFree", 1.f, HK_MONITOR_TYPE_INT) );

    for(int i=0; i<nblocks; blocks++, i++)
    {
        Block* block = *blocks;
        if( block )
        {
            HK_ASSERT_NO_MSG( 0xf04df1d4, block->m_allocator == this );
            //HK_ON_DEBUG( block->m_blockStream = 0 );
            //Log_Info( "Free {}" (void*)block );
            m_freeList.pushBackUnchecked( block );
        }
    }
    HK_ASSERT_NO_MSG( 0xf0457678, m_freeList.getSize() <= m_capacity );
    if ( m_enableAllocationTimerInfo)
    {
        HK_MONITOR_ADD_VALUE( "MemSizeUsed", float(getBytesUsed()), HK_MONITOR_TYPE_INT );
    }
    m_criticalSection.leave();
}

void hkFixedBlockStreamAllocator::trackerHandler(const hkReflect::Var& var, hkMemoryTrackerSnapshot& snapshot)
{
#ifdef HK_MEMORY_TRACKER_ENABLE
    const hkFixedBlockStreamAllocator* bsa = static_cast<const hkFixedBlockStreamAllocator*>(var.getAddress());
    if (bsa->m_blocks && !bsa->m_dontDeallocate)
    {
        int used = bsa->getBytesUsed();
        int free = bsa->getCapacity() - used;
        if (used)
        {
            snapshot.addLinkedBlock(
                hkMemoryTrackerSnapshot::Block("buffer_hkFixedBlockStreamAllocator_used", bsa->m_blocks, used), true);
        }
        if (free)
        {
            snapshot.addLinkedBlock(
                hkMemoryTrackerSnapshot::Block("buffer_hkFixedBlockStreamAllocator_free", hkAddByteOffset(bsa->m_blocks, used), free), true);
        }
        snapshot.addLinkedAlignedOverhead(bsa->m_blocks);
    }
#endif
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
