// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

// this: #include <Common/Base/Container/BlockStream/Allocator/Dynamic/hkDynamicBlockStreamAllocator.h>

#include <Common/Base/Container/BlockStream/Allocator/hkBlockStreamAllocator.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

/// A block stream allocator which dynamically allocates more blocks when its initial capacity is exceeded.
/// It "shrinks" (i.e. returns the blocks to the base allocator) only when it is destroyed or when you call
/// releaseUnusedBlocks().
class HK_EXPORT_COMMON hkDynamicBlockStreamAllocator : public hkBlockStreamAllocator
{
    public:

        enum FreePolicy
        {
            KEEP_BLOCKS,

            FREE_UNUSED_BLOCKS
        };

    public:

        HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE );
        HK_RECORD_ATTR(hk::MemoryTracker(handler=&hkDynamicBlockStreamAllocator::trackerHandler));

        /// Create an allocator with the given initial capacity.
        hkDynamicBlockStreamAllocator( int initialSize, FreePolicy freePolicy );

        /// Destructor. Frees all the blocks.
        ~hkDynamicBlockStreamAllocator();

        /// Allocate the requested number of blocks.
        virtual void blockAllocBatch( _Out_writes_(nblocks) Block** blocksOut, int nblocks );

        /// Free the given blocks.
        virtual void blockFreeBatch( _In_reads_(nblocks) Block** blocks, int nblocks );

        /// Return the number of bytes given out by this allocator.
        /// Note: The blocks in the thread-local allocator's cache will count as used from this allocator's point of view.
        virtual int getBytesUsed() const;

        /// Get the peak byte usage.
        virtual int getMaxBytesUsed() const;

        /// Return the current total bytes used and available through this allocator without expanding it.
        int getCapacity() const;

        /// Free all memory. The allocator will be completely empty. Asserts if any data has not been deallocated.
        virtual void freeAllRemainingAllocations();

        /// Return the memory statistics.
        virtual void getMemoryStatistics( hkMemoryAllocator::MemoryStatistics& statsOut ) const;

        /// Attempt to free reserved blocks if they are not used.
        void releaseUnusedBlocks();

    public:

        /// Internal use.
        static void trackerHandler(const hkReflect::Var& var, class hkMemoryTrackerSnapshot& snapshot);

    protected:

        // Reserve more blocks to allocate at least the given size in bytes.
        void expand( int numBytes );

    protected:

        /// A critical section to protect allocations and deallocations.
        hkCriticalSection m_criticalSection;

        /// This array contains all blocks.
        hkArray<Block*> m_blocks HK_ATTR( hk::Type(hkArray<hkUlong>) );

        /// This array contains all blocks available for allocation.
        hkArray<Block*> m_freeList HK_ATTR( hk::Type(hkArray<hkUlong>) );

        FreePolicy m_freePolicy;

        /// The maximum number of bytes ever allocated through this allocator.
        int m_maxBytesUsed;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
