// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : COMMON
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

    /// A view on a a range of an existing buffer
    ///
    /// When copying an instance of an array view, only the internal pointers
    /// are copied, not the content they point to. An array view does not own
    /// its content, it simply provides a view to it. For this reason, the
    /// const-ness of an instance of this class does not determine the const-
    /// ness of the content.
    ///
    /// Example:
    /// \code
    ///   int data[3] = { 0, 1, 2 };
    ///   const hkArrayView<int> constView(&data[0], 3);
    ///   hkArrayView<const int> nonconstView = constView;
    ///   constView[0] = 42; // This is legal because constView does not own its content (`data` owns it);
    ///   nonconstView[1] = 1337; // This is illegal in any case.
    /// \endcode
    ///
    /// If you want to maintain a const-view on your data, you can convert from
    /// hkArrayView<T> to hkArrayView<const T> in several ways:
    ///   1) Be explicit and use the method `asConstView()`.
    ///   2) Cast to hkArrayView<const T> by using static_cast, a function-style cast, or a C-style cast.
    ///   3) A hkArrayView<T> can be implicitly converted to hkArrayView<const T>.
    ///
    /// Example:
    /// \code
    ///   int data[3] = { 0, 1, 2 };
    ///   hkArrayView<int> myDataView(&data[0], 3);
    ///   hkArrayView<const int> immutableView1 = myDataView.asConstView(); // 1)
    ///   hkArrayView<const int> immutableView2 = static_cast< hkArrayView<const int> >(myDataView); // 2)
    ///   auto immutableView3 = [=]() -> hkArrayView<const int> { return myDataView; }(); // 3)
    /// \endcode
    ///
    /// \note An array view should be treated as if it was a pointer:
    ///       - It can be copied around at low cost. The data pointed to is not affected by copying.
    ///       - It's constness does not affect the pointee's constness.
    ///       - It can be null (see isEmpty()).
    ///
    /// \note \c hkArrayView<void> and \c hkArrayView<const void> are allowed and have the following noteworthy properties:
    ///       - The index operator (operator[]) is undefined. But you can use \c ptr(int)
    ///       - Search algorithms (find, indexOf) are undefined.
    ///       - Any hkArrayView<T> is implicitly convertible to
    ///         hkArrayView<void> or hkArrayView<const void> (if T is const)
    ///       - In case of \c void, all sizes and indices are treated as byte
    ///         offsets. See \c ptr(), or \c slice() for example.
    ///
    /// \see hkArrayViewT
template<typename T>
class hkArrayView
{
    public:

        typedef typename hkTrait::VoidValue<T>::Type ValueType;

            // PointerType will be T* in all cases except for when
            // T == [const ]void, in which case a wrapper struct is used that
            // supports pointer arithmetics.
        typedef typename hkTrait::VoidValue<T>::PtrType PointerType;

        HK_DECLARE_CLASS(hkArrayView, New, Reflect);
        HK_REFLECT_AS_ARRAY(&hkReflect::Detail::StaticArrayImpl::s_instance, T); 
        HK_RECORD_ATTR(hk::Serialize(false));

            /// Create an empty view.
        HK_INLINE hkArrayView() : m_begin(HK_NULL), m_end(HK_NULL) {}
            /// Create a view from begin and end pointers.
        HK_INLINE hkArrayView(_In_reads_to_ptr_(end) T* _begin, _Maybevalid_ T* _end) : m_begin(_begin), m_end(_end) { HK_ASSERT_NO_MSG(0x78e562e1, m_begin<=m_end); }

            /// Create a view of num elements from begin.
        template<typename U>
        HK_INLINE hkArrayView(_In_reads_bytes_(num*HK_MAX2(sizeof(U), 1)) U* _begin, _In_range_(>= , 0) hkUlong num)
        {
            *this = hkArrayView<U>(_begin, static_cast<typename hkArrayView<U>::PointerType>(_begin) + num);
        }

            /// Construct an array view from another, if they're compatible (see hkTrait::ViewAsOther)
            /// and have the same size in memory.
        template<typename U>
        HK_INLINE hkArrayView(const hkArrayView<U>& rhs);


            /// Provide a const view into the data of this array view.
        HK_INLINE hkArrayView<const T> asConstView() const { return *this; }

            /// Return the number of elements in this view.
        HK_INLINE _In_range_(>=, 0) int getSize() const { return hkLosslessCast<int>(m_end - m_begin); }

            /// Access the i'th element.
            ///
            /// \note Is not available if T == [const ]void. Use \c ptr(int) instead.
        HK_ALWAYS_INLINE ValueType& operator[](_In_range_(>= , 0) int i) const { HK_ASSERT_NO_MSG(0x5981b422, m_begin + i < m_end); return m_begin[i]; }

            /// Get the data pointer
        HK_ALWAYS_INLINE _Ret_maybenull_ T* data() const { return m_begin; }

            /// Get a T* at the given index (with bounds checking).
            ///
            /// You should prefer \c ptr(3) over \code begin() + 3 \endcode as the former does bounds checking.
            ///
            /// Also useful for array views of \c void or \c const \c void as
            /// a replacement of the typical \c &arrayView[index] operation.
            /// In such a case the \a index argument is treated as a byte
            /// offset.
        HK_INLINE _Ret_notnull_ T* ptr(_In_range_(>= , 0) int index) const;

            /// Checks if the size is zero.
        HK_ALWAYS_INLINE hkBool isEmpty() const { return m_begin == m_end; }

            /// Set to the empty array.
        HK_INLINE void clear() { m_begin = HK_NULL; m_end = HK_NULL; }

            /// Returns a reference to the first element
        HK_INLINE ValueType& front() const { HK_ASSERT_NO_MSG(0x6e1082d0, !isEmpty()); return *m_begin; }

            /// Returns a reference to the last element
        HK_INLINE ValueType& back() const { HK_ASSERT_NO_MSG(0x2b2d40f4, !isEmpty()); return *(m_end - 1); }

            // Iterator support

        template<typename P>
        struct _reverse_iterator
        {
            _reverse_iterator() : m_p(nullptr) {}
            _reverse_iterator(P p) : m_p(p) {}
            T& operator*() { return m_p[-1]; }
            T* operator->() { return m_p - 1; }
            _reverse_iterator operator++() { --m_p; return *this; }
            _reverse_iterator operator--() { ++m_p; return *this; }
            _reverse_iterator& operator+=(int i) { m_p-=i; return *this; }
            _reverse_iterator& operator-=(int i) { m_p+=i; return *this; }
            bool operator==(const _reverse_iterator& it) const { return m_p == it.m_p; }
            bool operator!=(const _reverse_iterator& it) const { return m_p != it.m_p; }
        private:
            P m_p;
        };

        typedef T* iterator;
        typedef _reverse_iterator<T*> reverse_iterator;

            // For those who wish to have constant access.
            // Note: The type `iterator` implicitly converts to const_iterator.
        typedef const T* const_iterator;
        typedef _reverse_iterator<const T*> const_reverse_iterator;

        HK_INLINE iterator begin() const { return m_begin; }
        HK_INLINE iterator end()   const { return m_end; }
        HK_INLINE reverse_iterator rbegin() const { return m_end; }
        HK_INLINE reverse_iterator rend()   const { return m_begin; }

            // Search

            /// Returns an iterator to the first element that compares equal to 'u', end() otherwise.
            /// \note Is not available if T == [const ]void.
        template<typename U>
        iterator find(const U& u) const;

            /// Returns an iterator to the last element that compares equal to 'u', end() otherwise.
            /// \note Is not available if T == [const ]void.
        template<typename U>
        iterator findLast(const U& u) const;

            /// Returns the index of the first occurrence of t, or -1 if not found.
            /// \note Is not available if T == [const ]void.
        template<typename U>
        int indexOf(const U& needle) const;

            /// Returns index of the last occurrence of t, or -1 if not found.
            /// \note Is not available if T == [const ]void.
        template<typename U>
        int lastIndexOf(const U& needle) const;

            /// Return a read-only view with n elements removed from the start.
            /// \pre \a n must be within bounds.
            ///
            /// \note In the case of T == [const ]void, \a n is treated as a
            /// byte offset.
        HK_INLINE hkArrayView<T> ltrim(int n) const;

            /// Return a view with n elements removed from the end.
            /// \pre \a n must be within bounds.
            ///
            /// \note In the case of T == [const ]void, \a n is treated as a
            /// byte offset.
        HK_INLINE hkArrayView<T> rtrim(int n) const;

            /// This slice is specified by a start index and count. Not start and end indices.
            /// \pre The slice described by \a idx and \a num must be within bounds.
            ///
            /// \note In the case of T == [const ]void, \a idx and \a num is
            /// treated as a byte offset.
        HK_INLINE hkArrayView<T> slice(int idx, int num) const;

    protected:

        template<typename U> friend class hkArrayView;

        PointerType m_begin;
        PointerType m_end;
};

namespace hkArrayViewT
{
    template<typename T>
    static HK_ALWAYS_INLINE hkArrayView<T> make(_In_reads_(n) T* base, _In_range_(>= , 0) hkUlong n) { return hkArrayView<T>(base, n); }

    template<typename T>
    static HK_ALWAYS_INLINE hkArrayView<T> make(_In_reads_to_ptr_(end) T* start, _Maybevalid_ T* end) { return hkArrayView<T>(start, end); }

        /// Create a view from a literal array. (T[N])
    template<typename T, unsigned N>
    static HK_ALWAYS_INLINE hkArrayView<T> make(T(&t)[N]) { return hkArrayView<T>(&t[0], N); }

    template<typename T>
    static HK_ALWAYS_INLINE hkArrayView<T> fromSingleObject(T& object) { return hkArrayView<T>(&object, (&object) + 1); }
}


#include <Common/Base/Container/ArrayView/hkArrayView.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
