// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Container/Array/hkVariantArray.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Memory/Tracker/hkMemoryTrackerSnapshot.h>

hkVariantArray::hkVariantArray(const hkVariantArray& rhs)
    : m_ptr(HK_NULL)
    , m_size(0)
    , m_capacityAndFlags(0)
    , m_type(rhs.m_type)
{
    operator=(rhs);
}

//
//  Sets the type
void hkVariantArray::setType(_In_opt_ const hkReflect::Type* type)
{
    if ( type )
    {
        if ( type->isField() )
        {
            type = type->getParent();
        }

        m_type.set(type, false);
    }
    else
    {
        m_type = HK_NULL;
    }
}

void hkVariantArray::allocate( int size, _In_opt_ const hkReflect::Type* type )
{
    HK_ASSERT_NO_MSG(0x3f89f7f7, m_capacityAndFlags == 0);

    setType(type);
    HK_ASSERT_NO_MSG(0x51c042dc, m_type.get());

    int sizeElem = m_type->getSizeOf();
    int bytes;
    if (hkCheckMulSizes(sizeElem, size, &bytes).isFailure())
    {
        HK_FATAL_ERROR(0x0, "Array size overflow {}*{}", sizeElem, size);
    }
    m_ptr = bytes > 0 ? hkMemHeapBlockAlloc<void>(bytes) : HK_NULL;
    m_capacityAndFlags = bytes / sizeElem; // note bytes may be less than req..
    m_size = m_capacityAndFlags;

    // construct all the elements
    if (m_size)
    {
        HK_ON_DEBUG(hkResult res = )hkReflect::TypeDetail::defaultConstruct(m_ptr, m_type, m_size);
        HK_ASSERT_NO_MSG(0x1ecc66f6, res.isSuccess());
    }
}

void hkVariantArray::clear()
{
    if(m_size)
    {
        hkReflect::TypeDetail::destruct(m_ptr, m_type, m_size);
    }
    m_size = 0;
}

void hkVariantArray::clearAndDeallocate()
{
    if (m_ptr)
    {
        HK_ASSERT_NO_MSG(0x21eb2cbc, m_type.get());

        // destruct all the elements
        if (m_size)
        {
            hkReflect::TypeDetail::destruct(m_ptr, m_type, m_size);
        }

        if(m_ptr && ((m_capacityAndFlags & hkReflect::Detail::HomogeneousArrayImpl::DONT_DEALLOCATE_FLAG) == 0))
        {
            int sizeElem = m_type->getSizeOf();
            hkMemHeapBlockFree<void>(m_ptr, sizeElem*m_capacityAndFlags);
        }
        m_capacityAndFlags = 0;
        m_size = 0;
        m_ptr = HK_NULL;
    }
}

hkVariantArray::~hkVariantArray()
{
    clearAndDeallocate();
}

hkVariantArray& hkVariantArray::operator=(const hkVariantArray& rhs)
{
    if ( this != &rhs )
    {
        // use the impl to replace the current elements with the new ones
        const hkReflect::Detail::ArrayImpl& impl = hkReflect::Detail::HomogeneousArrayImpl::s_instance;
        HK_ON_DEBUG(hkResult res = )impl.spliceInto( this, hkReflect::getType<hkVariantArray>().reinterpret<hkReflect::ArrayType>(),
            0, m_size, hkReflect::ArrayValue( rhs.m_ptr, rhs.m_size, rhs.m_type ) );
        HK_ASSERT_NO_MSG(0x6d6f25f1, res.isSuccess());
    }
    return *this;
}


void hkVariantArray::pushBack( const hkReflect::Var& src )
{
    const hkReflect::Detail::ArrayImpl& impl = hkReflect::Detail::HomogeneousArrayImpl::s_instance;
    HK_ON_DEBUG(hkResult res = )impl.spliceInto( this, hkReflect::getType<hkVariantArray>().reinterpret<hkReflect::ArrayType>(),
        m_size, 0, hkReflect::ArrayValue(src.getAddress(), 1, src.getType()));
    HK_ASSERT_NO_MSG(0x64ed18b2, res.isSuccess());
}

void hkVariantArray::append( const hkVariantArray& src )
{
    // this array cannot change type if it has already been initialized
    HK_ASSERT_NO_MSG(0x41e22f3e, !m_type || m_type->equals( src.m_type ) );
    m_type = src.m_type;

    const hkReflect::Detail::ArrayImpl& impl = hkReflect::Detail::HomogeneousArrayImpl::s_instance;
    impl.spliceInto( this, hkReflect::getType<hkVariantArray>().reinterpret<hkReflect::ArrayType>(),
        m_size, 0, hkReflect::ArrayValue( src.m_ptr, src.m_size, src.m_type ) );
}

void hkVariantArray::setElement(int index, const hkReflect::Var& elem)
{
    HK_ASSERT_NO_MSG(0x269f6922, m_type->equals( elem.getType() ) );
    HK_ASSERT_NO_MSG(0x5e244a8c, index < m_size );
    (*this)[index].assign( elem );
}

void hkVariantArray::trackerHandler(const hkReflect::Var& var, hkMemoryTrackerSnapshot& snapshot)
{
    const hkVariantArray* arr = hkDynCast(var);
    HK_ASSERT_NO_MSG(0x73682592, arr);

    if (int allocSize = arr->getAllocatedSizeInBytes())
    {
        hkMemoryTrackerSnapshot::Block block;
        block.m_name = "buffer_hkVariantArray";
        block.m_type = arr->getType();
        block.m_ptr = arr->getData();
        block.m_num = arr->getSize();
        block.m_size = allocSize;
        snapshot.addLinkedBlock(block, true);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
