// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/Base/Container/Array/hkFixedInplaceArray.h>

/// This is an array of fixed size with in-place storage. It will ASSERT if its capacity is exhausted.
/// Essentially a wrapper around a C array with additional safety checks in debug.
template <typename T, int N>
class hkFixedInplaceArray
{
    public:

    enum { CAPACITY = N };

        typedef T   ItemType;

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_ARRAY, hkFixedInplaceArray);

            /// Creates a zero length array.
        HK_INLINE hkFixedInplaceArray() : m_size(0) {}

            /// Returns the size.
        HK_INLINE int           getSize() const { return m_size; }

            /// Returns true is the array is full.
        HK_INLINE bool      isFull() const { return m_size == N; }

            /// Clear the array.
        HK_INLINE void      clear() { m_size = 0; }

            /// Returns the capacity.
        HK_INLINE int           getCapacity() const { return N; }

            /// Returns a pointer to the beginning of the array.
        HK_INLINE _Ret_notnull_ T* begin() { return m_data; }

    /// Returns a pointer to the end of the array.
    HK_INLINE T*        end() { return m_data + getSize(); }

            /// Returns a pointer to the beginning of the array.
        HK_INLINE _Ret_notnull_ const T* begin() const { return m_data; }

    /// Returns a pointer to the end of the array.
    HK_INLINE const T*  end() const { return m_data + getSize(); }

            /// Append elements to the array.
        HK_INLINE void append(_In_reads_(count) const T* items, int count) { while(count--) pushBack(*items++); }

    /// Append elements to the array.
    template <typename OTHER>
    HK_INLINE void      append( const OTHER& other ) { append( other.begin(), other.getSize() ); }

            /// Sets the size (i.e. number of used elements).
        HK_INLINE void setSize(_In_range_(0, N) int size) { m_size = size; HK_ASSERT(0x48488B10, m_size <= N, "Fixed inplace array capacity overflow"); }

            /// Sets the size (i.e. number of used elements).
        HK_INLINE void setSizeUnchecked(_In_range_(0, N) int size) { setSize(size); }

            /// Checks if the size is zero.
        HK_INLINE hkBool        isEmpty() const { return m_size == 0; }

            /// Increments the size by 1 and returns a reference to the first element created.
        HK_INLINE T&            expandOne() { HK_ASSERT(0x48488B11, m_size < N, "Fixed inplace array capacity overflow"); return m_data[m_size++]; }

        /// Increments the size by 1 and returns a reference to the first element created.
        HK_INLINE _Ret_notnull_ T* expandBy(int n) { HK_ASSERT(0x48488B11, m_size+n <= N, "Fixed inplace array capacity overflow"); T* res = m_data+m_size; m_size += n; return res; }

            /// Read/write access to the i'th element.
        HK_INLINE T& operator[] (_In_range_(>= , 0) int i) { HK_ASSERT(0x48488B12, i >= 0 && i < m_size, "Index out of range"); return m_data[i]; }

            /// Read only access to the i'th element.
        HK_INLINE const T& operator[] (_In_range_(>= , 0) int i) const { HK_ASSERT(0x48488B12, i >= 0 && i < m_size, "Index out of range"); return m_data[i]; }

            /// Return an array view of U. This is only allowed if U and T have the same size.
        template<typename U> HK_INLINE operator hkArrayView<U>() const
        {
            HK_COMPILE_TIME_ASSERT( sizeof( T ) == sizeof( U ) );
            return hkArrayView<U>( static_cast< U* >( &m_data[0] ), m_size );
        }

            /// Return an array view of U. This is only allowed if U and T have the same size.
        template<typename U> HK_INLINE operator hkArrayView<U>()
        {
            HK_COMPILE_TIME_ASSERT( sizeof( T ) == sizeof( U ) );
            return hkArrayView<U>( static_cast< U* >( &m_data[0] ), m_size );
        }

            /// Adds an element to the end.
        HK_INLINE void      pushBack(const T& e) { expandOne() = e; }

            /// Adds an element to the end.
        HK_INLINE void      pushBackUnchecked(const T& e) { pushBack(e); }

            /// Remove the last element.
        HK_INLINE void      popBack() { HK_ASSERT(0x48488B14, m_size > 0, "Empty array"); m_size--; }

            /// Remove the i'th element and replace its location with the last element of the array.
        HK_INLINE void removeAt(_In_range_(>= , 0) int i) { operator[](i) = back(); popBack(); }

            /// Inserts t at index i. Elements from i to the end are copied up one place.
        HK_INLINE void insertAt(_In_range_(>= , 0) int i, const T& t);

            /// Read/write access to the last element.
        HK_INLINE T&            back() { return operator[](m_size-1); }

            /// Read only access to the last element.
        HK_INLINE const T&  back() const { return operator[](m_size-1); }

            /// Linear search.
        inline int                  indexOf(const T& v) const { for(int i=0; i<m_size; ++i) { if(m_data[i] == v) return i; } return -1; }

    protected:

        T   m_data[N];
        int m_size;
};

#include <Common/Base/Container/Array/hkFixedInplaceArray.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
