// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Container/Array/hkArrayUtil.h>
#include <Common/Base/Container/hkContainerAllocators.h>
#include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>

    /// Base of all Havok array types.
    /// This class is allocator-less. Operations which may change the capacity must supply
    /// an allocator. Derived classes often have an implicit allocator.
    /// Note that, for performance reasons, order may not be preserved when deleting elements.
    ///
    /// The hkArray uses type traits for optimized handling the POD (plain old data)
    /// and non-POD (nontrivial constructor and/or destructor) items.
    ///
    /// Object lifetimes: Memory at elements [0, getSize()) are live objects.
    /// Memory at elements [getSize(),getCapacity()) is uninitialized.
    ///
    /// If the type of the array item is non-pod then the following must be satisfied:
    ///     - relocatable with memmove
    ///     - user-defined default constructor
    ///     - user-defined copy constructor
    ///     - user-defined destructor
    ///     - user-defined assignment
    ///
    /// Because of the first point, note that objects which contain pointers to data within
    /// themselves (such as hkInplaceArray or hkStringBuf) may not be used in a hkArray.
    ///
    /// If the class contains member of hkArray type you must implement a copy constructor.
    /// This is because the hkArray copy constructor is private to avoid accidental pass-by-value usage.
    /// Thus the compiler generated copy constructor is unable to access the hkArray copy constructor.
    ///
    /// Sometimes you may want to treat your class/struct as a pod type for hkArray
    /// performance reasons.
    ///
    /// To do this you should use one of the following macros:
    ///     - HK_DECLARE_POD_TYPE();
    ///       It must be specified in public area of the class body.
    ///     - HK_DECLARE_POD_TYPE_IF_POD( T );
    ///       It must be specified in public area of the class
    ///       template body, where the T is the template argument.
    ///     - HK_DEFINE_AS_POD_TYPE( typename );
    ///       It must be specified outside the class body. This macro can be used in
    ///       the cpp file to define the class as a pod type locally.
    ///
    /// Note: Please make sure that the class does not contain non-pod members when you
    /// define it as a pod.
    ///
    /// Note: For class/struct defined as pod, the requirements for constructors and destructors
    /// are not applicable.
template <typename T>
class hkArrayBase
{
        friend class hkArrayUtil;
        friend class hkArraySpuUtil;
        friend class hkaiArrayUtil;
        template<typename TYPE> friend class hkArraySpu;

    public:

        typedef T value_type;

        HK_DECLARE_CLASS(hkArrayBase, NewExplicit, TrackAs(hkReflect::Detail::GenericArray));

            /// Creates a zero length array.
        HK_INLINE hkArrayBase();

            /// Noncopying initialization from an existing external buffer.
            /// This does not copy the array but uses it in place until its capacity
            /// is exceeded at which point a reallocation occurs and the array behaves
            /// like a normal hkArray.
            /// The caller must ensure that the buffer is valid for the lifetime
            /// of this array and for deallocation of the buffer.
            /// It is assumed that buffer elements up to "size" are initialized, and those between "size" and "capacity" are uninitialized.
        HK_INLINE hkArrayBase(_In_count_(capacity) T* buffer, _In_range_(0, capacity) int size, _In_range_(>=, 0) int capacity);

    protected:

            /// Array cannot be copied without explicit allocator.
        HK_INLINE hkArrayBase(const hkArrayBase& a)  { HK_ASSERT_NO_MSG(0x54a6d8d1, 0); }

            /// Array cannot be copied without explicit allocator.
        HK_INLINE hkArrayBase& operator= (const hkArrayBase& a) { HK_ASSERT_NO_MSG(0x54a6d8d0, 0); return *this; }

    public:

            /// Destructs array members.
        HK_INLINE ~hkArrayBase();

            /// Read/write access to the i'th element.
        HK_ALWAYS_INLINE T& operator[] (_In_range_(0, m_size - 1) int i);

            /// Read only access to the i'th element.
        HK_ALWAYS_INLINE const T& operator[] (_In_range_(0, m_size - 1) int i) const;

            /// Return an array view of U. This is only allowed if U and T have the same size.
        template<typename U> HK_INLINE operator hkArrayView<U>() const;

            /// Return a view into this array.
            /// \see hkArrayView
        _Check_return_
        HK_INLINE hkArrayView<T> view();

            /// Return a read-only view into this array.
            /// \see hkArrayView
        _Check_return_
        HK_INLINE hkArrayView<const T> view() const;

            /// Read/write access to the first element.
        _Check_return_
        HK_ALWAYS_INLINE T& front();

            /// Read only access to the first element.
        _Check_return_
        HK_ALWAYS_INLINE const T& front() const;

            /// Read/write access to the last element.
        _Check_return_
        HK_ALWAYS_INLINE T& back();

            /// Read only access to the last element.
        _Check_return_
        HK_ALWAYS_INLINE const T& back() const;

            /// Returns the size.
        _Check_return_
        _Ret_range_(>=, 0)
        HK_ALWAYS_INLINE int getSize() const;

            /// Returns the capacity.
        _Check_return_
        HK_ALWAYS_INLINE int getCapacity() const;

            /// Checks if the size is zero.
        _Check_return_
        HK_ALWAYS_INLINE hkBool isEmpty() const;

            /// Sets the size to zero.
        HK_INLINE void clear();

            /// Sets the size to zero and deallocates storage.
        HK_INLINE void _clearAndDeallocate(_Inout_ hkMemoryAllocator& alloc);

            /// Tries to reduce the capacity to avoid wasting storage. If shrinkExact is true the resulting capacity
            /// is size+numElementsLeft
        HK_INLINE void _optimizeCapacity(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(>=, 0) int numFreeElemsLeft,
            hkBool32 shrinkExact=false );

            /// Removes the element at the specified index. The last array element is used to replace the removed element, and the size is reduced by 1.
            /// This is very fast, but note that the order of elements is changed.
        HK_INLINE void removeAt(_In_range_(0, m_size - 1) int index);

            /// Removes the element at the specified index, copying elements down one slot as in the STL array.
            /// Slower than removeAt(), but the order is unchanged.
        HK_INLINE void removeAtAndCopy(_In_range_(0, m_size - 1) int index);

            /// Removes several elements at the specified index, copying elements down as in the STL array.
        HK_INLINE void removeAtAndCopy(
            _In_range_(0, m_size - 1) int index,
            _In_range_(1, m_size - index) int numToRemove);

            /// Returns the index of the first occurrence of t, or -1 if not found.
        _Pre_satisfies_(end < 0 || (start <= end && end <= m_size))
        _Check_return_
        HK_INLINE int indexOf(
            const T& t,
            _In_range_(0, m_size - 1) int start=0,
            int end=-1) const;

            /// Returns index of the last occurrence of t, or -1 if not found.
        _Check_return_
        HK_INLINE int lastIndexOf(const T& t) const;

            /// Removes the last element.
        HK_INLINE void popBack(_In_range_(0, m_size) int numElemsToRemove = 1);

            /// Adds an element to the end.
        HK_INLINE void _pushBack(
            _Inout_ hkMemoryAllocator& alloc,
            const T& e);

        /// Checks if there is enough capacity for one more element and grows the storage if there isn't.
        void _ensureHasCapacityForOneMore(_Inout_ hkMemoryAllocator& alloc);

        /// Constructs a new element at passed in position in the array with provided perfectly forwarded argument(s).
        
        void _emplace(_Inout_ hkMemoryAllocator& alloc, _In_range_(0, m_size) int index);

        template <typename T1>
        void _emplace(_Inout_ hkMemoryAllocator& alloc, _In_range_(0, m_size) int index, T1&& t1);

        template <typename T1, typename T2>
        void _emplace(_Inout_ hkMemoryAllocator& alloc, _In_range_(0, m_size) int index, T1&& t1, T2&& t2);

        template <typename T1, typename T2, typename T3>
        void _emplace(_Inout_ hkMemoryAllocator& alloc, _In_range_(0, m_size) int index, T1&& t1, T2&& t2, T3&& t3);

        template <typename T1, typename T2, typename T3, typename T4>
        void _emplace(_Inout_ hkMemoryAllocator& alloc, _In_range_(0, m_size) int index, T1&& t1, T2&& t2, T3&& t3, T4&& t4);

        template <typename T1, typename T2, typename T3, typename T4, typename T5>
        void _emplace(_Inout_ hkMemoryAllocator& alloc, _In_range_(0, m_size) int index, T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5);

        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6>
        void _emplace(_Inout_ hkMemoryAllocator& alloc, _In_range_(0, m_size) int index, T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5, T6&& t6);

        /// Constructs a new element at the end of the array.
        
        
        void _emplaceBack(_Inout_ hkMemoryAllocator& alloc);

        /// Constructs a new element at the end of the array with provided perfectly forwarded argument.
        template <typename T1>
        void _emplaceBack(_Inout_ hkMemoryAllocator& alloc, T1&& t1);

        /// Constructs a new element at the end of the array with provided perfectly forwarded arguments.
        template <typename T1, typename T2>
        void _emplaceBack(_Inout_ hkMemoryAllocator& alloc, T1&& t1, T2&& t2);

        /// Constructs a new element at the end of the array with provided perfectly forwarded arguments.
        template <typename T1, typename T2, typename T3>
        void _emplaceBack(_Inout_ hkMemoryAllocator& alloc, T1&& t1, T2&& t2, T3&& t3);

        /// Constructs a new element at the end of the array with provided perfectly forwarded arguments.
        template <typename T1, typename T2, typename T3, typename T4>
        void _emplaceBack(_Inout_ hkMemoryAllocator& alloc, T1&& t1, T2&& t2, T3&& t3, T4&& t4);

        /// Constructs a new element at the end of the array with provided perfectly forwarded arguments.
        template <typename T1, typename T2, typename T3, typename T4, typename T5>
        void _emplaceBack(_Inout_ hkMemoryAllocator& alloc, T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5);

        /// Constructs a new element at the end of the array with provided perfectly forwarded arguments.
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6>
        void _emplaceBack(_Inout_ hkMemoryAllocator& alloc, T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5, T6&& t6);

            /// Adds an element to the end. No check for resize.
        _Pre_satisfies_(m_size < (m_capacityAndFlags & CAPACITY_MASK))
        HK_INLINE void pushBackUnchecked(const T& e);

            /// Adds an element to the end. No check for resize.
            /// This function is thread-safe with respect to other atomic operations.
        HK_INLINE void pushBackUncheckedAtomic(const T& e);

            /// Ensures no reallocation occurs until at least size n.
            /// Returns HK_FAILURE if any memory allocation failed, HK_SUCCESS otherwise.
        HK_INLINE hkResult _reserve(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(>=, 0) int n);

            /// Ensures no reallocation occurs until size n.
        HK_INLINE hkResult _reserveExactly(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(>=, 0) int n);

            /// Sets the size.
            /// If the array is expanded, new elements are uninitialized.
        HK_INLINE void _setSize(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(>=, 0) int size);

            /// Sets the size to n.
            /// If the array is expanded, new elements initialized with 'fill'.
        HK_INLINE void _setSize(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(>=, 0) int size,
            const T& fill);

            /// Safely try to set the size.
            /// If the array is expanded, new elements are uninitialized.
            /// Returns HK_FAILURE if any memory allocation failed, HK_SUCCESS otherwise.
        _Must_inspect_result_
        HK_INLINE hkResult _trySetSize(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(>=, 0) int size);

            /// Safely try to set the size.
            /// If the array is expanded, new elements are initialized with 'fill'.
            /// Returns HK_FAILURE if any memory allocation failed, HK_SUCCESS otherwise.
        _Must_inspect_result_
        HK_INLINE hkResult _trySetSize(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(>=, 0) int size,
            const T& fill);

            /// Sets the size assuming the capacity to be sufficient.
            /// If the array is expanded, new elements are uninitialized.
        HK_INLINE void setSizeUnchecked(_In_range_(>=, 0) int size);

            /// Sets the size assuming the capacity to be sufficient.
            /// If the array is expanded, new elements initialized with 'fill'.
        HK_INLINE void setSizeUnchecked(_In_range_(>=, 0) int size, _In_ const T& fill);

            /// Increments the size by 1 and returns a reference to the first element created.
        HK_INLINE T& _expandOne(_Inout_ hkMemoryAllocator& alloc);

            /// Increments the size by n and returns a pointer to the first element created.
        HK_INLINE T* _expandBy(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(>=, 0) int n);

            /// Increments the size by n, fills in the results with fill, and returns a pointer to the first element created.
        HK_INLINE T* _expandBy(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(>=, 0) int n,
            const T& fill );

            /// Increments the size by n and returns a pointer to the first element created. No check for resize!
        HK_INLINE T* expandByUnchecked(_In_range_(>=, 0) int n);

            /// Increments the size by n and returns a pointer to the first element created. No check for resize!
            /// This function is thread-safe with respect to other atomic operations.
        HK_INLINE T* expandByUncheckedAtomic(int n);

            /// Expands the array by numToInsert at the specified index.
        HK_INLINE T* _expandAt(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(0, m_size) int index,
            _In_range_(>=, 0) int numToInsert);

            /// Inserts the array a at index i.
        HK_INLINE void _insertAt(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(0, m_size) int i,
            _In_reads_(numElems) const T* a,
            _In_range_(>=, 0) int numElems);

            /// Inserts t at index i.
            /// Elements from i to the end are copied up one place.
        HK_INLINE void _insertAt(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(0, m_size) int i,
            const T& t);

            /// Appends the array a.
        HK_INLINE void _append(
            _Inout_ hkMemoryAllocator& alloc,
            _In_reads_(numElems) const T* a,
            _In_range_(>=, 0) int numElems);

            /// Replaces elements [i,i+ndel) with the supplied array.
            /// This method avoids redundant copying associated with separate remove & insert steps.
        HK_INLINE void _spliceInto(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(0, m_size) int i,
            _In_range_(0, m_size - i) int ndel,
            _In_reads_(numElems) const T* p,
            _In_range_(>=, 0) int numElems);

            /// Removes all occurrences of t, preserving the order of the remaining elements.
        HK_INLINE void removeAllAndCopy(const T& t);

            /// Adds the element at the end of the array, or returns false if the array would have
            /// to be resized first.
        _Must_inspect_result_
        HK_INLINE hkBool tryPushBack(const T& t);


        //
        // Iterator-based methods
        //

            /// Random access iterator to T.
        typedef T* iterator;

            /// Random access iterator to const T.
        typedef const T* const_iterator;

            /// Returns an STL-like iterator to the first element.
        _Check_return_
        HK_ALWAYS_INLINE iterator begin();

            /// Returns an STL-like iterator to the 'one past the last' element.
        _Check_return_
        HK_ALWAYS_INLINE iterator end();

            /// Returns an STL-like const iterator to the first element.
        _Check_return_
        HK_ALWAYS_INLINE const_iterator begin() const;

            /// Returns an STL-like const iterator to the 'one past the last' element.
        _Check_return_
        HK_ALWAYS_INLINE const_iterator end() const;

            /// Erase an element at the given position. Return an iterator to the element that replaced it, or end().
            /// Like removeAt(), the last array element is used to replace the removed element. This is very fast, but the order of elements is changed.
        HK_INLINE iterator erase(_In_ iterator it);

            /// Erase the range of elements [first,last[. Return an iterator to the element that followed the last erased element, or end().
            /// Unlike erase(it), the elements are moved and order is always preserved.
        HK_INLINE iterator eraseAndCopy(_In_ptrdiff_count_(last) iterator first, _In_ iterator last);


        //
        // Utilities
        //

            /// Utility to copy some elements.
        template<typename U>
        static void HK_CALL copy(
            _Out_writes_all_(n) T* dst,
            _In_reads_(n) const U* src,
            _In_range_(>=, 0) int n);

            /// Set the storage of the array to use the provided
            /// external memory allocation. Set size and capacity as provided
            /// by the user. Existing array data is forgotten without deallocating
            /// or destructing objects, use clearAndDeallocate if required.
            /// The array objects must be of POD type. In particular, destructors will
            /// be called on any objects remaining in the array when it is destructed.
            /// The memory block is owned by the Array and will be deleted on destruction
            /// or resizing
        HK_INLINE void setDataAutoFree(
            _In_count_(capacity) T* ptr,
            _In_range_(0, capacity) int size,
            _In_range_(>=, 0) int capacity);

            /// Set the storage of the array to use the provided
            /// external memory allocation. Set size and capacity as provided
            /// by the user. Existing array data is forgotten without deallocating
            /// or destructing objects, use clearAndDeallocate if required.
            /// The array objects must be of POD type. In particular, destructors will
            /// be called on any objects remaining in the array when it is destructed.
            /// The memory block is not owned by the Array and will not be deleted on destruction
            /// or resizing
        HK_INLINE void setDataUserFree(
            _In_count_(capacity) T *ptr,
            _In_range_(0, capacity) int size,
            _In_range_(>=, 0) int capacity);

            /// Advanced use only. Set the storage of the array, bypassing
            /// correctness checks. Size, capacity and flags are set as provided.
            /// Note that constructor/destructor semantics of non-POD types may not
            /// be respected.
        HK_INLINE void _setDataUnchecked(
            _In_count_(capacityAndFlags & CAPACITY_MASK) T *ptr,
            _In_range_(0, capacityAndFlags & CAPACITY_MASK) int size,
            int capacityAndFlags);

            /// Get the capacity and the flags - advanced use
        HK_ALWAYS_INLINE int getCapacityAndFlags() const;

    protected:

            // Internal type checking
        HK_INLINE void _setData(
            _In_count_(capacityAndFlags & CAPACITY_MASK) T *ptr,
            _In_range_(0, capacityAndFlags & CAPACITY_MASK) int size,
            int capacityAndFlags,
            hkTrait::TypeIsPod);

        HK_INLINE void _reserveInitial(
            _Inout_ hkMemoryAllocator& alloc,
            _In_range_(>=, 0) int n);

        template <typename U>
        HK_INLINE hkArrayBase<T>& copyFromArray(
            _Inout_ hkMemoryAllocator& alloc,
            _In_reads_(size) const U* data,
            _In_range_(>= , 0) int size);

        template <typename U>
        HK_INLINE void populateFromData(
            _In_reads_(size) const U* data,
            _In_range_(>= , 0) int size,
            hkTrait::TypeIsPod, hkTrait::TypeIsPod);

        template <typename U, typename TTrait, typename UTrait>
        HK_INLINE void populateFromData(
            _In_reads_(size) const U* data,
            _In_range_(>= , 0) int size,
            TTrait, UTrait);

    public:

        // Public so that the serialization can access it.
        enum
        {
            CAPACITY_MASK        = int(0x3FFFFFFF),
            FLAG_MASK            = int(0xC0000000),
            DONT_DEALLOCATE_FLAG = int(0x80000000), // Indicates that the storage is not the array's to delete
            FORCE_SIGNED = -1,
            RESERVE_GROWTH_FACTOR = 2
        };

    protected:

        _Field_size_part_opt_((m_capacityAndFlags & CAPACITY_MASK), m_size) T* m_data;
        _Field_range_(>=, 0) int m_size;
        int m_capacityAndFlags; // highest 2 bits indicate any special considerations about the allocation for the array
};

    /// Disallow hkArrayBase of const elements. Use hkArrayView to represent immutable arrays.
template <typename T> class hkArrayBase<const T>;

namespace hkArrayDetail
{
    /**
        Check that the type T has
        - a getSize() method returning int
        - a begin() method returning T::iterator
    */
    template<typename T, typename = void>
    struct HasContainerAPI : hkTrait::FalseType {};

    template<typename T>
    struct HasContainerAPI<T, typename hkTrait::Voider<typename T::iterator>::Type>
    {
        template<typename U> static char check(U* u,
            typename hkTrait::EnableIf<hkTrait::TypesAreEqual<int, decltype(u->getSize())>::result, U>::Type* = nullptr,
            typename hkTrait::EnableIf<hkTrait::TypesAreEqual<typename U::iterator, decltype(u->begin())>::result, U>::Type* = nullptr);
        template<typename U> static int check(...);
        enum {
#if defined(_MSC_VER) && _MSC_VER < 1900
            result = true
#else
            result = sizeof(check<T>(nullptr)) == sizeof(char)
#endif
        };
    };
}


    /// Array which uses the hkMemoryRouter::heap allocator.
template <typename T, typename Allocator=hkContainerHeapAllocator>
class hkArray : public hkArrayBase<T>
{
    public:

        HK_DECLARE_CLASS(hkArray, NewTemplate, Reflect);
        HK_REFLECT_AS_ARRAY(&hkReflect::Detail::hkArrayImpl::s_instance, T);
        HK_RECORD_ATTR( hk::ReflectDetails(parents=false, fields=true) );
        HK_RECORD_ATTR( hk::FixupReflectedMethods(&hkReflect::Detail::hkArrayImpl::fixupReflectedMethods) );
        HK_RECORD_ATTR( hk::MemoryTracker(opaque=true, handler=&hkReflect::Tracker::hkArrayHandler<T, Allocator>::func) );
        HK_RECORD_ATTR( hk::Validate, hk::HasCustomValidation );

        typedef T value_type;
        typedef Allocator allocator_type;
        friend class hkArrayUtil;

        typedef hkArray<T,hkContainerTempAllocator> Temp;
        typedef hkArray<T,hkContainerDebugAllocator> Debug;
        typedef hkArray<T,hkContainerHeapAllocator> Heap;
        typedef hkArray<T,hkContainerAlignedHeapAllocator<T>> Aligned;

            /// Creates a zero length array.
        HK_INLINE hkArray() : hkArrayBase<T>() {}

        /// Copy another array
        template <typename Container>
        HK_INLINE hkArray(const Container& a, typename hkTrait::EnableIf<hkArrayDetail::HasContainerAPI<Container>::result, Container>::Type* = nullptr);

        HK_INLINE hkArray(const hkArray& a);

            ///
        HK_INLINE ~hkArray();

            /// Creates an array of size n. If hkTrait::IsPodType<T>::result is true, all elements are uninitialized. Otherwise the default constructor is called for each element.
        explicit HK_INLINE hkArray(_In_range_(>=, 0) int size);

            /// Creates an array of n elements initialized to 'fill'.
        HK_INLINE hkArray(_In_range_(>=, 0) int size, const T& fill);

            /// Noncopying initialization from an existing external buffer.
            /// This does not copy the array but uses it in place until its capacity
            /// is exceeded at which point a reallocation occurs and the array behaves
            /// like a normal hkArray.
            /// The caller must ensure that the buffer is valid for the lifetime
            /// of this array and for deallocation of the buffer.
            /// It is assumed that buffer elements up to "size" are initialized, and those between "size" and "capacity" are uninitialized.
        HK_INLINE hkArray(
            _In_count_(capacity) T* buffer,
            _In_range_(0, capacity) int size,
            _In_range_(>=, 0) int capacity)
            : hkArrayBase<T>(buffer, size, capacity) {}

    public:

        /// Copy another array
        template <typename Container>
        HK_INLINE typename hkTrait::EnableIf<hkArrayDetail::HasContainerAPI<Container>::result, hkArray<T, Allocator>&>::Type operator= (const Container& a);

        HK_INLINE hkArray& operator= (const hkArray& a);

            /// Swaps this array's internal storage with 'a'.
        void swap(_Inout_ hkArray& a);

        HK_INLINE void clearAndDeallocate()
        { hkArrayBase<T>::_clearAndDeallocate(allocator_type().get(this)); }

        HK_INLINE void optimizeCapacity(
            _In_range_(>=, 0) int numFreeElemsLeft,
            hkBool32 shrinkExact=false)
        { hkArrayBase<T>::_optimizeCapacity(allocator_type().get(this), numFreeElemsLeft, shrinkExact); }

        HK_INLINE void pushBack(const T& e)
        { hkArrayBase<T>::_pushBack(allocator_type().get(this), e); }

        /// Constructs a new element at the end of the array with provided perfectly forwarded argument.
        
        T& emplace(_In_range_(0, m_size) int index);

        template <typename T1>
        T& emplace(_In_range_(0, m_size) int index, T1&& t1);

        template <typename T1, typename T2>
        T& emplace(_In_range_(0, m_size) int index, T1&& t1, T2&& t2);

        template <typename T1, typename T2, typename T3>
        T& emplace(_In_range_(0, m_size) int index, T1&& t1, T2&& t2, T3&& t3);

        template <typename T1, typename T2, typename T3, typename T4>
        T& emplace(_In_range_(0, m_size) int index, T1&& t1, T2&& t2, T3&& t3, T4&& t4);

        template <typename T1, typename T2, typename T3, typename T4, typename T5>
        T& emplace(_In_range_(0, m_size) int index, T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5);

        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6>
        T& emplace(_In_range_(0, m_size) int index, T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5, T6&& t6);

        /// Constructs a new element at the end of the array.
        
        
        T& emplaceBack();

        template <typename T1>
        T& emplaceBack(T1&& t1);

        template <typename T1, typename T2>
        T& emplaceBack(T1&& t1, T2&& t2);

        template <typename T1, typename T2, typename T3>
        T& emplaceBack(T1&& t1, T2&& t2, T3&& t3);

        template <typename T1, typename T2, typename T3, typename T4>
        T& emplaceBack(T1&& t1, T2&& t2, T3&& t3, T4&& t4);

        template <typename T1, typename T2, typename T3, typename T4, typename T5>
        T& emplaceBack(T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5);

        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6>
        T& emplaceBack(T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5, T6&& t6);

        HK_INLINE hkResult reserve(_In_range_(>=, 0) int size)
        { return hkArrayBase<T>::_reserve(allocator_type().get(this), size); }

        HK_INLINE hkResult reserveExactly(_In_range_(>=, 0) int size)
        { return hkArrayBase<T>::_reserveExactly(allocator_type().get(this), size); }

        HK_INLINE void setSize(_In_range_(>=, 0) int size)
        { hkArrayBase<T>::_setSize(allocator_type().get(this), size); }

        HK_INLINE void setSize(
            _In_range_(>=, 0) int size,
            const T& fill)
        { hkArrayBase<T>::_setSize(allocator_type().get(this), size, fill); }

        _Must_inspect_result_
        HK_INLINE hkResult trySetSize(_In_range_(>=, 0) int size )
        { return hkArrayBase<T>::_trySetSize(allocator_type().get(this), size ); }

        _Must_inspect_result_
        HK_INLINE hkResult trySetSize(_In_range_(>=, 0) int size, const T& fill)
        { return hkArrayBase<T>::_trySetSize(allocator_type().get(this), size, fill); }

        HK_INLINE T& expandOne()
        { return hkArrayBase<T>::_expandOne(allocator_type().get(this)); }

        HK_INLINE T* expandBy(_In_range_(>=, 0) int n)
        { return hkArrayBase<T>::_expandBy(allocator_type().get(this), n); }

        HK_INLINE T* expandBy( _In_range_(>=, 0 ) int n, const T& fill)
        { return hkArrayBase<T>::_expandBy(allocator_type().get(this), n, fill); }

        HK_INLINE T* expandAt(
            _In_range_(0, m_size) int i,
            _In_range_(>=, 0) int n)
        { return hkArrayBase<T>::_expandAt(allocator_type().get(this), i, n); }

        HK_INLINE void insertAt(
            _In_range_(0, m_size) int i,
            const T& e)
        { hkArrayBase<T>::_insertAt(allocator_type().get(this),i,e); }

        HK_INLINE void insertAt(
            _In_range_(0, m_size) int i,
            _In_reads_(numElems) const T* a,
            _In_range_(>=, 0) int numElems )
        { hkArrayBase<T>::_insertAt(allocator_type().get(this),i,a,numElems); }

        HK_INLINE void append(
            _In_reads_(numElems) const T* a,
            _In_range_(>=, 0) int numElems )
        { hkArrayBase<T>::_append(allocator_type().get(this),a,numElems); }

        HK_INLINE void spliceInto(
            _In_range_(0, m_size) int i,
            _In_range_(0, m_size - i) int ndel,
            _In_reads_(numElems) const T* p,
            _In_range_(>=, 0) int numElems )
        { return hkArrayBase<T>::_spliceInto(allocator_type().get(this),i,ndel,p,numElems); }

        template <typename K>
        HK_INLINE void append(const hkArrayBase<K>& other)
        { append(other.begin(), other.getSize()); }

        HK_INLINE void append(hkArrayView<const T> other)
        { append(other.begin(), other.getSize()); }

        hkMemoryAllocator& getAllocator() const { return allocator_type().get(this); }
};

template<typename T, typename Allocator>
bool hkArray<T, Allocator>::reflectValidate() const
{
    // Size >= 0
    // Capacity >= Size
    // If capacity is >0, data must be !=0, else data must be ==0
    return (this->getSize() >= 0 && (this->getCapacity() >= this->getSize())
        && (this->getCapacity() ? (bool)this->m_data : !this->m_data));
}


namespace hkTrait
{
    // Helper class for UninitializedStorage.
    // Can also be used directly to get elements with alignment ALIGN.
    template<int ALIGN>
    struct AlignedStorageElement;

    // Use natural alignment for small values, rather than HK_ALIGN, so that UninitializedStorage
    // can be used in managed functions. AlignedStorageElement is implicitly reflected.
    template<> struct AlignedStorageElement<1> { typedef hkUint8 Type; };
    template<> struct AlignedStorageElement<2> { typedef hkUint16 Type; };
    template<> struct AlignedStorageElement<4> { typedef hkUint32 Type; };
    template<> struct AlignedStorageElement<8> { typedef HK_ALIGN(char Type[8], 8); };
    template<> struct AlignedStorageElement<16> { typedef HK_ALIGN(char Type[16], 16); };
    template<> struct AlignedStorageElement<32> { typedef HK_ALIGN(char Type[32], 32); };

    // Provides a (POD) type for use as uninitialized storage for any object whose size is at
    // most NUMBYTES and whose alignment requirement is a divisor of ALIGN.
    template<unsigned NUMBYTES, unsigned ALIGN>
    struct AlignedStorage
    {
        HK_DECLARE_CLASS(AlignedStorage, New, Reflect);

        typename AlignedStorageElement<ALIGN>::Type m_storage[NUMBYTES/ALIGN];
    };

    // Uninitialized aligned storage for N elements of type T.
    // We can't just store a T m_storage[N] because we don't want to default construct those elements.
    template<typename T, unsigned N>
    struct UninitializedStorage
    {
        HK_DECLARE_CLASS(UninitializedStorage, New, Reflect);

        _Check_return_ inline T* get() { return reinterpret_cast<T*>(&m_storage); }
        _Check_return_ inline const T* get() const { return reinterpret_cast<const T*>(&m_storage); }

        AlignedStorage<sizeof(T) * N, HK_ALIGN_OF(T)> m_storage;
    };
}

    /// Array that has an internal storage capacity within the class itself.
    /// Originally hkArray::m_data points to hkInplaceArray::m_storage.
    /// It is safe to expand the capacity beyond the internal capacity. In this
    /// case the array behaves like a normal hkArray (i.e., m_data points to heap
    /// memory instead of to &m_storage[0]).
    /// Note that once the builtin capacity has been exceeded,
    /// the inplace elements are unused even if subsequently resized smaller
    /// than the original capacity.
template <typename T, unsigned N, typename Allocator=hkContainerHeapAllocator>
class hkInplaceArray : public hkArray<T,Allocator>
{
    public:

        typedef T value_type;
        static const unsigned FixedSize = N;

        typedef hkInplaceArray<T,N,hkContainerTempAllocator> Temp;
        typedef hkInplaceArray<T,N,hkContainerDebugAllocator> Debug;
        typedef hkInplaceArray<T,N,hkContainerHeapAllocator> Heap;

        HK_DECLARE_CLASS(hkInplaceArray, NewExplicit, TrackAs(hkArray<T, Allocator>), Reflect);
        HK_REFLECT_AS_ARRAY(&hkReflect::Detail::hkInplaceArrayImpl::s_instance, T);

            /// Creates an array with no elements.
        HK_INLINE hkInplaceArray();

            /// Creates an array with the specified initial size.
        explicit HK_INLINE hkInplaceArray(_In_range_(>=, 0) int size);

        HK_INLINE hkInplaceArray(const hkInplaceArray<T,N,Allocator>& a);
        HK_INLINE hkInplaceArray(const hkArrayBase<T>& a);
        HK_INLINE hkInplaceArray(hkArrayView<const T> a);

        HK_INLINE ~hkInplaceArray() {}

            /// Copies the array a.
        HK_INLINE hkArray<T, Allocator>& operator= (const hkArrayBase<T>& a);
        HK_INLINE hkArray<T, Allocator>& operator= (const hkInplaceArray<T,N,Allocator>& a);
        HK_INLINE hkArray<T, Allocator>& operator= (hkArrayView<const T> a);

            /// Tries to reduce the capacity to avoid wasting storage
        HK_INLINE void optimizeCapacity(
            _In_range_(>=, 0) int numFreeElemsLeft,
            hkBool32 shrinkExact=false);

        _Check_return_
        HK_INLINE hkBool wasReallocated() const;

            /// returns true if the array is still using its inplace buffer.
            /// This check is done using only the mask field, so do not use
            /// this function for arrays which can be serialized
        _Check_return_
        HK_INLINE int stillInplaceUsingMask() const;

    protected:

        hkTrait::UninitializedStorage<T,N> m_storage;
};


    /// An array that has a small internal storage capacity, aligned to HK_REAL_ALIGNMENT bytes within the class itself.
template <typename T, unsigned N, typename Allocator=hkContainerHeapAllocator>
class hkInplaceArrayAligned16 : public hkArray<T, Allocator>
{
    public:

        typedef T value_type;
        static const unsigned FixedSize = N;

        typedef hkInplaceArrayAligned16<T,N,hkContainerTempAllocator> Temp;
        typedef hkInplaceArrayAligned16<T,N,hkContainerDebugAllocator> Debug;
        typedef hkInplaceArrayAligned16<T,N,hkContainerHeapAllocator> Heap;

        HK_DECLARE_CLASS(hkInplaceArrayAligned16, NewExplicit, TrackAs(hkArray<T, Allocator>));

            /// Creates an array with the specified initial size.
        explicit HK_INLINE hkInplaceArrayAligned16(_In_range_(>=, 0) int size = 0);

        HK_INLINE ~hkInplaceArrayAligned16(){}

            /// Copies the array a.
        HK_INLINE hkArray<T>& operator= (const hkArrayBase<T>& a);

            /// Copies the array a.
        HK_INLINE hkArray<T>& operator= (const hkInplaceArrayAligned16<T,N>& a);

        _Check_return_
        HK_INLINE hkBool wasReallocated() const;

            /// returns true if the array is still using its inplace buffer.
            /// This check is done using only the mask field, so do not use
            /// this function for arrays which can be serialized
        _Check_return_
        HK_INLINE int stillInplaceUsingMask() const;

        HK_INLINE T& getStorage(_In_range_(0, N - 1) int index) { return ((T*)m_storage)[index]; }
        HK_INLINE const T& getStorage(_In_range_(0, N - 1) int index) const { return ((T*)m_storage)[index]; }

    public:

        int m_padding; // sizeof(base class) + padding == 16 bytes
        HK_ALIGN_REAL( hkUint8 m_storage[sizeof(T) * N] );
};

#include <Common/Base/Container/Array/hkArray.inl>

// Workaround for alignof in hkPreBuild generated file, needed for VS2012
#if defined(_MSC_VER) && _MSC_VER < 1900
#define alignof __alignof
#endif
#include <Common/Base/_Auto/TemplateTypes/hkArray_Types.inl>
#if defined(_MSC_VER) && _MSC_VER < 1900
#undef alignof
#endif

// These are needed to avoid duplicate symbols in the dll build
#if !defined(__HAVOK_PARSER__) && defined(HK_DYNAMIC_DLL)
HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON hkArrayBase<int>;
HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON hkArray<int>;
HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template class HK_EXPORT_COMMON hkInplaceArray<int, 64>;
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
