// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// Heap Sort, O(n log(n)), not recursive
// see http://www.csse.monash.edu.au/~lloyd/tildeAlgDS/Sort/Heap.html for explanation

namespace hkAlgorithm
{

    template<typename T, typename L>
void HK_CALL downHeap(_Inout_updates_(n) T *pArr, int k, _In_range_(>=, k) int n, L cmpLess)
    {
        // precondition: array pArr[k+1,n] is a heap
        // postcondition: array pArr[k,n] is a heap

        T temp = move( pArr[ k - 1 ] );
        //find them maximum of the new 'parent' and all of its children
        while ( k <= n / 2 )
        {
            int child = 2 * k;
            //don't process children 'outside' of the array
            if ( ( child < n ) && cmpLess( pArr[ child - 1 ], pArr[ child ] ) )
            {
                child++;
            }
            //take the maximum
            if ( cmpLess( temp, pArr[ child - 1 ] ) )
            {
                pArr[ k - 1 ] = move( pArr[ child - 1 ] );
                k = child;
            }
            else
            {
                break;
            }
        }
        pArr[ k - 1 ] = move( temp );
    }


    template<typename T, typename L>
void HK_CALL heapSort(_Inout_updates_(iSize) T *pArr, int iSize, L cmpLess)
    {
        int k;
        int n = iSize;
        //create a heap
        for ( k = n / 2; k > 0; k-- )
        {
            downHeap<T, L>( pArr, k, n, cmpLess );
        }

        // now swap the
        while ( n >= 1 )
        {
            //a[0] is now known to be largest, so put it to the end, and downheap 0..n-1
            swap( pArr[ 0 ], pArr[ n - 1 ] );

            n = n - 1;
            downHeap<T, L>( pArr, 1, n, cmpLess );
        }
    }

    template<typename T, typename L>
HK_INLINE void HK_CALL insertionSort(_Inout_updates_(size) T* pArr, int size, L cmpLess)
    {
        for ( int i = 0; i < size; i++ )
        {
            T key = move( pArr[ i ] );
            int j = i;
            while ( j > 0 && cmpLess( key, pArr[ j - 1 ] ) )
            {
                pArr[ j ] = move( pArr[ j - 1 ] );
                j--;
            }
            pArr[ j ] = move( key );
        }
    }

    template<typename T, typename L>
void HK_CALL quickSortRecursive(_Inout_updates_(h + 1) T *pArr, int d, int h, L cmpLess)
    {
        int i, j;
        #ifndef HK_ARCH_ARM
        HK_ALIGN16( T str );
        #else
        T str;
        #endif
        begin:
        i = h;
        j = d;

        str = pArr[ ( d + h ) >> 1 ];

        do {

            while ( cmpLess( pArr[ j ], str ) ) { j++; }
            while ( cmpLess( str, pArr[ i ] ) ) { i--; }

            if ( i >= j )
            {
                if ( i != j )
                {
                    T& pi_tmp = pArr[ i ];
                    T& pj_tmp = pArr[ j ];
                    #ifndef HK_ARCH_ARM
                    swapAligned16( pi_tmp, pj_tmp );
                    #else
                    swap( pi_tmp, pj_tmp );
                    #endif
                }

                i--;
                j++;
            }
        } while ( j <= i );

        if ( d < i )
        {
            quickSortRecursive<T, L>( pArr, d, i, cmpLess );
        }

        if ( j < h )
        {
            d = j;
            goto begin;
        }
    }

    template<typename T, typename L>
void HK_CALL quickSort(_Inout_updates_(iSize) T *pArr, int iSize, L cmpLess)
    {
        if ( iSize > 1 )
        {
            int low = 0;
            int high = iSize - 1;
            quickSortRecursive<T, L>( pArr, low, high, cmpLess );
        }
    }


    template<typename T, typename L>
HK_INLINE void HK_CALL _explicitStackQuickSort(_Inout_updates_(numMem) T* base, int numMem, L cmpLess)
    {
        #ifndef HK_ARCH_ARM
        HK_ALIGN16( T pivot );
        #else
        T pivot;
        #endif
        const int maxDepth = sizeof( base ) * 8;

        if ( numMem <= 1 ) return;

        T* lbStack[ maxDepth ], *ubStack[ maxDepth ];

        lbStack[ 0 ] = base;
        ubStack[ 0 ] = base + ( numMem - 1 );

        for ( int stackPos = 0; stackPos >= 0; stackPos-- )
        {
            T* lb = lbStack[ stackPos ];
            T* ub = ubStack[ stackPos ];
            while ( true )
            {
                T* j = lb;
                T* i = ub;

                pivot = lb[ ( ub - lb ) >> 1 ];
                do
                {
                    while ( cmpLess( *j, pivot ) ) { j++; }
                    while ( cmpLess( pivot, *i ) ) { i--; }

                    if ( i >= j )
                    {
                        if ( i != j )
                        {
                            #ifndef HK_ARCH_ARM
                            swapAligned16( *i, *j );
                            #else
                            swap( *i, *j );
                            #endif
                        }
                        i--;
                        j++;
                    }
                } while ( j <= i );

                if ( lb < i )
                {
                    if ( j < ub )
                    {
                        // Both are sortable
                        // Stack the larger of the two sub-parts, this is needed to ensure no stack overflow
                        if ( ub - j > i - lb )
                        {
                            lbStack[ stackPos ] = j;
                            ubStack[ stackPos++ ] = ub;
                            //
                            ub = i;
                        }
                        else
                        {
                            // Stack it
                            lbStack[ stackPos ] = lb;
                            ubStack[ stackPos++ ] = i;
                            //
                            lb = j;
                        }
                        continue;
                    }
                    else
                    {
                        ub = i;
                        continue;
                    }
                }
                else
                {
                    if ( j < ub )
                    {
                        lb = j;
                        continue;
                    }
                }
                break;
            }
        }
    }

    template<typename T, typename L>
void HK_CALL explicitStackQuickSort(_Inout_updates_(numMem) T* base, int numMem, L cmpLess)
    {
        _explicitStackQuickSort<T, L>( base, numMem, cmpLess );
    }

    template< typename T, typename L >
void HK_CALL bubbleSort(_Inout_updates_(iSize) T *pArr, int iSize, L cmpLess )
    {
        bool didSwap = true;

        for ( int i = 1; ( i < iSize ) && didSwap; i++ )
        {
            didSwap = false;

            for ( int j = 0; j < iSize - i; j++ )
            {
                if ( cmpLess( pArr[ j + 1 ], pArr[ j ] ) )
                {
                    swap( pArr[ j + 1 ], pArr[ j ] );

                    didSwap = true;
                }
            }
        }
    }

    //
    //  Merge-sort For Linked Lists. Algorithm described at
    //  http://www.chiark.greenend.org.uk/~sgtatham/algorithms/listsort.html

    template <typename T, typename GetNext, typename Less>
T* HK_CALL sortList(_Inout_opt_ T* headPtr, GetNext getNext, Less cmpLess)
    {
        if ( !headPtr )
        {
            return HK_NULL;
        }

        for ( int sortSize = 1; ; sortSize *= 2 )
        {
            int numMerges = 0;

            T* p = headPtr;
            T preHead; getNext( &preHead ) = HK_NULL;
            T* tail = &preHead;

            while ( p )
            {
                numMerges++;

                T* q = p;
                int psize;
                for ( psize = 0; psize < sortSize && q; psize++ )
                {
                    q = getNext( q );
                }
                int qsize = sortSize;

                // while current lists not empty
                while ( ( psize > 0 ) && ( qsize > 0 ) && q )
                {
                    T* next;

                    if ( cmpLess( *q, *p ) )
                    {
                        next = q;
                        q = getNext( q );
                        qsize--;
                    }
                    else // ( p <= q )
                    {
                        next = p;
                        p = getNext( p );
                        psize--;
                    }

                    getNext( tail ) = next;
                    tail = next;
                }
                // one or both lists empty
                while ( psize > 0 )
                {
                    getNext( tail ) = p;
                    tail = p;
                    p = getNext( p );
                    psize--;
                }
                while ( ( qsize > 0 ) && q )
                {
                    getNext( tail ) = q;
                    tail = q;
                    q = getNext( q );
                    qsize--;
                }
                p = q;
            }
            getNext( tail ) = HK_NULL;

            if ( numMerges <= 1 )
            {
                return getNext( &preHead );
            }
            else
            {
                headPtr = getNext( &preHead );
            }
        }
    }

    template <typename T, typename FUNCTION>
    HK_INLINE int segregateInplace( hkArrayView<T> items, const FUNCTION& isLeftSide )
    {
        T* HK_RESTRICT  set0 = items.begin();
        T* HK_RESTRICT  set1 = items.end() - 1;

        // Consume left.
        while ( set0 <= set1 && isLeftSide( *set0 ) ) set0++;

        // Consume right.
        while ( set1 > set0 && !isLeftSide( *set1 ) ) set1--;

        // Segregate left-overs.
        if ( set0 < set1 )
        {
            do
            {
                // Swap one element.
                hkMath::swap( *set0, *set1 );

                // Advance left.
                do set0++; while ( set0 < set1 && isLeftSide( *set0 ) );

                // Advance right.
                do set1--; while ( set1 > set0 && !isLeftSide( *set1 ) );
            } while ( set0 < set1 );
        }

        // Return pivot.
        return int( set0 - items.begin() );
    }

    template<typename T, typename L>
    void HK_CALL quickFindKAux(T *pArr, int d, int h, int k, L cmpLess)
    {
        int i,j;
    #ifndef HK_ARCH_ARM
        HK_ALIGN16( T str );
    #else
        T str;
    #endif
    begin:
        i = h;
        j = d;

        str = pArr[(d+h)>>1];

        do {

            while ( cmpLess( pArr[j], str) ){ j++; }
            while ( cmpLess( str, pArr[i]) ){ i--; }

            if ( i >= j )
            {
                if ( i != j )
                {
                    T& pi = pArr[i];
                    T& pj = pArr[j];
                    #ifndef HK_ARCH_ARM
                    swapAligned16( pi, pj );
                    #else
                    swap( pi, pj );
                    #endif
                }

                i--;
                j++;
            }
        } while (j <= i);

        if (d < i && i >= k)
        {
            h = i;
            goto begin;
        }
        else if (j < h && j < k)
        {
            d = j;
            goto begin;
        }
    }

    template<typename T, typename L>
    void HK_CALL quickFindK(T *pArr, int iSize, int k, L cmpLess)
    {
        if (iSize > 1 && k > 0 && k < iSize)
        {
            int low = 0;
            int high = iSize - 1;
            quickFindKAux(pArr,low,high,k,cmpLess);
        }
    }

    template<typename T, typename L>
    T& HK_CALL nthElement(T* pArr, int iSize, int k, L cmpLess)
    {
        quickFindK(pArr, iSize, k, cmpLess);
        int maxIdx = k;
        for(int i = k + 1; i < iSize; i++)
        {
            if(cmpLess(pArr[maxIdx], pArr[i]))
            {
                maxIdx = i;
            }
        }

        if(maxIdx != k)
        {
            swap(pArr[k], pArr[maxIdx]);
        }

        return pArr[k];
    }

}  // namespace hkAlgorithm

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
