// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/Base/Algorithm/Sort/hkSort.h>

#include <Common/Base/Algorithm/Sort/hkRadixSort.h>

namespace hkAlgorithm
{
    // Do not use the one in <utility> to avoid dragging in the CRT from this header.
    template<typename T>
    HK_INLINE T&& move( T& x )
    {
        return static_cast<T&&>(x);
    }

    /// swap the elements
    template<typename T>
    HK_INLINE void HK_CALL swap(T& x, T& y)
    {
        T t = move(x);
        x = move(y);
        y = move(t);
    }


    /// swap the 16 byte aligned elements
    template<typename T>
    HK_INLINE void HK_CALL swapAligned16(T& x, T& y)
    {
        HK_ALIGN16( T t ) = move(x);
        x = move(y);
        y = move(t);
    }

    /// swap the 32 byte aligned elements
    template<typename T>
    HK_INLINE void HK_CALL swapAligned32(T& x, T& y)
    {
        HK_ALIGN32( T t ) = move(x);
        x = move(y);
        y = move(t);
    }

    /// swap SIMD aligned elements
    template<typename T>
    HK_INLINE void HK_CALL swapAlignedReal(T& x, T& y)
    {
        HK_ALIGN_REAL( T t ) = move(x);
        x = move(y);
        y = move(t);
    }

    /// use to change endian-ness. Swaps the bytes pointed to by start.
    HK_INLINE void swapBytes(_Inout_updates_bytes_(count) void* start, int count)
    {
        hkUint8* front = (hkUint8*)start;
        hkUint8* back  = (hkUint8*)hkAddByteOffset(start, count-1);
        for (int i = 0; i < count / 2; i++)
        {
            hkAlgorithm::swap(*(front++), *(back--));
        }
    }

    /// function object that routes calls to operator<
    template<typename T>
    class less
    {
        public:

            HK_INLINE hkBool32 operator() ( const T& a, const T& b )
            {
                return ( a < b );
            }
    };


    /// function object that routes calls to operator>
    template<typename T>
    class greater
    {
    public:

        HK_INLINE hkBool32 operator() ( const T& a, const T& b )
        {
            return ( a > b );
        }
    };

    /// function object that routes calls to operator==
    template <typename T>
    class eq
    {
    public:

        HK_INLINE hkBool32 operator() (const T& a, const T& b)
        {
            return (a == b);
        }
    };

    /// function object for linked lists
    template <typename T> struct DefaultNext
    {
        HK_INLINE T*& operator()(_In_ T* a) {   return a->m_next;   }
    };

    /// Binary search, returns the index of the key in the array or -1 if not found.
    template <typename K, typename T, typename L>
    _Ret_range_(-1, numItems - 1)
    static inline int   binarySearch(const K& key, _In_reads_(numItems) const T* items, int numItems, L cmpLess)
    {
        int i=0, j = numItems-1;
        while(j > i)
        {
            const int m = (i+j) >> 1;
            if(cmpLess(items[m] , key)) i = m + 1; else j = m;
        }
        if(i == j && !(cmpLess(items[i] , key) || cmpLess(key , items[i])))
            return i;
        else
            return -1;
    }

    /// Computes the insertion index for key K in the sorted array of items.
    template <typename K, typename T, typename L>
    _Ret_range_(0, numItems)
    static HK_INLINE int HK_CALL findInsertionIndex(const K& key, _In_reads_(numItems) const T* items, int numItems, L cmpLess)
    {
        int low = 0, high = numItems;
        while ( low < high )
        {
            const int mid = (low + high) >> 1;
            if ( cmpLess(items[mid], key) )
            {
                low = mid + 1;
            }
            else if ( cmpLess(key, items[mid]) )
            {
                high = mid;
            }
            else
            {
                return mid;
            }
        }
        return low;
    }

    /// Binary search, returns the index of the key in the array or -1 if not found.
    template <typename K, typename T>
    _Ret_range_(-1, numItems - 1)
    static inline int binarySearch(const K& key, _In_reads_(numItems) const T* items, int numItems)
    {
        return binarySearch(key, items, numItems, less<T>());
    }

    /// Inplace segregation of an array in two parts (left and right).
    /// Returns the pivot such that the left part of the array start at index 0 and has 'pivot' elements.
    /// FUNCTION takes a item T and returns true if the item belongs to the left part.
    /// Complexity between O(N/2) and O(N).

    /// Example:
    ///
    /// int items[] = { 1, 2, 3, 4 };
    /// int pivot = hkAlgorithm::segregateInplace( hkArrayViewT::make( items ), []( int x ) {return ( x & 1 ) == 0; } );
    ///
    /// gives:
    ///
    /// pivot = 2
    /// items = { 4, 2, 3, 1 }
    ///
    template <typename T, typename FUNCTION>
    HK_INLINE int segregateInplace( hkArrayView<T> items, const FUNCTION& isLeftSide );

    /// heap sort. You supply the functor, see hkAlgorithm::less for an example functor.
    template<typename T, typename L>
    void HK_CALL heapSort(_Inout_updates_(iSize) T *pArr, int iSize, L cmpLess);


    /// Heap sort for the elements of the specified array.
    ///
    /// \param *pArr        A pointer to the array to sort.
    /// \param iSize        The size of the array pointed to by *pArr.
    ///
    template<typename T>
    void HK_CALL heapSort(_Inout_updates_(iSize) T *pArr, int iSize)
    {
        heapSort( pArr, iSize, less<T>() );
    }

    // used by heapSort
    template<typename T, typename L>
    void HK_CALL downHeap(_Inout_updates_(2*k) T *pArr, int k, int n, L cmpLess);

    template<typename T, typename L>
    void HK_CALL insertionSort(_Inout_updates_(size) T* pArr, int size, L cmpLess);

    template<typename T>
    void HK_CALL insertionSort(_Inout_updates_(size) T* pArr, int size)
    {
        insertionSort( pArr, size, less<T>() );
    }

    /// quick sort. You supply the functor, see hkAlgorithm::less for an example functor.
    template<typename T, typename L>
    HK_INLINE   void HK_CALL quickSort(_Inout_updates_(iSize) T *pArr, int iSize, L cmpLess);

    /// Quick sort for the elements of the specified array.
    ///
    /// \param *pArr        A pointer to the data to sort.
    /// \param iSize        The size of the array pointed to by *pArr.
    ///
    template<typename T>
    HK_INLINE   void HK_CALL quickSort(_Inout_updates_(iSize) T *pArr, int iSize)
    {
        quickSort( pArr, iSize, less<T>() );
    }

    template< typename T, typename L >
    void HK_CALL quickSortRecursive(_Inout_updates_(h + 1) T *pArr, int d, int h, L cmpLess);

        /// Quick sort using an explicit stack for efficiency reasons.
    template<typename T>
    HK_INLINE void HK_CALL explicitStackQuickSort(_Inout_updates_(iSize) T *pArr, int iSize)
    {
        explicitStackQuickSort( pArr, iSize, less<T>() );
    }


    /// Bubble sort. You supply the functor, see hkAlgorithm::less for an example functor.
    template<typename T, typename L>
    HK_INLINE   void HK_CALL bubbleSort(_Inout_updates_(iSize) T *pArr, int iSize, L cmpLess);


    /// Bubble sort for the elements of the specified array.
    /// THIS IS NOT AN EFFICIENT SORTING ALGORITHM: O(n^2).
    /// It's almost always better to use quickSort or heapSort.
    /// Unlike quickSort or heapSort, this algorithm is STABLE:
    /// the relative order of equal elements is preserved.
    /// Bubble sort is a good choice for sorting or re-sorting a list
    /// that is approximately sorted already.
    ///
    /// \param *pArr        A pointer to the data to sort.
    /// \param iSize        The size of the array pointed to by *pArr.
    ///
    template<typename T>
    HK_INLINE   void HK_CALL bubbleSort(_Inout_updates_(iSize) T *pArr, int iSize)
    {
        bubbleSort( pArr, iSize, less<T>() );
    }

        /// Inplace singly linked list sort.
        /// Elements are sorted by address.
        /// Performs a merge sort (O(n log n) without any allocations.
    template <typename T, typename GetNext, typename Less>
    T* sortList(_Inout_opt_ T* head, GetNext getNext, Less cmpLess);

        /// Inplace singly linked list sort.
        /// Assumes the first element of each object is a pointer to next.
        /// Elements are sorted by address.
        /// Performs a merge sort (O(n log n) without any allocations.
    template<typename T>
    T* sortList(_Inout_opt_ T* head)
    {
        return sortList(head, DefaultNext<T>(), less<T>());
    }

    /// Find the lowest elements of the specified array.
    ///
    /// The array is rearranged such that the lowest k items are at
    /// the lowest k indices. The array is not otherwise sorted.
    ///
    /// This algorithm has O(n) expected performance.
    ///
    /// \param *pArr        A pointer to the data to sort.
    /// \param iSize        The size of the array pointed to by *pArr.
    /// \param k            The number of lowest elements to find.
    template<typename T>
    HK_INLINE void HK_CALL quickFindK(T *pArr, int iSize, int k)
    {
        quickFindK(pArr, iSize, k, less<T>());
    }

    /// An overload of quickFindK, which takes a user specified less than function.
    template<typename T, typename L>
    HK_INLINE void HK_CALL quickFindK(T *pArr, int iSize, int k, L cmpLess);

    template<typename T, typename L>
    void HK_CALL quickFindKAux(T *pArr, int d, int h, int k, L cmpLess);

    /// Rearranges the elements in the given array such that:
    /// 1. The element at index k is equal to the element at index k of the
    ///    array were fully sorted.
    /// 2. All elements at a lower index than k compare less than or equal to
    ///    the element at index k.
    /// 3. All elements at a higher index than k compare greater than or equal
    ///    to the element at index k.
    ///
    /// This is stronger than \ref quickFindK. quickFindK only guarantees
    /// condition 2.
    template<typename T>
    HK_INLINE T& HK_CALL nthElement(T* pArr, int iSize, int k)
    {
        return nthElement(pArr, iSize, k, less<T>());
    }

    /// An overload of nthElement, which takes a user specified less than function.
    template<typename T, typename L>
    T& HK_CALL nthElement(T* pArr, int iSize, int k, L cmpLess);

    /// Removes identical entries in the given list. Returns the number of unique elements found.
    /// The list is assumed to be sorted
    template <typename T, typename Eq>
    HK_INLINE int removeDuplicatesFromSortedList(_Inout_updates_(numElements) T* HK_RESTRICT elements, int numElements, Eq eq)
    {
        if ( ! numElements )
        {
            return 0;
        }

        int prev = 0;
        for (int crt = 1; crt < numElements; crt++)
        {
            if ( !eq(elements[prev], elements[crt]) )
            {
                // Element not equal, crt survives
                elements[++prev] = move(elements[crt]);
            }
        }

        return prev + 1;
    }

    template <typename T>
    HK_INLINE int removeDuplicatesFromSortedList(_Inout_updates_(numElements) T* HK_RESTRICT elements, int numElements)
    {
        return removeDuplicatesFromSortedList(elements, numElements, eq<T>());
    }

    /// Computes the intersection of two sorted lists. Returns the number of elements in the intersection
    template <typename T, typename Less>
    HK_INLINE int HK_CALL intersectionOfSortedLists(    _In_reads_(numElementsA) const T* HK_RESTRICT elementsA, int numElementsA,
                                                        _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB,
                                                        _Out_writes_(min(numElementsA, numElementsB)) T* HK_RESTRICT elementsAB, Less cmpLess)
    {
        int iA = 0, iB = 0, iAB = 0;
        while ( (iA < numElementsA) && (iB < numElementsB) )
        {
            const T& valA = elementsA[iA];
            const T& valB = elementsB[iB];
            if      ( cmpLess(valA, valB) ) {   iA++;           }
            else if ( cmpLess(valB, valA) ) {   iB++;           }
            else /* ( valA == valB )*/      {   elementsAB[iAB++] = valA;   iA++;   iB++;   }

        }
        return iAB;
    }

    /// Computes the indices of sorted list Bs elements in sorted list A (if they exist). Returns the number of indices.
    template <typename T, typename Less>
    HK_INLINE int HK_CALL indicesOfSortedListBinA(  const T* HK_RESTRICT elementsA, int numElementsA,
                                                    const T* HK_RESTRICT elementsB, int numElementsB,
                                                    int* HK_RESTRICT indicesOfBinA, Less cmpLess)
    {
        int iA = 0, iB = 0, iAB = 0;
        while ( (iA < numElementsA) && (iB < numElementsB) )
        {
            const T& valA = elementsA[iA];
            const T& valB = elementsB[iB];
            if      ( cmpLess(valA, valB) ) {   iA++;           }
            else if ( cmpLess(valB, valA) ) {   iB++;           }
            else /* ( valA == valB )*/      { indicesOfBinA[iAB++] = iA++;  iB++;   }
        }
        return iAB;
    }

    /// Computes the union of two sorted lists. Returns the number of elements in the union
    template <typename T, typename Less>
    HK_INLINE int HK_CALL unionOfSortedLists(_In_reads_(numElementsA) const T* HK_RESTRICT elementsA, int numElementsA,
                                             _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB,
                                             _Out_writes_(numElementsA + numElementsB) T* HK_RESTRICT elementsAB, Less cmpLess)
    {
        int iA = 0, iB = 0, iAB = 0;
        while ( (iA < numElementsA) && (iB < numElementsB) )
        {
            const T& valA = elementsA[iA];
            const T& valB = elementsB[iB];
            if      ( cmpLess(valA, valB) )     {   elementsAB[iAB++] = valA;   iA++;           }
            else if ( cmpLess(valB, valA) )     {   elementsAB[iAB++] = valB;   iB++;           }
            else    /*( valA == valB )*/        {   elementsAB[iAB++] = valA;   iA++;   iB++;   }
        }
        while ( iA < numElementsA ) {   elementsAB[iAB++] = elementsA[iA++];    }
        while ( iB < numElementsB ) {   elementsAB[iAB++] = elementsB[iB++];    }

        return iAB;
    }

    /// Merges the two sorted arrays (i.e. keeping duplicates).
    template <typename T, typename Less>
    HK_INLINE void HK_CALL mergeSortedLists(    _In_reads_(numElementsA) const T* HK_RESTRICT elementsA, int numElementsA,
                                                _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB,
                                                _Out_writes_(numElementsA + numElementsB) T* HK_RESTRICT elementsAB, Less cmpLess)
    {
        int iA = 0, iB = 0, iAB = 0;
        while ( (iA < numElementsA) && (iB < numElementsB) )
        {
            const T& valA = elementsA[iA];
            const T& valB = elementsB[iB];
            if      ( cmpLess(valA, valB) )     {   elementsAB[iAB++] = valA;   iA++;   }
            else if ( cmpLess(valB, valA) )     {   elementsAB[iAB++] = valB;   iB++;   }
            else    /*( valA == valB )*/        {   elementsAB[iAB++] = valA;   iA++;
                                                    elementsAB[iAB++] = valB;   iB++;   }
        }
        while ( iA < numElementsA ) {   elementsAB[iAB++] = elementsA[iA++];    }
        while ( iB < numElementsB ) {   elementsAB[iAB++] = elementsB[iB++];    }

        HK_ASSERT_NO_MSG(0x21827b61, iAB == iA + iB);
    }

    /// Merges the second sorted array into the first one. The final size of the first sorted array will be
    /// numElementsA + numElementsB.
    template <typename T, typename Less>
    HK_INLINE void HK_CALL mergeIntoSortedList(
        T* HK_RESTRICT elementsA, int numElementsA,
        const T* HK_RESTRICT elementsB, int numElementsB,
        Less cmpLess)
    {
        int iA = numElementsA-1, iB = numElementsB-1, iOut = numElementsA+numElementsB-1;
        while((iA >= 0) && (iB >= 0))
        {
            const T& valA = elementsA[iA];
            const T& valB = elementsB[iB];
            if(cmpLess(valA, valB))
            {
                elementsA[iOut] = valB;
                --iB;
            }
            else
            {
                elementsA[iOut] = valA;
                --iA;
            }
            --iOut;
        }

        // If there are remaining A's, they're already in the right place.

        while(iB >= 0)
        {
            elementsA[iOut--] = elementsB[iB--];
        }

        HK_ASSERT_NO_MSG(0x21827b71, iOut == iA);
    }

    /// Computes the difference of two sorted lists. Returns the number of elements in the difference
    template <typename T, typename Less>
    HK_INLINE int HK_CALL differenceOfSortedLists(  _In_reads_(numElementsA) const T* HK_RESTRICT elementsA, int numElementsA,
                                                    _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB,
                                                    _Out_writes_(numElementsA) T* HK_RESTRICT elementsAB, Less cmpLess)
    {
        int iA = 0, iB = 0, iAB = 0;
        while ( (iA < numElementsA) && (iB < numElementsB) )
        {
            const T& valA = elementsA[iA];
            const T& valB = elementsB[iB];
            if      ( cmpLess(valA, valB) )     {   elementsAB[iAB++] = valA;   iA++;           }
            else if ( cmpLess(valB, valA) )     {   iB++;           }
            else /* ( valA == valB ) */         {   iA++;   iB++;   }
        }
        while ( iA < numElementsA ) {   elementsAB[iAB++] = elementsA[iA++];    }

        return iAB;
    }

    template<typename T, typename Less>
    HK_INLINE int HK_CALL inplaceDifferenceOfSortedLists(_Inout_updates_(numElementsA) T* HK_RESTRICT elementsA, int numElementsA,
                                                         _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB,
                                                         Less cmpLess)
    {
        int iA = 0, tA = 0, iB = 0;
        while ((iA < numElementsA) && (iB < numElementsB))
        {
            const T& valA = elementsA[iA];
            const T& valB = elementsB[iB];
            if      (cmpLess(valA, valB)) { if (tA != iA) { elementsA[tA] = valA; } ++tA; ++iA; }
            else if (cmpLess(valB, valA)) { ++iB; }
            else /* (valA == valB) */     { ++iA; ++iB; }
        }
        if (iA == tA)
        {
            return numElementsA;
        }
        while (iA < numElementsA) { elementsA[tA++] = elementsA[iA++]; }
        return tA;
    }

    /// Returns true if the sorted list A is included in the sorted list B
    template <typename T, typename Less, typename Eq>
    HK_INLINE bool HK_CALL testInclusionOfSortedLists(_In_reads_(numElementsA) const T* HK_RESTRICT elementsA, int numElementsA,
                                                            _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB,
                                                            Less cmpLess, Eq cmpEq)
    {
        int ia = numElementsA - 1, ib = numElementsB - 1;
        while ( (ia >= 0) && (ib >= 0) )
        {
            const T& eA = elementsA[ia];
            const T& eB = elementsB[ib];

            if ( cmpLess(eB, eA) )
            {
                return false;   // eA is unique to A, no containment!
            }

            if ( cmpEq(eA, eB) )
            {
                ia--;   // eA = eB is common to both A and B
            }
            ib--;       // eA <= eB
        }
        return (ia < 0);
    }

    template <typename T>
    HK_INLINE int HK_CALL intersectionOfSortedLists(    _In_reads_(numElementsA) const T* HK_RESTRICT elementsA, int numElementsA,
                                                        _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB,
                                                        _Out_writes_(min(numElementsA, numElementsB)) T* HK_RESTRICT elementsAB)
    {
        return intersectionOfSortedLists(elementsA, numElementsA, elementsB, numElementsB, elementsAB, less<T>());
    }

    template <typename T>
    HK_INLINE int HK_CALL unionOfSortedLists(   _In_reads_(numElementsA) const T* HK_RESTRICT elementsA, int numElementsA,
                                                _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB,
                                                _Out_writes_(numElementsA + numElementsB) T* HK_RESTRICT elementsAB)
    {
        return unionOfSortedLists(elementsA, numElementsA, elementsB, numElementsB, elementsAB, less<T>());
    }

    template <typename T>
    HK_INLINE void HK_CALL mergeSortedLists(    _In_reads_(numElementsA) const T* HK_RESTRICT elementsA, int numElementsA,
                                                _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB,
                                                _Out_writes_(numElementsA + numElementsB) T* HK_RESTRICT elementsAB)
    {
        mergeSortedLists(elementsA, numElementsA, elementsB, numElementsB, elementsAB, less<T>());
    }

    template <typename T>
    HK_INLINE void HK_CALL mergeIntoSortedList(
        T* HK_RESTRICT elementsA, int numElementsA,
        const T* HK_RESTRICT elementsB, int numElementsB)
    {
        mergeIntoSortedList(elementsA, numElementsA, elementsB, numElementsB, less<T>());
    }

    template <typename T>
    HK_INLINE int HK_CALL differenceOfSortedLists(  _In_reads_(numElementsA) const T* HK_RESTRICT elementsA, int numElementsA,
                                                    _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB,
                                                    _Out_writes_(numElementsA) T* HK_RESTRICT elementsAB)
    {
        return differenceOfSortedLists(elementsA, numElementsA, elementsB, numElementsB, elementsAB, less<T>());
    }

    template <typename T>
    HK_INLINE int HK_CALL inplaceDifferenceOfSortedLists(_Inout_updates_(numElementsA) T* HK_RESTRICT elementsA, int numElementsA,
                                                         _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB)
    {
        return inplaceDifferenceOfSortedLists(elementsA, numElementsA, elementsB, numElementsB, less<T>());
    }

    template <typename T>
    HK_INLINE bool HK_CALL testInclusionOfSortedLists(_In_reads_(numElementsA) const T* HK_RESTRICT elementsA, int numElementsA,
                                                        _In_reads_(numElementsB) const T* HK_RESTRICT elementsB, int numElementsB)
    {
        return testInclusionOfSortedLists(elementsA, numElementsA, elementsB, numElementsB, less<T>(), eq<T>());
    }
}

#define hkSort hkAlgorithm::quickSort
//#define hkSort hkAlgorithm::explicitStackQuickSort

#include <Common/Base/Algorithm/Sort/hkSort.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
